"""
Documentation and OpenAPI mixin for Zenith applications.

Contains methods for setting up API documentation and OpenAPI endpoints.
"""


class DocsMixin:
    """Mixin for OpenAPI documentation methods."""

    def add_docs(
        self,
        title: str | None = None,
        version: str = "1.0.0",
        description: str | None = None,
        docs_url: str = "/docs",
        redoc_url: str = "/redoc",
        openapi_url: str = "/openapi.json",
        servers: list[dict[str, str]] | None = None,
    ) -> None:
        """
        Add OpenAPI documentation routes.

        Args:
            title: API title (defaults to app name)
            version: API version
            description: API description
            docs_url: Swagger UI URL (set to None to disable)
            redoc_url: ReDoc URL (set to None to disable)
            openapi_url: OpenAPI spec JSON URL
            servers: Server configurations
        """
        from zenith.openapi.docs import setup_docs_routes

        # Add the basic OpenAPI endpoints (spec, docs, redoc) with proper configuration
        self._add_openapi_endpoints(
            docs_url=docs_url, redoc_url=redoc_url, openapi_url=openapi_url
        )

        docs_router = setup_docs_routes(
            routers=self.routers,
            title=title or "Zenith API",
            version=version,
            description=description or "API built with Zenith framework",
            docs_url=docs_url,
            redoc_url=redoc_url,
            openapi_url=openapi_url,
            servers=servers,
        )

        self.include_router(docs_router)

    def _add_openapi_endpoints(
        self,
        docs_url: str | None = None,
        redoc_url: str | None = None,
        openapi_url: str = "/openapi.json",
    ) -> None:
        """Add OpenAPI documentation endpoints."""
        from starlette.responses import HTMLResponse, JSONResponse

        from zenith.openapi import generate_openapi_spec

        # Only register OpenAPI spec endpoint
        @self._app_router.get(openapi_url)
        async def openapi_spec():
            """OpenAPI specification endpoint."""
            # Collect all routes from all routers
            all_routes = []
            for router in self.routers:
                all_routes.extend(router.routes)

            spec = generate_openapi_spec(
                routes=all_routes,
                title=f"{self.__class__.__name__} API",
                version="1.0.0",
                description="API documentation generated by Zenith Framework",
            )
            return JSONResponse(spec)

        # Only register Swagger UI if docs_url is provided
        if docs_url:

            @self._app_router.get(docs_url)
            async def swagger_ui():
                """Swagger UI documentation."""
                html_content = """
            <!DOCTYPE html>
            <html>
            <head>
                <title>API Documentation</title>
                <link rel="stylesheet" type="text/css" href="https://unpkg.com/swagger-ui-dist@3.52.5/swagger-ui.css" />
                <style>
                    html { box-sizing: border-box; overflow: -moz-scrollbars-vertical; overflow-y: scroll; }
                    *, *:before, *:after { box-sizing: inherit; }
                    body { margin:0; background: #fafafa; }
                </style>
            </head>
            <body>
                <div id="swagger-ui"></div>
                <script src="https://unpkg.com/swagger-ui-dist@3.52.5/swagger-ui-bundle.js"></script>
                <script>
                    SwaggerUIBundle({
                        url: '/openapi.json',
                        dom_id: '#swagger-ui',
                        presets: [
                            SwaggerUIBundle.presets.apis,
                            SwaggerUIBundle.presets.standalone
                        ],
                        layout: "BaseLayout"
                    });
                </script>
            </body>
            </html>
            """
                return HTMLResponse(html_content)

        # Only register ReDoc if redoc_url is provided
        if redoc_url:

            @self._app_router.get(redoc_url)
            async def redoc():
                """ReDoc documentation."""
                html_content = """
            <!DOCTYPE html>
            <html>
            <head>
                <title>API Documentation</title>
                <meta charset="utf-8"/>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <link href="https://fonts.googleapis.com/css?family=Montserrat:300,400,700|Roboto:300,400,700" rel="stylesheet">
                <style>
                    body { margin: 0; padding: 0; }
                </style>
            </head>
            <body>
                <redoc spec-url='/openapi.json'></redoc>
                <script src="https://cdn.jsdelivr.net/npm/redoc@2.0.0/bundles/redoc.standalone.js"></script>
            </body>
            </html>
            """
                return HTMLResponse(html_content)

        # Add metrics endpoint (production only for security)
        if not self.config.debug:
            from starlette.responses import PlainTextResponse

            @self._app_router.get("/metrics")
            async def metrics_endpoint():
                """Prometheus metrics endpoint."""
                from zenith.monitoring.metrics import metrics_endpoint as get_metrics

                content = await get_metrics()
                return PlainTextResponse(
                    content, media_type="text/plain; version=0.0.4"
                )
