import unittest
from datetime import datetime, timezone
from unittest.mock import Mock, patch

from botocore.exceptions import ClientError

from sagemaker_studio.execution.remote_execution_client import (
    DEFAULT_IMAGE_VERSION,
    DEFAULT_INSTANCE_TYPE,
    RemoteExecutionClient,
)
from sagemaker_studio.models.execution import (
    ConflictError,
    ExecutionConfig,
    InternalServerError,
    ValidationError,
)


class TestRemoteExecutionClient(unittest.TestCase):

    @patch.object(RemoteExecutionClient, "_RemoteExecutionClient__set_default_tooling_environment")
    @patch.object(RemoteExecutionClient, "_RemoteExecutionClient__set_sagemaker_environment")
    @patch.object(RemoteExecutionClient, "_RemoteExecutionClient__set_sagemaker_client")
    @patch.object(RemoteExecutionClient, "_RemoteExecutionClient__set_ec2client")
    @patch.object(RemoteExecutionClient, "_RemoteExecutionClient__set_ssmclient")
    @patch.object(RemoteExecutionClient, "_RemoteExecutionClient__set_stack")
    def setUp(
        self,
        mock_set_stack,
        mock_set_ssmclient,
        mock_set_ec2client,
        mock_set_sagemaker_client,
        mock_set_sagemaker_environment,
        mock_set_default_tooling_environment,
    ):
        # setup datazone api mock
        self.datazone_api = Mock()
        self.datazone_api.get_environment_credentials.return_value = self._default_env_creds()
        # setup project api mock
        self.project_api = Mock()
        project_api_instance = self.project_api.return_value
        project_api_instance.get_project_sagemaker_environment.return_value = (
            self._default_sage_maker_environment_summary()
        )
        self.base_session = Mock()
        self.sagemaker_client = Mock()
        self.base_session.client.return_value = self.sagemaker_client
        self.execution_config = ExecutionConfig()
        self.execution_config.domain_identifier = "domain_123"
        self.execution_config.project_identifier = "project_123"
        self.execution_config.datazone_environment_id = "environment_123"
        self.remote_client = RemoteExecutionClient(
            self.datazone_api, self.project_api, self.execution_config
        )

    def _default_env_creds(self):
        return {
            "accessKeyId": "123",
            "expiration": "1970",
            "secretAccessKey": "456",
            "sessionToken": "789",
        }

    def _default_sage_maker_environment_summary(self):
        return {
            "awsAccountId": "1234567890",
            "awsAccountRegion": "us-east-1",
            "domainId": "domain_123",
            "environmentBlueprintId": "bogus_env_blueprint_id",
            "environmentProfileId": "bogus_env_profile_id",
            "id": "bogus_env_id",
            "name": "bogus_env_name",
            "projectId": "project_123",
            "provider": "Amazon SageMaker",
            "provisionedResources": [
                {
                    "name": "s3BucketPath",
                    "provider": "Amazon SageMaker",
                    "type": "string",
                    "value": "bogus_s3_bucket_path",
                },
                {
                    "name": "kmsKeyArn",
                    "provider": "Amazon SageMaker",
                    "type": "string",
                    "value": "bogus_project_kms_key_arn",
                },
                {
                    "name": "otherProvisionedResource",
                    "provider": "Amazon SageMaker",
                    "type": "string",
                    "value": "bogus_provisioned_resource_value",
                },
            ],
            "status": "ACTIVE",
        }

    def test_execution_client_api_fails_no_domain_identifier(self):
        execution_config = ExecutionConfig()
        execution_config.project_identifier = "project_123"
        with self.assertRaises(InternalServerError):
            remote_client = RemoteExecutionClient(
                self.datazone_api, self.project_api, execution_config
            )
            remote_client.list_executions()

    def test_execution_client_api_fails_no_project_identifier(self):
        execution_config = ExecutionConfig()
        execution_config.domain_identifier = "domain_123"
        with self.assertRaises(InternalServerError):
            remote_client = RemoteExecutionClient(
                self.datazone_api, self.project_api, execution_config
            )
            remote_client.list_executions()

    def test_validate_execution_name(self):
        # valid name can have 26 chars maximum
        RemoteExecutionClient.validate_execution_name("a-valid-name-with-26-chars")

        invalid_names = ["-startswithspecialchar", "_startswithspecialchar", "b" * 27, ""]
        for name in invalid_names:
            with self.assertRaises(ValidationError) as ve:
                RemoteExecutionClient.validate_execution_name(name)
            self.assertEqual(
                str(ve.exception),
                f"Execution name {name} does not match required pattern '^[a-zA-Z0-9]([-a-zA-Z0-9]){0, 25}$'",
            )

    def test_validate_input_config(self):
        with self.assertRaises(ValidationError) as ve:
            RemoteExecutionClient.validate_input_config({})
        self.assertEqual(str(ve.exception), "InputConfig is required for remote execution")

        with self.assertRaises(ValidationError) as ve:
            RemoteExecutionClient.validate_input_config({"notebook_config": {}})
        self.assertEqual(
            str(ve.exception),
            "'inputPath' in InputConfig notebookConfig is required for remote execution",
        )

        # valid input confg must have input_path in notebook_config
        RemoteExecutionClient.validate_input_config(
            {"notebook_config": {"input_path": "s3://bucket/key"}}
        )
        # input config can also have input_parameters
        RemoteExecutionClient.validate_input_config(
            {
                "notebook_config": {
                    "input_path": "s3://bucket/key",
                    "input_parameters": {str(i): str(i) for i in range(100)},
                }
            }
        )
        with self.assertRaises(ValidationError) as ve:
            RemoteExecutionClient.validate_input_config(
                {
                    "notebook_config": {
                        "input_path": "s3://bucket/key",
                        "input_parameters": {str(i): str(i) for i in range(101)},
                    }
                }
            )
        self.assertEqual(
            str(ve.exception),
            "inputParameters in InputConfig notebookConfig cannot have more than 100 entries",
        )
        RemoteExecutionClient.validate_input_config(
            {
                "notebook_config": {
                    "input_path": "s3://bucket/key",
                    "input_parameters": {"n" * 256: "n"},
                }
            }
        )
        with self.assertRaises(ValidationError) as ve:
            RemoteExecutionClient.validate_input_config(
                {
                    "notebook_config": {
                        "input_path": "s3://bucket/key",
                        "input_parameters": {"n" * 257: "n"},
                    }
                }
            )
        self.assertEqual(
            str(ve.exception), "The input parameters key length cannot exceed 256 characters."
        )
        RemoteExecutionClient.validate_input_config(
            {
                "notebook_config": {
                    "input_path": "s3://bucket/key",
                    "input_parameters": {"n": "n" * 2500},
                }
            }
        )
        with self.assertRaises(ValidationError) as ve:
            RemoteExecutionClient.validate_input_config(
                {
                    "notebook_config": {
                        "input_path": "s3://bucket/key",
                        "input_parameters": {"n": "n" * 2501},
                    }
                }
            )
        self.assertEqual(
            str(ve.exception), "The input parameters value length cannot exceed 2500 characters."
        )

    def test_validate_image_version(self):
        # Test valid versions
        self.assertTrue(RemoteExecutionClient.validate_image_version("2.6"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("2.6.1"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("2.7"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("2.8"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("2.8.1"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("2"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("3.0.0"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("3.0"))
        self.assertTrue(RemoteExecutionClient.validate_image_version("3"))

        self.assertFalse(RemoteExecutionClient.validate_image_version("2.2.0"))
        self.assertFalse(RemoteExecutionClient.validate_image_version("2.2"))
        self.assertFalse(RemoteExecutionClient.validate_image_version("1.0"))
        self.assertFalse(RemoteExecutionClient.validate_image_version("2.0.1"))
        self.assertFalse(RemoteExecutionClient.validate_image_version("2.1"))
        self.assertFalse(RemoteExecutionClient.validate_image_version("4.0"))
        with self.assertRaises(ValidationError):
            self.assertTrue(RemoteExecutionClient.validate_image_version("latest"))

    # Test list_executions
    def test_list_executions(self):
        mock_sagemaker_client = Mock()
        self.remote_client.sagemaker_client = mock_sagemaker_client
        self.remote_client.ec2_client = self.remote_client.ssm_client = Mock()
        self.remote_client.security_group = self.remote_client.subnets = (
            self.remote_client.user_role_arn
        ) = self.remote_client.kms_key_identifier = Mock()

        mock_sagemaker_client.search.return_value = {
            "Results": [
                {
                    "TrainingJob": {
                        "TrainingJobName": "job-1",
                        "TrainingJobStatus": "Completed",
                        "TrainingStartTime": datetime(2024, 8, 8, 10, 0, 0, tzinfo=timezone.utc),
                        "TrainingEndTime": datetime(2024, 8, 8, 11, 0, 0, tzinfo=timezone.utc),
                        "Tags": [
                            {"Key": "sagemaker-notebook-execution", "Value": "TRUE"},
                            {"Key": "sagemaker-notebook-name", "Value": "test.ipynb"},
                            {"Key": "AmazonDataZoneDomain", "Value": "domain_123"},
                            {"Key": "AmazonDataZoneProject", "Value": "project_123"},
                            {"Key": "AmazonDataZoneEnvironment", "Value": "environment_123"},
                        ],
                    }
                },
                {
                    "TrainingJob": {
                        "TrainingJobName": "job-2",
                        "TrainingJobStatus": "Completed",
                        "TrainingStartTime": datetime(2024, 8, 9, 10, 0, 0, tzinfo=timezone.utc),
                        "TrainingEndTime": datetime(2024, 8, 9, 11, 0, 0, tzinfo=timezone.utc),
                        "Tags": [
                            {"Key": "sagemaker-notebook-execution", "Value": "TRUE"},
                            {"Key": "sagemaker-notebook-name", "Value": "test.ipynb"},
                            {"Key": "AmazonDataZoneDomain", "Value": "domain_123"},
                            {"Key": "AmazonDataZoneProject", "Value": "project_123"},
                        ],
                    }
                },
            ],
            "NextToken": "token",
        }

        # test list_executions
        response = self.remote_client.list_executions()
        executions = response.get("executions")
        self.assertEqual(len(executions), 2)  # type: ignore
        self.assertEqual(executions[0].get("id"), "job-1")  # type: ignore
        self.assertEqual(executions[0].get("name"), "job-1")  # type: ignore
        self.assertEqual(executions[0].get("status"), "COMPLETED")  # type: ignore
        self.assertEqual(executions[0].get("start_time"), 1723111200000)  # type: ignore
        self.assertEqual(executions[0].get("end_time"), 1723114800000)  # type: ignore

        # Verify that search was called with the correct parameters
        mock_sagemaker_client.search.assert_called_once()

    def test_stop_execution_failed_status(self):
        # Arrange
        mock_sagemaker_client = Mock()
        self.remote_client.sagemaker_client = mock_sagemaker_client
        self.remote_client.security_group = self.remote_client.subnets = (
            self.remote_client.user_role_arn
        ) = self.remote_client.kms_key_identifier = Mock()
        self.remote_client.ec2_client = self.remote_client.ssm_client = Mock()
        client_error = ClientError(
            error_response={
                "Error": {
                    "Code": "ValidationException",
                    "Message": "The request was rejected because the training job is in status Failed",
                },
                "ResponseMetadata": {
                    "RequestId": "1234567890ABCDEF",
                    "HostId": "host-id-1-A2Z",
                    "HTTPStatusCode": 400,
                    "HTTPHeaders": {
                        "x-amzn-requestid": "1234567890ABCDEF",
                        "content-type": "application/x-amz-json-1.1",
                        "content-length": "123",
                        "date": "Wed, 01 Jan 2023 12:00:00 GMT",
                    },
                    "RetryAttempts": 0,
                },
            },
            operation_name="StopTrainingJob",
        )
        mock_sagemaker_client.stop_training_job.side_effect = client_error

        # Act and Assert
        with self.assertRaises(ConflictError) as context:
            self.remote_client.stop_execution(execution_id="test_job_id")

        self.assertEqual(
            str(context.exception),
            "Execution with id test_job_id is in Failed status and cannot be Stopped",
        )

        mock_sagemaker_client.stop_training_job.assert_called_once_with(
            TrainingJobName="test_job_id"
        )

    @patch("uuid.uuid4")
    def test_start_execution_happy_path(self, mock_uuid):
        # Arrange
        mock_uuid.return_value = "1234-5678-9012-3456"

        mock_sagemaker_client = Mock()
        mock_sagemaker_client.create_training_job.return_value = {
            "TrainingJobArn": "arn:aws:sagemaker:us-west-2:123456123456:training-job/test-job-1234-5678-9012-3456"
        }

        self.remote_client.sagemaker_client = mock_sagemaker_client
        self.remote_client.kms_key_identifier = (
            "arn:aws:kms:us-west-2:123456123456:key/1234abcd-12ab-34cd-56ef-1234567890ab"
        )
        self.remote_client.user_role_arn = "arn:aws:iam::123456123456:role/SageMakerRole"
        self.remote_client.default_tooling_environment = {
            "awsAccountId": "123456123456",
            "awsAccountRegion": "us-west-2",
            "id": "env-12345",
            "name": "Default Environment",
            "provisionedResources": [
                {"name": "VpcId", "value": "vpc-12345678"},
                {"name": "SubnetIds", "value": "subnet-12345678,subnet-87654321"},
                {"name": "SecurityGroupId", "value": "sg-12345678"},
                {"name": "s3BucketPath", "value": "s3://my-bucket/my-project/"},
                {"name": "codeRepositoryName", "value": "my-code-repository"},
            ],
        }
        self.remote_client._utils = Mock()
        self.remote_client._utils._get_project_s3_path.return_value = "s3://my-bucket/my-project/"

        self.remote_client.domain_identifier = "domain-123"
        self.remote_client.project_identifier = "project-456"
        self.remote_client.datazone_endpoint = "https://datazone.amazonaws.com"
        self.remote_client.datazone_domain_region = "us-west-2"
        self.remote_client.datazone_stage = "prod"
        self.remote_client.datazone_environment_id = "env-789"
        self.remote_client.project_s3_path = "s3://my-bucket/my-project/"
        self.remote_client.security_group = "sg-12345678"
        self.remote_client.subnets = ["subnet-12345678"]

        ecr_uri = (
            "123456123456.dkr.ecr.us-west-2.amazonaws.com/sagemaker-distribution-prod:3.0.0-cpu"
        )
        input = {"path": "s3://my-bucket/my-project/workflows/project-files", "file": "test.ipynb"}
        output = {"path": "s3://my-bucket/my-project/workflows/output", "file": "_test.ipynb"}

        mock_ec2_client = Mock()
        mock_ec2_client.describe_instance_types.return_value = {
            "InstanceTypes": [
                {
                    "InstanceType": "t2.micro",
                    "VCpuInfo": {"DefaultVCpus": 1},
                    "MemoryInfo": {"SizeInMiB": 1024},
                    # ... other details
                },
            ]
        }
        self.remote_client.ec2_client = mock_ec2_client

        mock_ssm_client = Mock()
        mock_ssm_client.get_parameter.return_value = {"Parameter": {"Value": "123456123456"}}
        self.remote_client.ssm_client = mock_ssm_client

        # Act
        result = self.remote_client.start_execution(
            execution_name="test-job",
            compute={
                "instance_type": "ml.m5.large",
                "volume_size_in_gb": 50,
                "image_details": {
                    "image_name": "sagemaker-distribution-prod",
                    "image_version": "3.0.0",
                },
            },
            input_config={"notebook_config": {"input_path": "src/test.ipynb"}},
            termination_condition={"max_runtime_in_seconds": 3600},
            output_config={"notebook_config": {"input_path": "src/test.ipynb"}},
            tags={"a": "b"},
        )

        # Assert
        mock_sagemaker_client.create_training_job.assert_called_once()
        call_args = mock_sagemaker_client.create_training_job.call_args[1]

        self.assertEqual(call_args["TrainingJobName"], "test-job-1234-5678-9012-3456")
        self.assertEqual(call_args["AlgorithmSpecification"]["TrainingImage"], ecr_uri)
        self.assertEqual(call_args["RoleArn"], self.remote_client.user_role_arn)
        self.assertEqual(call_args["OutputDataConfig"]["S3OutputPath"], output["path"])
        self.assertEqual(
            call_args["OutputDataConfig"]["KmsKeyId"], self.remote_client.kms_key_identifier
        )
        self.assertEqual(call_args["ResourceConfig"]["InstanceType"], "ml.m5.large")
        self.assertEqual(call_args["ResourceConfig"]["VolumeSizeInGB"], 50)
        self.assertEqual(
            call_args["ResourceConfig"]["VolumeKmsKeyId"], self.remote_client.kms_key_identifier
        )
        self.assertEqual(
            call_args["InputDataConfig"][0]["DataSource"]["S3DataSource"]["S3Uri"], input["path"]
        )
        self.assertEqual(call_args["HyperParameters"], {})
        self.assertEqual(call_args["StoppingCondition"]["MaxRuntimeInSeconds"], 3600)
        self.assertEqual(call_args["Environment"]["SM_INPUT_NOTEBOOK_NAME"], input["file"])
        self.assertEqual(call_args["Environment"]["SM_OUTPUT_NOTEBOOK_NAME"], output["file"])
        self.assertEqual(
            call_args["VpcConfig"]["SecurityGroupIds"], [self.remote_client.security_group]
        )
        self.assertEqual(call_args["VpcConfig"]["Subnets"], self.remote_client.subnets)

        self.assertEqual(result["execution_id"], "test-job-1234-5678-9012-3456")
        self.assertEqual(result["tags"], {"a": "b"})

    @patch("uuid.uuid4")
    def test_start_execution_with_minimum_input_fields_provided(self, mock_uuid):
        # Arrange
        mock_uuid.return_value = "1234-5678-9012-3456"

        mock_sagemaker_client = Mock()
        mock_sagemaker_client.create_training_job.return_value = {
            "TrainingJobArn": "arn:aws:sagemaker:us-west-2:123456123456:training-job/test-job-1234-5678-9012-3456"
        }
        self.remote_client.sagemaker_client = mock_sagemaker_client
        self.remote_client.kms_key_identifier = (
            "arn:aws:kms:us-west-2:123456123456:key/1234abcd-12ab-34cd-56ef-1234567890ab"
        )
        self.remote_client.user_role_arn = "arn:aws:iam::123456123456:role/SageMakerRole"
        self.remote_client.default_tooling_environment = {
            "awsAccountId": "123456123456",
            "awsAccountRegion": "us-west-2",
            "id": "env-12345",
            "name": "Default Environment",
            "provisionedResources": [
                {"name": "VpcId", "value": "vpc-12345678"},
                {"name": "SubnetIds", "value": "subnet-12345678,subnet-87654321"},
                {"name": "SecurityGroupId", "value": "sg-12345678"},
                {"name": "s3BucketPath", "value": "s3://my-bucket/my-project/"},
                {"name": "codeRepositoryName", "value": "my-code-repository"},
            ],
        }
        self.remote_client._utils = Mock()
        self.remote_client._utils._get_project_s3_path.return_value = "s3://my-bucket/my-project/"
        self.remote_client.domain_identifier = "domain-123"
        self.remote_client.project_identifier = "project-456"
        self.remote_client.datazone_endpoint = "https://datazone.amazonaws.com"
        self.remote_client.datazone_domain_region = "us-west-2"
        self.remote_client.datazone_stage = ""
        self.remote_client.datazone_environment_id = "env-789"
        self.remote_client.project_s3_path = "s3://my-bucket/my-project/"
        self.remote_client.security_group = "sg-12345678"
        self.remote_client.subnets = ["subnet-12345678"]

        input = {"path": "s3://my-bucket/my-project/workflows/project-files", "file": "test.ipynb"}
        output = {"path": "s3://my-bucket/my-project/workflows/output", "file": "_test.ipynb"}

        mock_ec2_client = Mock()
        mock_ec2_client.describe_instance_types.return_value = {
            "InstanceTypes": [
                {
                    "InstanceType": "t2.micro",
                    "VCpuInfo": {"DefaultVCpus": 1},
                    "MemoryInfo": {"SizeInMiB": 1024},
                },
            ]
        }
        self.remote_client.ec2_client = mock_ec2_client

        mock_ssm_client = Mock()
        mock_ssm_client.get_parameter.return_value = {"Parameter": {"Value": "123456123456"}}
        self.remote_client.ssm_client = mock_ssm_client

        # Act
        result = self.remote_client.start_execution(
            execution_name="test-job",
            input_config={"notebook_config": {"input_path": "src/test.ipynb"}},
        )

        # Assert
        mock_sagemaker_client.create_training_job.assert_called_once()
        call_args = mock_sagemaker_client.create_training_job.call_args[1]

        self.assertEqual(call_args["TrainingJobName"], "test-job-1234-5678-9012-3456")
        default_ecr_uri = f"123456123456.dkr.ecr.us-west-2.amazonaws.com/sagemaker-distribution-loadtest:{DEFAULT_IMAGE_VERSION}-cpu"
        self.assertEqual(call_args["AlgorithmSpecification"]["TrainingImage"], default_ecr_uri)
        self.assertEqual(call_args["RoleArn"], self.remote_client.user_role_arn)
        self.assertEqual(call_args["OutputDataConfig"]["S3OutputPath"], output["path"])
        self.assertEqual(
            call_args["OutputDataConfig"]["KmsKeyId"], self.remote_client.kms_key_identifier
        )
        self.assertEqual(call_args["ResourceConfig"]["InstanceType"], DEFAULT_INSTANCE_TYPE)
        self.assertEqual(call_args["ResourceConfig"]["VolumeSizeInGB"], 30)
        self.assertEqual(
            call_args["ResourceConfig"]["VolumeKmsKeyId"], self.remote_client.kms_key_identifier
        )
        self.assertEqual(
            call_args["InputDataConfig"][0]["DataSource"]["S3DataSource"]["S3Uri"], input["path"]
        )
        self.assertEqual(call_args["HyperParameters"], {})
        self.assertEqual(call_args["StoppingCondition"]["MaxRuntimeInSeconds"], 86400)
        self.assertEqual(call_args["Environment"]["SM_INPUT_NOTEBOOK_NAME"], input["file"])
        self.assertEqual(call_args["Environment"]["SM_OUTPUT_NOTEBOOK_NAME"], output["file"])
        self.assertEqual(
            call_args["VpcConfig"]["SecurityGroupIds"], [self.remote_client.security_group]
        )
        self.assertEqual(call_args["VpcConfig"]["Subnets"], self.remote_client.subnets)

        self.assertEqual(result["execution_id"], "test-job-1234-5678-9012-3456")
