import json
import os
from unittest import TestCase
from unittest.mock import Mock, patch

from botocore.exceptions import ClientError

from sagemaker_studio import ClientConfig, Project
from sagemaker_studio._test.test_sagemaker_studio_api import TestSageMakerUIHelper
from sagemaker_studio.connections import Connection
from sagemaker_studio.models.execution import ExecutionConfig
from sagemaker_studio.utils import AIRFLOW_PREFIX

from .utils import _create_mock_paginator

LIST_PROJECT_PAGINATED_RESPONSE = [
    {
        "items": [
            {
                "name": "bogus_project_name_00",
                "id": "bogus_project_id_00",
                "domainId": "bogus_domain_id_00",
            },
            {
                "name": "bogus_project_name_01",
                "id": "bogus_project_id_01",
                "domainId": "bogus_domain_id_00",
            },
        ]
    },
    {
        "items": [
            {
                "name": "bogus_project_name_02",
                "id": "bogus_project_id_02",
                "domainId": "bogus_domain_id_00",
            },
            {
                "name": "bogus_project_name_03",
                "id": "bogus_project_id_03",
                "domainId": "bogus_domain_id_00",
            },
        ]
    },
]

GET_PROJECT_RESPONSE = {
    "domainId": "dzd_bx2a2afyvn2hlc",
    "id": "afdkyxtdm79rf4",
    "name": "My_Project_m20x7i9x",
    "description": "",
    "projectStatus": "ACTIVE",
    "createdBy": "189173f0-3081-70a5-b523-2cdd41c58ccf",
    "createdAt": "2024-10-08T20:55:58.523645+00:00",
    "lastUpdatedAt": "2024-10-09T15:54:33.875126+00:00",
    "domainUnitId": "3ryvfqu3wxb0r4",
    "projectProfileId": "4q5i2nvtok5onk",
    "userParameters": [
        {
            "environmentConfigurationName": "DataLake",
            "environmentParameters": [
                {"name": "consumerGlueDbName", "value": "consumer_db"},
                {"name": "producerGlueDbName", "value": "producer_db"},
            ],
        }
    ],
    "environmentDeploymentDetails": {
        "overallDeploymentStatus": "SUCCESSFUL",
        "environmentFailureReasons": {},
    },
}

GET_DOMAIN_RESPONSE_EXPRESS = {
    "id": "dzd_1234",
    "name": "test-domain",
    "preferences": {"DOMAIN_MODE": "EXPRESS"},
}

GET_DOMAIN_RESPONSE_STANDARD = {"id": "dzd_1234", "name": "test-domain"}

DEFAULT_TOOLING_ENV = {
    "awsAccountId": "1234567890",
    "awsAccountRegion": "us-east-1",
    "domainId": "bogus_domain_id",
    "environmentBlueprintId": "bogus_env_blueprint_id",
    "environmentProfileId": "bogus_env_profile_id",
    "id": "bogus_env_id",
    "name": "bogus_env_name",
    "projectId": "bogus_project_id",
    "provider": "Amazon SageMaker",
    "provisionedResources": [
        {
            "name": "s3BucketPath",
            "provider": "Amazon SageMaker",
            "type": "string",
            "value": "bogus_s3_bucket_path_tooling_env",
        },
        {
            "name": "kmsKeyArn",
            "provider": "Amazon SageMaker",
            "type": "string",
            "value": "bogus_project_kms_key_arn",
        },
        {
            "name": "otherProvisionedResource",
            "provider": "Amazon SageMaker",
            "type": "string",
            "value": "bogus_provisioned_resource_value",
        },
    ],
    "status": "ACTIVE",
}


def mock_build_execution_config(*args, **kwargs):
    return ExecutionConfig()


@patch(
    "sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._build_execution_config",
    mock_build_execution_config,
)
class TestProject(TestCase):
    def setUp(self):
        self.mock_datazone_api = Mock()
        self.list_projects_paginator = _create_mock_paginator(LIST_PROJECT_PAGINATED_RESPONSE)
        self.mock_datazone_api.get_paginator = Mock()
        self.mock_datazone_api.get_paginator.side_effect = lambda x: self.list_projects_paginator
        self.mock_datazone_api.get_project = Mock()
        self.mock_datazone_api.get_project.return_value = GET_PROJECT_RESPONSE
        # Mock get_domain to return STANDARD mode by default
        self.mock_datazone_api.get_domain = Mock()
        self.mock_datazone_api.get_domain.return_value = GET_DOMAIN_RESPONSE_STANDARD
        with open(
            TestSageMakerUIHelper.get_mock_sagemaker_space_metadata_path(), "w"
        ) as metadata_json:
            json.dump(
                TestSageMakerUIHelper.get_mock_sagemaker_space_metadata(),
                metadata_json,
                indent=4,
            )

        self.mock_project_api = Mock()
        self.mock_project_api.is_default_environment_present = Mock()

    def test_project_initialization_name_and_id_not_provided(self):
        with self.assertRaises(ValueError) as context:
            Project()
            self.assertTrue(
                "Project name not found in environment. Please specify a project name."
                in str(context.exception)
            )

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    def test_project_initialization_domain_id_not_provided(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api

        project = Project(id="abc123")

        self.assertEqual(project.domain_id, "dzd_1234")
        self.assertEqual(project.name, GET_PROJECT_RESPONSE["name"])
        self.assertEqual(project.project_status, GET_PROJECT_RESPONSE["projectStatus"])
        self.assertEqual(project.domain_unit_id, GET_PROJECT_RESPONSE["domainUnitId"])
        self.assertEqual(project.project_profile_id, GET_PROJECT_RESPONSE["projectProfileId"])

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    def test_project_initialization_only_project_name_provided_project_exists(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api

        project = Project(name="bogus_project_name_01")

        self.assertEqual(project.domain_id, "dzd_1234")
        self.assertEqual(project.name, GET_PROJECT_RESPONSE["name"])
        self.assertEqual(project.project_status, GET_PROJECT_RESPONSE["projectStatus"])
        self.assertEqual(project.domain_unit_id, GET_PROJECT_RESPONSE["domainUnitId"])
        self.assertEqual(project.project_profile_id, GET_PROJECT_RESPONSE["projectProfileId"])

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    def test_project_initialization_only_project_name_provided_project_does_not_exist(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api

        with self.assertRaises(RuntimeError) as context:
            Project(name="does_not_exist")
            self.assertTrue("Project does_not_exist not found" in str(context.exception))

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    def test_get_project_s3_path_from_environment_variable(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api

        os.environ["ProjectS3Path"] = "bogus_project_s3_path_environment_variable"
        project = Project(id="aa76bmnbd042v")
        s3_path = project.s3.root
        self.assertEqual(s3_path, "bogus_project_s3_path_environment_variable")
        del os.environ["ProjectS3Path"]

        os.environ[f"{AIRFLOW_PREFIX}PROJECT_S3_PATH"] = "bogus_airflow_project_s3_path"
        s3_path_airflow = project.s3.root
        self.assertEqual(s3_path_airflow, "bogus_airflow_project_s3_path")
        del os.environ[f"{AIRFLOW_PREFIX}PROJECT_S3_PATH"]

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils.SAGEMAKER_METADATA_JSON_PATH",
        "test_sagemaker_space_metadata.json",
    )
    def test_get_project_s3_path_from_sagemaker_json(
        self,
        get_domain_id_mock: Mock,
        get_aws_client_mock: Mock,
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api

        client_config = ClientConfig(overrides={"execution": {"local": True}})
        project = Project(id="aa76bmnbd042v", config=client_config)
        s3_path = project.s3.root
        self.assertEqual(
            s3_path, "s3://bogus_bucket_name/bogus_domain_id_sm_space/bogus_project_id_sm_space/dev"
        )

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_default_s3_path_from_default_tooling_env(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        s3_path = project.s3.root
        self.assertEqual(s3_path, "bogus_s3_bucket_path_tooling_env")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_default_s3_path_from_default_tooling_env_not_found(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = {}
        project = Project(id="aa76bmnbd042v")
        with self.assertRaises(RuntimeError) as context:
            project.s3.root
            self.assertTrue("s3BucketPath not found" in str(context.exception))

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_kms_key_arn(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        kms_key_arn = project.kms_key_arn
        self.assertEqual(kms_key_arn, "bogus_project_kms_key_arn")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_kms_key_arn_from_default_tooling_env_not_found(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = {}
        project = Project(id="aa76bmnbd042v")
        with self.assertRaises(RuntimeError) as context:
            project.kms_key_arn
            self.assertTrue(
                "kmsKeyArn provisioned resource not found in Tooling" in str(context.exception)
            )

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_datalake_consumer_glue_db_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.datalake_consumer_glue_db
        self.assertEqual("bogus_s3_bucket_path_tooling_env/data/catalogs/", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_datalake_athena_workgroup_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.datalake_athena_workgroup
        self.assertEqual("bogus_s3_bucket_path_tooling_env/sys/athena/", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_workflow_output_directory_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.workflow_output_directory
        self.assertEqual("bogus_s3_bucket_path_tooling_env/workflows/output/", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_workflow_temp_storage_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.workflow_temp_storage
        self.assertEqual("bogus_s3_bucket_path_tooling_env/workflows/tmp/", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_emr_ec2_log_destination_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.emr_ec2_log_destination
        self.assertEqual("bogus_s3_bucket_path_tooling_env/sys/emr", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_emr_ec2_certificates_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.emr_ec2_certificates
        self.assertEqual("bogus_s3_bucket_path_tooling_env/sys/emr/certs", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_emr_ec2_log_bootstrap_path(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = True
        s3_path = project.s3.emr_ec2_log_bootstrap
        self.assertEqual("bogus_s3_bucket_path_tooling_env/sys/emr/boot-strap", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_s3_path_non_root_environment_is_not_present(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        project._sagemaker_studio_api.project_api = self.mock_project_api
        self.mock_project_api.return_value = False
        with self.assertRaises(Exception) as context:
            project.s3.emr_ec2_log_bootstrap
            self.assertTrue("s3BucketPath not found" in str(context.exception))

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_environment_path_env_exists(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        project = Project(id="aa76bmnbd042v")
        s3_path = project.s3.environment_path("env_exists")
        self.assertEqual("bogus_s3_bucket_path_tooling_env/env_exists", s3_path)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils._internal.InternalUtils._get_default_tooling_environment_summary"
    )
    def test_get_project_environment_path_env_does_not_exist(
        self, utils_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        error_response = {
            "Error": {"Code": "ResourceNotFound", "Message": "The environment does not exist"}
        }
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        utils_mock.return_value = DEFAULT_TOOLING_ENV
        self.mock_datazone_api.get_environment = Mock()
        self.mock_datazone_api.get_environment.side_effect = ClientError(
            error_response, "GetEnvironment"  # type: ignore
        )
        project = Project(id="aa76bmnbd042v")
        with self.assertRaises(ClientError) as context:
            project.s3.environment_path("env_does_not_exist")
            self.assertTrue("The environment does not exist" in str(context.exception))

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_mlflow_tracking_server_arn")
    def test_mlflow_tracking_server_arn(
        self,
        mlflow_tracking_server_arn_mock: Mock,
        get_domain_id_mock: Mock,
        get_aws_client_mock: Mock,
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mlflow_tracking_server_arn_mock.return_value = "arn:aws:ml-flow-tracking-server"
        project = Project(id="aa76bmnbd042v")
        ml_flow_tracking_server_arn = project.mlflow_tracking_server_arn
        self.assertEqual(ml_flow_tracking_server_arn, "arn:aws:ml-flow-tracking-server")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_get_project_iam_role_from_default_iam_environment(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        """Test iam_role property in EXPRESS domain mode (uses default.iam)"""
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        # Set domain to EXPRESS mode
        self.mock_datazone_api.get_domain.return_value = GET_DOMAIN_RESPONSE_EXPRESS
        get_connection_by_name_mock.return_value = Connection(
            {"environmentUserRole": "arn:aws:iam:default_env_usr_role"},
            Mock(),
            Mock(),
            Mock(),
            Mock(),
            Mock(),
        )
        project = Project(id="aa76bmnbd042v")
        self.assertEqual(project.iam_role, "arn:aws:iam:default_env_usr_role")
        get_connection_by_name_mock.assert_called_with(name="default.iam")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_get_project_iam_role_from_project_iam_environment(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        """Test iam_role property in STANDARD domain mode (uses project.iam)"""
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        # Set domain to STANDARD mode (default behavior)
        self.mock_datazone_api.get_domain.return_value = GET_DOMAIN_RESPONSE_STANDARD
        get_connection_by_name_mock.return_value = Connection(
            {"environmentUserRole": "arn:aws:iam:project_env_usr_role"},
            Mock(),
            Mock(),
            Mock(),
            Mock(),
            Mock(),
        )
        project = Project(id="aa76bmnbd042v")
        self.assertEqual(project.iam_role, "arn:aws:iam:project_env_usr_role")
        get_connection_by_name_mock.assert_called_with(name="project.iam")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_get_project_iam_role_connection_fails(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        """Test iam_role property when the domain-determined IAM connection fails"""
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        # Set domain to STANDARD mode (uses project.iam)
        self.mock_datazone_api.get_domain.return_value = GET_DOMAIN_RESPONSE_STANDARD

        # Connection lookup fails
        get_connection_by_name_mock.side_effect = AttributeError("project.iam not found")
        project = Project(id="aa76bmnbd042v")

        with self.assertRaises(AttributeError) as context:
            project.iam_role
        self.assertTrue("project.iam not found" in str(context.exception))
        get_connection_by_name_mock.assert_called_with(name="project.iam")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.list_connections")
    def test_connections_property_fallback_to_project_iam(
        self, list_connections_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        """Test connections property in environment that has project.iam but not default.iam"""
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api

        # Create mock connections without default.iam
        project_iam_conn = Mock()
        project_iam_conn.name = "project.iam"
        other_conn = Mock()
        other_conn.name = "other.connection"

        # Return connections without default.iam
        list_connections_mock.return_value = [other_conn, project_iam_conn]

        project = Project(id="aa76bmnbd042v")
        connections = project.connections

        # project.iam should be first since default.iam doesn't exist
        self.assertEqual(connections[0], project_iam_conn)
        self.assertEqual(len(connections), 2)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils.SAGEMAKER_METADATA_JSON_PATH",
        "test_sagemaker_space_metadata.json",
    )
    def test_get_user_id_from_sagemaker_json(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        project = Project(id="aa76bmnbd042v")
        self.assertEqual(project.user_id, "bogus_user_id_sm_space")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch(
        "sagemaker_studio.utils.SAGEMAKER_METADATA_JSON_PATH",
        "/tmp/this-will-not-exist-ever-anywhere.json",
    )
    def test_get_user_id_from_sagemaker_json_file_not_exists(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        project = Project(id="aa76bmnbd042v")
        with self.assertRaises(RuntimeError) as context:
            project.user_id
            self.assertTrue(
                "Encountered an error getting the current user ID" in str(context.exception)
            )

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_shared_files_success(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mock_connection = Mock()
        mock_connection.data.s3_uri = "s3://shared-bucket/shared-files/"
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")
        result = project.shared_files
        get_connection_by_name_mock.assert_called_with(name="default.s3_shared")
        self.assertEqual(result, "s3://shared-bucket/shared-files/")

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_shared_files_connection_not_found(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        get_connection_by_name_mock.return_value = None
        project = Project(id="aa76bmnbd042v")

        with self.assertRaises(RuntimeError) as context:
            project.shared_files
        self.assertIn("No connection found for shared files.", str(context.exception))

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_shared_files_runtime_error_accessing_s3_uri(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mock_connection = Mock()
        # Remove the s3_uri attribute to simulate AttributeError
        del mock_connection.data.s3_uri
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        with self.assertRaises(RuntimeError) as context:
            project.shared_files
        error_message = str(context.exception)
        self.assertIn("Encountered an error getting the shared files path", error_message)
        self.assertIn("project 'My_Project_m20x7i9x'", error_message)
        self.assertIn("domain 'dzd_1234'", error_message)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_shared_files_attribute_error_accessing_data(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mock_connection = Mock()
        # Remove the data attribute to simulate AttributeError
        del mock_connection.data
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        with self.assertRaises(RuntimeError) as context:
            project.shared_files
        error_message = str(context.exception)
        self.assertIn("Encountered an error getting the shared files path", error_message)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_connection_by_name_default_in_express_mode(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        """Test connection() method in EXPRESS domain mode (uses default.iam)"""
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        # Set domain to EXPRESS mode
        self.mock_datazone_api.get_domain.return_value = GET_DOMAIN_RESPONSE_EXPRESS
        mock_connection = Mock()
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        result = project.connection()
        get_connection_by_name_mock.assert_called_once_with(name="default.iam")
        self.assertEqual(result, mock_connection)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_connection_by_name_default_in_standard_mode(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        """Test connection() method in STANDARD domain mode (uses project.iam)"""
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        # Set domain to STANDARD mode (default behavior)
        self.mock_datazone_api.get_domain.return_value = GET_DOMAIN_RESPONSE_STANDARD
        mock_connection = Mock()
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        result = project.connection()
        get_connection_by_name_mock.assert_called_once_with(name="project.iam")
        self.assertEqual(result, mock_connection)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_connection_by_name_explicit(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mock_connection = Mock()
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        result = project.connection("test_connection")
        get_connection_by_name_mock.assert_called_once_with(name="test_connection")
        self.assertEqual(result, mock_connection)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_name")
    def test_connection_by_name_keyword(
        self, get_connection_by_name_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mock_connection = Mock()
        get_connection_by_name_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        result = project.connection(name="test_connection")
        get_connection_by_name_mock.assert_called_once_with(name="test_connection")
        self.assertEqual(result, mock_connection)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    @patch("sagemaker_studio.connections.ConnectionService.get_connection_by_id")
    def test_connection_by_id(
        self, get_connection_by_id_mock: Mock, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        mock_connection = Mock()
        get_connection_by_id_mock.return_value = mock_connection
        project = Project(id="aa76bmnbd042v")

        result = project.connection(id="12345")
        get_connection_by_id_mock.assert_called_once_with(connection_id="12345")
        self.assertEqual(result, mock_connection)

    @patch("sagemaker_studio.sagemaker_studio_api.SageMakerStudioAPI._get_aws_client")
    @patch("sagemaker_studio.utils._internal.InternalUtils._get_domain_id")
    def test_connection_both_name_and_id_raises_error(
        self, get_domain_id_mock: Mock, get_aws_client_mock: Mock
    ):
        get_domain_id_mock.return_value = "dzd_1234"
        get_aws_client_mock.side_effect = lambda x, y: self.mock_datazone_api
        project = Project(id="aa76bmnbd042v")

        with self.assertRaises(ValueError) as context:
            project.connection(name="test_connection", id="12345")
        self.assertIn("Cannot specify both 'name' and 'id' parameters", str(context.exception))
