from uuid import UUID
from typing import List

from textual.app import App, ComposeResult
from textual.widgets import Button, Header
from textual import on
from textual.app import App, ComposeResult
from textual.widgets import Input, Label, Pretty, Footer, Select
from textual.containers import Horizontal
from textual.validation import Length, Regex
from exploitfarm.utils.config import ClientConfig
from exploitfarm.model import Language
from exploitfarm.utils.config import EXPLOIT_CONFIG_REGEX
from exploitfarm.utils.config import ExploitConfig
from exploitfarm.utils.config import check_exploit_config_exists
from rich import print

class ExploitConf(App):
    
    def __init__(self,
        config: ClientConfig,
        edit:bool,
        name:str|None = None,
        lang:Language|None = None,
        service:str|None = None
    ):
        super().__init__()
        self.config = config
        self.services = [(srv["name"], srv["id"]) for srv in self.config.status["services"]]+[("[bold]+ Add a new service", None)]
        self.langs = [(lang.value, lang.value) for lang in list(Language)]
        
        search_service = list(filter(lambda x: x[1] == service, self.services))
        self.service = search_service[0][1] if search_service else self.services[0][1]
        
        search_lang = list(filter(lambda x: Language(x[1]) == lang, self.langs))
        self.lang = search_lang[0][1] if search_lang else Language.python.value
        
        self.edit = edit
        self.title = f"xFarm - {'Editing' if self.edit else 'Creating'} Exploit"
        self.service_name = ""
        self.exploit_name = name if name else ""
    
    BINDINGS = [
        ("ctrl+s", "save()", f"Save exploit"),
        ("ctrl+c", "cancel()", "Cancel operation")
    ]
    
    CSS = """
    Input.-valid {
        border: tall $success 60%;
    }
    Input.-valid:focus {
        border: tall $success;
    }
    Input {
        margin: 1 1;
    }
    Pretty {
        margin-left: 2;
    }
    #error{
        margin: 2 0 0 2;
    }
    .input-label, .form-input {
        margin: 1 2;
    }
    .error-box {
        margin: 1 2;
        height: 1.8;
    }
    Button {
        margin: 1 2;
    }
    .button-box {
        height: 4;
        margin: 2 0;
        margin-top: 1;
    }
    .max-width {
        width: 100%;
    }
    .hidden {
        display: none;
    }
    """

    def compose(self) -> ComposeResult:
        yield Header(f"xFarm - {'Editing' if self.edit else 'Creating'} exploit")
        yield Label("[bold]Exploit name:[/]", classes="input-label")
        yield Input(
            placeholder="funny_service_sqli",
            value=self.exploit_name,
            validators=[Regex(EXPLOIT_CONFIG_REGEX)],
            id="exploit_name",
            classes="form-input"
        )
        yield Horizontal(
            Label("[bold]Errors:[/]"),
            Pretty([], id="name_errors"),
            classes="error-box hidden",
            id="name_errors_box"
        )
        
        yield Select(self.langs, id="lang", classes="input-label", value=self.lang, allow_blank=False)
        yield Select(self.services, id="srv", classes="input-label", value=self.service, allow_blank=False)
        
        yield Label("[bold]New service name:[/]", classes="input-label hidden add-srv")
        yield Input(
            placeholder="service-name",
            value=self.service_name,
            validators=[Length(minimum=1)],
            id="srv_name",
            classes="form-input hidden add-srv"
        )
        yield Horizontal(
            Label("[bold]Errors:[/]"),
            Pretty([], id="srv_name_errors"),
            classes="error-box hidden",
            id="srv_name_errors_box"
        )
        yield Label("", classes="hidden", id="error")
        yield Horizontal(
            Button("Edit" if self.edit else "Create", id="save", variant="success"),
            Button("Cancel", id="cancel", variant="error"),
            classes="max-width button-box"
        )
        yield Footer()
    
    def show_error(self, s:str):
        self.query_one("#error", Label).update(s)
        self.query_one("#error", Label).remove_class("hidden")
    
    @on(Select.Changed, "#lang")
    def lang_changed(self, event: Select.Changed) -> None:
        if event.value == Language.other.value:
            self.show_error("[bold yellow]Remember to set teh correct main file and interpreter in the config file!")
        else:
            self.query_one("#error", Label).add_class("hidden")
        self.lang = event.value
    
    @on(Select.Changed, "#srv")
    def srv_changed(self, event: Select.Changed) -> None:
        if event.value is None:
            self.query(".add-srv").remove_class("hidden")
        else:
            self.query(".add-srv").add_class("hidden")
            self.query_one("#srv_name_errors_box", Horizontal).add_class("hidden")
        self.service = event.value
    
    def action_save(self):
        self.save()
    
    def action_cancel(self):
        self.cancel()
        
    @on(Button.Pressed, "#save")
    def save(self):
        input_forms: List[Input] = self.query(".form-input")
        errors = []
        for form in input_forms:
            errors.extend([ele.failure_description for ele in form.validators if ele.failure_description])
        
        if len(errors) == 0 and self.exploit_name:
            if self.service is None:
                try:
                    self.service  = UUID(self.config.reqs.new_service({"name": self.service_name})["id"])
                    self.config.fetch_status()
                except Exception as e:
                    print(f"[bold yellow] Service not subscribed!: {e}")
                    self.service = UUID("00000000-0000-0000-0000-000000000000")
            if self.edit:
                x_conf = ExploitConfig.read(".")
                x_conf.name = self.exploit_name
                x_conf.language = Language(self.lang)
                x_conf.service = self.service
            else:
                if check_exploit_config_exists(self.exploit_name):
                    self.show_error(f"[bold red]The exploit named '{self.exploit_name}' already exists")
                    return
                x_conf = ExploitConfig.new(self.exploit_name, Language(self.lang), self.service)
            x_conf.write(x_conf.name if not self.edit else ".")
            try:
                x_conf.publish_exploit(self.config)
            except Exception as e:
                self.exit(99)
                return
            self.exit(0)
        else:
            self.show_error(f"[bold red]{', '.join(errors)}")
            
    @on(Button.Pressed, "#cancel")
    def cancel(self):
        self.exit(1)
    
    @on(Input.Changed, "#exploit_name")
    def name_check(self, event: Input.Changed) -> None:
        # Updating the UI to show the reasons why validation failed
        error_label = self.query_one("#name_errors", Pretty)
        error_box = self.query_one("#name_errors_box", Horizontal)
        if not event.validation_result.is_valid:
            error_box.remove_class("hidden")
            error_label.update(event.validation_result.failure_descriptions)
        else:
            error_box.add_class("hidden")
            self.exploit_name = event.input.value
    
    @on(Input.Changed, "#srv_name")
    def srv_name_check(self, event: Input.Changed) -> None:
        error_label = self.query_one("#srv_name_errors",Pretty)
        error_box = self.query_one("#srv_name_errors_box", Horizontal)
        if not event.validation_result.is_valid:
            error_box.remove_class("hidden")
            error_label.update(event.validation_result.failure_descriptions)
        else:
            error_box.add_class("hidden")
            self.service_name = event.input.value

