
from pydantic import BaseModel
from uuid import UUID
from enum import Enum

class ServiceDTO(BaseModel):
    id: UUID
    name: str

class Language(Enum):
    python = "python"
    java = "java"
    javascript = "javascript"
    typescript = "typescript"
    csharp = "c#"
    cpp = "c++"
    php = "php"
    r = "r"
    kotlin = "kotlin"
    go = "go"
    ruby = "ruby"
    rust = "rust"
    lua = "lua"
    dart = "dart"
    perl = "perl"
    haskell = "haskell"
    other = "other"

class ExploitStatus(Enum):
    active = 'active'
    noflags = 'noflags'
    disabled = 'disabled'

def get_lang(lang: str|Language) -> Language:
    if isinstance(lang, Language):
        return lang
    res = [ele for ele in list(Language) if ele.value == lang]
    if len(res) == 0:
        return Language.other
    return res[0]

PYTHON_DEFAULT_CONTENT = """
from exploitfarm import *

host = get_host()

print(f"Hello {host}! This text should contain a lot of flags!")
"""

def get_default_file(lang: Language|str) -> str:
    lang = get_lang(lang)
    shebang = f"#!/usr/bin/env {get_interpreter(lang)}\n\n" if get_interpreter(lang) else "\n"
    match lang:
        case Language.python:
            return shebang+PYTHON_DEFAULT_CONTENT
        case _:
            return f"{shebang}Your code here\nPlease print the flags to stdout, exploitfarm will filter flags automatically\n"

def get_interpreter(lang: Language) -> str|None:
    match lang:
        case Language.python:
            return "python3"
        case Language.java:
            return "java"
        case Language.javascript:
            return "node"
        case Language.typescript:
            return "npm run"
        case Language.csharp:
            return None
        case Language.cpp:
            return None
        case Language.php:
            return "php"
        case Language.r:
            return "Rscript"
        case Language.kotlin:
            return "java -jar"
        case Language.go:
            return None
        case Language.ruby:
            return "ruby"
        case Language.rust:
            return None
        case Language.lua:
            return "lua"
        case Language.dart:
            return "dart run"
        case Language.perl:
            return "perl"
        case Language.haskell:
            return None
        case Language.other:
            return None

def get_filename(lang: Language) -> str:
    match lang:
        case Language.python:
            return "main.py"
        case Language.java:
            return "main.java"
        case Language.javascript:
            return "main.js"
        case Language.typescript:
            return "main.ts"
        case Language.csharp:
            return "main.cs"
        case Language.cpp:
            return "main.cpp"
        case Language.php:
            return "main.php"
        case Language.r:
            return "main.r"
        case Language.kotlin:
            return "main.kt"
        case Language.go:
            return "main.go"
        case Language.ruby:
            return "main.rb"
        case Language.rust:
            return "main.rs"
        case Language.lua:
            return "main.lua"
        case Language.dart:
            return "main.dart"
        case Language.perl:
            return "main.pl"
        case Language.haskell:
            return "main.hs"
        case Language.other:
            return "main"

class ExploitStatus(Enum):
    active = 'active'
    disabled = 'disabled'

class AttackExecutionStatus(Enum):
    done = 'done'
    noflags = 'noflags'
    crashed = 'crashed'

class AttackMode(Enum):
    WAIT_FOR_TIME_TICK = "wait-for-time-tick"
    TICK_DELAY = "tick-delay"
    LOOP_DELAY = "loop-delay"

class SetupStatus(Enum):
    SETUP = "setup"
    RUNNING = "running"
