from pydantic import BaseModel, validate_call
from uuid import UUID, uuid4
import os, toml
from exploitfarm.utils import DEFAULT_SERVER_PORT, create_lock
from exploitfarm.model import Language, Language
from typing_extensions import Self
from exploitfarm.model import get_lang, get_interpreter, get_filename, get_default_file
from exploitfarm.model import ExploitStatus
from datetime import datetime, timedelta

class _ClientServerConfig(BaseModel):
    https: bool = False
    address: str|None = None
    port: int = DEFAULT_SERVER_PORT
    auth_key: str|None = None

class ClientConfig(BaseModel):
    client_id: UUID|None = None
    client_name: str|None = None
    server: _ClientServerConfig = _ClientServerConfig()
    last_status_api: dict|None = None
    
    __status_fetched = False
    
    @property
    def reqs(self):
        from exploitfarm.utils.reqs import Reqs
        return Reqs(self)

    @classmethod
    def read(cls) -> Self:
        return read_client_config()
    
    def write(self):
        write_client_config(self)
    
    def test_server(self) -> bool:
        try:
            self.fetch_status()
            return True
        except Exception:
            return False
        
    def fetch_status(self, save:bool = True) -> dict:
        self.last_status_api = self.reqs.status()
        if save:
            self.write()
            self.__status_fetched = True
        return self.last_status_api

    def delta_fetch_status(self) -> timedelta:
        init_request = datetime.now()
        self.fetch_status()
        return datetime.now() - init_request

    @property
    def fetched(self) -> bool:
        return self.__status_fetched
    
    @property
    def status(self) -> dict:
        if self.last_status_api is None:
            return self.fetch_status()
        return self.last_status_api

CLIENT_CONFIG_LOCK = create_lock(os.path.expanduser("~/.exploitfarm/client_config.json.lock"))

def _ensure_config_dir():
    config_dir = os.path.expanduser("~/.exploitfarm")
    if not os.path.exists(config_dir): os.makedirs(config_dir)
    
def write_client_config(config:ClientConfig):
    _ensure_config_dir()
    config_path = os.path.expanduser("~/.exploitfarm/client_config.json")
    with open(config_path, "w") as f:
        f.write(config.model_dump_json())

def read_client_config() -> ClientConfig:
    _ensure_config_dir()
    config_path = os.path.expanduser("~/.exploitfarm/client_config.json")
    if not os.path.isfile(config_path):
        return ClientConfig()
    with open(config_path, "r") as f:
        return ClientConfig.model_validate_json(f.read())
    
# Exploit config

EXPLOIT_CONFIG_REGEX = r"^[a-zA-Z0-9_\-]+$"
EXPLOIT_CONFIG_FILE_NAME = "config.toml"

class ExploitConfig(BaseModel):
    uuid: UUID
    name: str
    interpreter: str|None = None
    run: str
    language: Language
    service: UUID
    
    __exploit_lock = None
    
    @classmethod
    @validate_call()
    def new(cls, name:str, lang: Language|str, srv:UUID) -> Self:
        if isinstance(lang, str):
            lang = get_lang(lang)
        return cls(
            uuid=uuid4(),
            name=name,
            language=lang,
            service=srv,
            interpreter=get_interpreter(lang),
            run=get_filename(lang)
        )
    
    def model_dump_toml(self) -> str:
        return toml.dumps(self.model_dump(mode="json"))
    
    @classmethod
    def model_validate_toml(cls, data:str) -> Self:
        return cls(**toml.loads(data))
    
    def write(self, path:str):
        if not os.path.exists(path):
            os.makedirs(path)
            with open(os.path.join(path, get_filename(self.language)), "wt") as f:
                f.write(get_default_file(self.language))
        with open(os.path.join(path, EXPLOIT_CONFIG_FILE_NAME), "w") as f:
            f.write(self.model_dump_toml())
    
    def publish_exploit(self, config: ClientConfig):
        config.reqs.new_exploit({
            "id": self.uuid,
            "name": self.name,
            "language": self.language.value,
            "status": ExploitStatus.disabled.value,
            "service": self.service,
            "created_by": config.client_id
        })
        
    @property
    def lock(self):
        if self.__exploit_lock is None:
            self.__exploit_lock = create_lock(os.path.expanduser(f"~/.exploitfarm/exploit-{self.uuid}.lock"))
        return self.__exploit_lock
    
    def lock_exploit(self):
        return self.lock.acquire(blocking=False)
    
    def release_exploit(self):
        if self.lock.acquired:
            return self.lock.release()
    
    @classmethod
    def read(cls, path:str) -> Self:
        with open(os.path.join(path, EXPLOIT_CONFIG_FILE_NAME), "r") as f:
            return cls.model_validate_toml(f.read())
        

def check_exploit_config_exists(path:str) -> bool:
    path = os.path.join(os.getcwd(), path)
    if not os.path.exists(path):
        return False
    if not os.path.isdir(path):
        return False
    if not os.path.isfile(os.path.join(path, "config.toml")):
        return False
    return True


