#!/usr/bin/env python3

"""
Some of this code is taken by https://github.com/DestructiveVoice/DestructiveFarm/blob/master/client/start_sploit.py
And that's the reason why this submitter is called "exploitfarm"
"""

import dateutil.parser
import os, re, orjson
import subprocess, time, threading, traceback
from concurrent.futures import ThreadPoolExecutor
from math import ceil
from exploitfarm.utils.config import ClientConfig, ExploitConfig
from multiprocessing import Queue
from datetime import datetime as dt, timedelta
import datetime, io
from exploitfarm.model import AttackExecutionStatus, AttackMode
from rich import print
from uuid import UUID
from copy import deepcopy
from rich.markup import escape

os_windows = (os.name == 'nt')

class g:
    pool_size:int = 50
    config: ClientConfig
    shared_memory: dict = None
    submit_pool_timeout: int = 3
    server_status_refresh_period: int = 5
    exploit_config: ExploitConfig = None
    print_queue: Queue = None
    last_attack_time: dt = None
    config_update_event:Queue = Queue(100)
    attack_storage:"AttackStorage" = None 
    instance_storage:"InstanceStorage" = None
    exit_event = None
    server_id = None
    restart_event = None

class InstanceStorage:
    """
    Storage comprised of a dictionary of all running sploit instances and some statistics.

    Always acquire instance_lock before using this class. Do not release the lock
    between actual spawning/killing a process and calling register_start()/register_stop().
    """

    def __init__(self):
        self._counter = 0
        self._instances = {}

        self.n_completed = 0
        self.n_killed = 0
        self.lock = threading.RLock()

    @property
    def instances(self):
        return self._instances

    def register_start(self, process):
        with self.lock:
            instance_id = self._counter
            self._instances[instance_id] = process
            self._counter += 1
            return instance_id

    def register_stop(self, instance_id, was_killed):
        with self.lock:
            del self._instances[instance_id]

            self.n_completed += 1
            self.n_killed += was_killed

class AttackStorage:
    """
    Thread-safe storage comprised of a set and a post queue.

    Any number of threads may call add(), but only one "consumer thread"
    may call pick_flags() and mark_as_sent().
    """

    def __init__(self, client_id: UUID|None = None):
        self._queue = []
        self._lock = threading.RLock()
        self.client_id = client_id

    def add(self,
        flags: list[str],
        team: int,
        start_time: dt,
        end_time: dt,
        status: AttackExecutionStatus,
        output: bytes,
    ):
        with self._lock:
            flag_cache:set = g.shared_memory.get("flag_cache", set())
            flags = [flag for flag in flags if flag not in flag_cache]
            flag_cache.update(flags)
            g.shared_memory["flag_cache"] = flag_cache
            team_info_list = g.shared_memory.get(f"team-{team}", [])
            execution_submit = {
                "start_time": start_time,
                "end_time": end_time,
                "status": status.value,
                "error": None if status == AttackExecutionStatus.done else output,
                "executed_by": self.client_id,
                "target": team,
                "flags": flags,
            }
            info_submit = execution_submit.copy()
            info_submit["flags"] = len(flags)
            team_info_list.append(info_submit)
            g.shared_memory[f"team-{team}"] = team_info_list
            
            self._queue.append(execution_submit)

    def pick_attacks(self):
        with self._lock:
            return self._queue[:]

    def mark_as_sent(self, count):
        with self._lock:
            self._queue = self._queue[count:]

    @property
    def queue_size(self):
        with self._lock:
            return len(self._queue)

WARNING_RUNTIME = 5

def qprint(*args, end="\n"):
    try:
        for arg in args:
            if isinstance(arg, str):
                g.print_queue.put(escape(arg), block=False)
            else:
                g.print_queue.put(arg, block=False)
        g.print_queue.put(end, block=False)
    except (KeyboardInterrupt, ValueError):
        traceback.print_exc()
        pass

class InvalidSploitError(Exception):
    pass


if os_windows:
    # By default, Ctrl+C does not work on Windows if we spawn subprocesses.
    # Here we fix that using WinApi. See https://stackoverflow.com/a/43095532

    import signal
    import ctypes
    from ctypes import wintypes

    kernel32 = ctypes.WinDLL("kernel32", use_last_error=True)

    # BOOL WINAPI HandlerRoutine(
    #   _In_ DWORD dwCtrlType
    # );
    PHANDLER_ROUTINE = ctypes.WINFUNCTYPE(wintypes.BOOL, wintypes.DWORD)

    win_ignore_ctrl_c = PHANDLER_ROUTINE()  # = NULL

    def _errcheck_bool(result, _, args):
        if not result:
            raise ctypes.WinError(ctypes.get_last_error())
        return args

    # BOOL WINAPI SetConsoleCtrlHandler(
    #   _In_opt_ PHANDLER_ROUTINE HandlerRoutine,
    #   _In_     BOOL             Add
    # );
    kernel32.SetConsoleCtrlHandler.errcheck = _errcheck_bool
    kernel32.SetConsoleCtrlHandler.argtypes = (PHANDLER_ROUTINE, wintypes.BOOL)

    @PHANDLER_ROUTINE
    def win_ctrl_handler(dwCtrlType):
        if dwCtrlType == signal.CTRL_C_EVENT:
            kernel32.SetConsoleCtrlHandler(win_ignore_ctrl_c, True)
            shutdown()
        return False

    kernel32.SetConsoleCtrlHandler(win_ctrl_handler, True)


class APIException(Exception):
    pass

def once_in_a_period(period):
    iter_no = 1
    while(True):
        start_time = time.time()
        yield iter_no

        time_spent = time.time() - start_time
        if period > time_spent:
            if g.exit_event.wait(timeout=period - time_spent): return
        iter_no += 1

def repush_flags():
    try:
        with open(".flag_queue.json", "rt") as f:
            old_queue_data = orjson.loads(f.read())
        if old_queue_data["server_id"] != g.server_id:
            raise Exception("Server ID mismatch")
        for ele in old_queue_data["queue"]:
            g.attack_storage.add(ele["flags"], ele["target"], ele["start_time"], ele["end_time"], AttackExecutionStatus(ele["status"]), ele["error"])
    except Exception:
        if os.path.exists(".flag_queue.json"):
            os.remove(".flag_queue.json")

def update_server_config():
    try:
        for _ in once_in_a_period(g.server_status_refresh_period):
            try:
                g.shared_memory["runtime_timeout"] = calc_runtime_timeout()
                g.config.fetch_status()
                if g.config.status:
                    if not g.config.status["loggined"]:
                        qprint('Not loggined, login first to get the config and run the exploit')
                        g.shared_memory["config_update"] = False
                        continue
                    else:
                        g.shared_memory["config_update"] = True
                    if not g.server_id:
                        g.server_id = g.config.status["config"]["SERVER_ID"]
                        repush_flags()
                    else:
                        if g.server_id != g.config.status["config"]["SERVER_ID"]:
                            qprint('Server ID changed, restart the exploit')
                            shutdown(restart=True)
                            return
                    new_conf = g.config.status
                    new_conf_comp = deepcopy(new_conf)
                    old_conf = g.shared_memory["config"]
                    for ignore_key in ["server_time", "start_time", "end_time", "submitter", "services"]:
                        if new_conf_comp.get(ignore_key, None):
                            del new_conf_comp[ignore_key]
                        if old_conf.get(ignore_key, None):
                            del old_conf[ignore_key]
                    if new_conf_comp != old_conf:
                        qprint('--- Config updated! ---')
                        g.shared_memory["config"] = new_conf
                        g.config_update_event.put("update", block=False)
            except Exception as e:
                qprint(f"Can't get config from the server: {repr(e)}")
                g.shared_memory["config_update"] = False
    except Exception as e:
        qprint(traceback.format_exc())
        qprint(f'Config update loop died: {repr(e)}')
        shutdown()

def post_attacks(attacks):
    try:
        g.config.reqs.submit_flags(attacks, exploit=g.exploit_config.uuid)
        g.attack_storage.mark_as_sent(len(attacks))
        qprint(f'{sum([len(ele["flags"]) for ele in attacks])} flags posted to the server')
        g.shared_memory["submitter_status"] = True
    except Exception as e:
        g.shared_memory["submitter_status"] = False
        qprint(f"Can't post flags to the server: {repr(e)}")

def run_post_loop():
    try:
        for _ in once_in_a_period(g.submit_pool_timeout):
            attack_to_post = g.attack_storage.pick_attacks()[:50] #50 is to avoid the backend goes in timeout
            g.shared_memory["submitter_flags_in_queue"] = sum([len(ele["flags"]) for ele in attack_to_post])
            if attack_to_post: post_attacks(attack_to_post)
    except Exception as e:
        g.shared_memory["submitter_status"] = False
        qprint(traceback.format_exc())
        qprint(f'Posting loop died: {repr(e)}')

def process_sploit_filter(proc:subprocess.Popen, team: dict, start_time: dt, killed: bool, read_data:callable, final_output: bool = True):
    try:
        end_time = dt.now(datetime.timezone.utc)
        output = read_data()
        
        if isinstance(output, bytes):
            output = output.decode('utf-8', errors='replace')
        
        if final_output:
            qprint(f"Output of the sploit of {team['host']} killed[{killed}] status[{proc.returncode}]: '{team.get('short_name') or team.get('name') or team.get('id')}' started: {start_time} ended: {end_time}\n{output}")
        
        if not final_output and killed:
            qprint("☠️ xFarm killed the sploit due to timeout!")
        
        config = deepcopy(g.shared_memory["config"])
        flag_format = re.compile(config["config"]["FLAG_REGEX"])
        flags = list(map(str, set(flag_format.findall(output))))
        
        if killed or proc.returncode != 0:
            if killed:
                output = "THIS PROCESS HAS BEEN KILLED BY EXPLOITFARM DUE TO TIMEOUT\n-------- OUTPUT -------\n\n" + output
            g.attack_storage.add(flags, team["id"], start_time, end_time, AttackExecutionStatus.crashed, output)
        elif flags:
            g.attack_storage.add(flags, team["id"], start_time, end_time, AttackExecutionStatus.done, b'')
        else:
            g.attack_storage.add(flags, team["id"], start_time, end_time, AttackExecutionStatus.noflags, output)
    except Exception as e:
        traceback.print_exc()
        qprint(f'Failed to process sploit output: {repr(e)}')

def read_and_print(stdout:io.BytesIO, buffer:io.BytesIO):
    # Read stdout line by line in a separate thread
    for line in iter(stdout.readline, b''):
        qprint(line.decode('utf-8', errors='replace'), end="")  # Print to main stdout in real-time
        buffer.write(line)
        buffer.flush()
    buffer.write(stdout.read())
    stdout.close()

def launch_sploit(team:dict, max_runtime:float, stream_output:bool=False) -> tuple[subprocess.Popen, int, callable]:
    # For sploits written in Python, this env variable forces the interpreter to flush
    # stdout and stderr after each newline. Note that this is not default behavior
    # if the sploit's output is redirected to a pipe.
    env = os.environ.copy()
    env['PYTHONUNBUFFERED'] = '1'
    env["XFARM_HOST"] = team["host"]
    env["XFARM_TEAM"] = orjson.dumps(team)
    env["XFARM_RUNTIME"] = str(max_runtime)
    
    try:
        config = deepcopy(g.shared_memory["config"])
        services = config["services"]
        services = list(filter(lambda x: x["id"] == str(g.exploit_config.service), services))
        if len(services) > 0:
            env["XFARM_SERVICE"] = orjson.dumps(services[0])        
    except Exception as e:
        qprint(f"Can't get service info: {repr(e)}")

    command = f"{g.exploit_config.interpreter} {g.exploit_config.run}".strip()
    need_close_fds = (not os_windows)

    if os_windows:
        # On Windows, we block Ctrl+C handling, spawn the process, and
        # then recover the handler. This is the only way to make Ctrl+C
        # intercepted by us instead of our child processes.
        kernel32.SetConsoleCtrlHandler(win_ignore_ctrl_c, True)

    proc = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=need_close_fds, env=env, shell=True)
    read_function = None
    if stream_output:
        pipe = proc.stdout
        buffer = io.BytesIO()
        thr = threading.Thread(target=read_and_print, args=(pipe, buffer))
        thr.start()
        def func():
            thr.join()
            return buffer.getvalue()
        read_function = func
    if os_windows:
        kernel32.SetConsoleCtrlHandler(win_ignore_ctrl_c, False)
        
    if not read_function:
        def func():
            return proc.stdout.read()
        read_function = func

    return proc, g.instance_storage.register_start(proc), read_function


def run_sploit(team:dict, max_runtime:float, stream_output:bool=False):
    start_time = dt.now(datetime.timezone.utc)
    if g.exit_event.is_set(): return
    try:
        g.shared_memory["team-"+str(team["id"])+"-executing"] = True
        proc, instance_id, read_data = launch_sploit(team, max_runtime, stream_output)
    except Exception as e:
        if isinstance(e, FileNotFoundError):
            qprint(f'Sploit file or the interpreter for it not found: {repr(e)}')
        else:
            qprint(traceback.format_exc())
            qprint(f'Failed to run sploit: {repr(e)}')
        return

    try:
        need_kill = False
        try:
            start_time_exec = time.time()
            for _ in once_in_a_period(0.1):
                if not proc.poll() is None:
                    break
                if (time.time() - start_time_exec) > max_runtime:
                    need_kill = True
                    break
            if g.exit_event.is_set(): return
        except subprocess.TimeoutExpired:
            need_kill = True
        with g.instance_storage.lock:
            proc.kill()
        
        g.shared_memory["team-"+str(team["id"])+"-executing"] = False
        process_sploit_filter(proc, team, start_time, need_kill, read_data, not stream_output)
        g.instance_storage.register_stop(instance_id, need_kill)
    except Exception as e:
        qprint(traceback.format_exc())
        qprint(f'Failed to finish sploit: {repr(e)}')

def next_timout() -> float|int:
    config = deepcopy(g.shared_memory["config"])
    attack_mode = AttackMode(config["config"]["ATTACK_MODE"])
    this_time = dt.now(datetime.timezone.utc)
    if not g.last_attack_time:
        return 0 #Start the attack immediately
    match attack_mode:
        case AttackMode.TICK_DELAY:
            timeout = config["config"]["TICK_DURATION"] - (this_time - g.last_attack_time).total_seconds()
            return timeout if timeout > 0 else 0
        case AttackMode.WAIT_FOR_TIME_TICK:
            start_time = dateutil.parser.parse(config["config"]["START_TIME"])
            tick = 1
            next_time:dt = None
            if start_time > this_time:
                qprint("Waiting for the start time")
                return (start_time - this_time).total_seconds()
            while True:
                next_time = start_time + timedelta(seconds=config["config"]["TICK_DURATION"] * tick) + timedelta(seconds=config["config"]["ATTACK_TIME_TICK_DELAY"])
                if next_time > g.last_attack_time:
                    break
                tick += 1
            return (next_time - g.last_attack_time).total_seconds()            
        case AttackMode.LOOP_DELAY:
            timeout = config["config"]["LOOP_ATTACK_DELAY"] - (this_time - g.last_attack_time).total_seconds()
            return timeout if timeout > 0 else 0

def get_teams():
    return [ele for ele in deepcopy(g.shared_memory["config"]["teams"]) if g.shared_memory.get("team-"+str(ele["id"])+"-enabled", True)]

def calc_runtime_timeout():
    config = deepcopy(g.shared_memory["config"])
    teams = len(get_teams())
    if teams == 0:
        return 0
    result = ceil(config["config"]["TICK_DURATION"] / (teams / g.pool_size))
    if result < WARNING_RUNTIME:
        return WARNING_RUNTIME
    return result

def set_http_timeout():
    import exploitfarm.utils.reqs
    exploitfarm.utils.reqs.HTTP_TIMEOUT = 30

def xploit(path: str):
    set_http_timeout()
    os.chdir(path)
    threading.Thread(target=run_post_loop).start()
    threading.Thread(target=update_server_config).start()
    pool = ThreadPoolExecutor(max_workers=g.pool_size)
    try:
        while True:
            
            if g.instance_storage.n_completed > 0:
                qprint('Total {:.1f}% of instances ran out of time'.format(
                    float(g.instance_storage.n_killed) / g.instance_storage.n_completed * 100))
            
            max_runtime = calc_runtime_timeout()
            
            if max_runtime == WARNING_RUNTIME:
                qprint(f"⚠️ WARNING: The runtime of the attack is too low: consider increasing the number thread pool size")
            g.shared_memory["runtime_timeout"] = max_runtime
            for team in get_teams():
                pool.submit(run_sploit, team, max_runtime)
            
            g.last_attack_time = dt.now(datetime.timezone.utc)
            try:
                timeout = next_timout()
                g.shared_memory["next_timeout"] = timeout
                g.shared_memory["last_attack"] = g.last_attack_time
                g.shared_memory["next_attack_at"] = g.last_attack_time + timedelta(seconds=timeout)
                qprint(f"--> Next attacks will be in {timeout} seconds")
                g.config_update_event.get(timeout=timeout) #If the config is updated, all attacks will be "restarted"
            except Exception:
                pass
    except (KeyboardInterrupt, RuntimeError):
        pass
    finally:
        shutdown()

def run_printer_queue():
    try:
        while True:
            print(g.print_queue.get(), end="")
    except Exception:
        pass

def shutdown(restart:bool=False):
    g.config_update_event.put("shutdown")
    # Kill all child processes (so consume_sploit_ouput and run_sploit also will stop)
    with g.instance_storage.lock:
        for proc in g.instance_storage.instances.values():
            proc.kill()
    if restart:
        g.restart_event.set()
    g.exit_event.set()
    try:
        data = {"server_id": g.server_id, "queue":g.attack_storage.pick_attacks()}
        if data:
            with open(".flag_queue.json", "wb") as f:
                f.write(orjson.dumps(data))
        else:
            if os.path.exists(".flag_queue.json"):
                os.remove(".flag_queue.json")
    except Exception as e:
        pass
        
    #os._exit(0)

def xploit_one(config: ClientConfig, team: str, path: str, timeout: float = 30):
    set_http_timeout()
    os.chdir(path)
    try:
        g.config = config
        try:
            g.config.fetch_status()
        except Exception as e:
            qprint(f"[bold yellow]Can't get config from the server: {repr(e)}")
            return
        try:
            g.shared_memory = {"config": g.config.status}
        except Exception as e:
            qprint(f"[bold yellow]Can't get config from the server: {repr(e)}")
            g.shared_memory = {}
        g.exploit_config = ExploitConfig.read(path)
        g.print_queue = Queue(100)
        g.attack_storage = AttackStorage(config.client_id)
        g.instance_storage = InstanceStorage()
        g.exit_event = threading.Event()
        g.restart_event = threading.Event()
        threading.Thread(target=run_printer_queue).start()
        run_sploit({"id":0, "host":team}, timeout, stream_output=True)
        try:
            while True:
                attacks = g.attack_storage.pick_attacks()
                if len(attacks) == 0:
                    break
                for attack in attacks:
                    if len(attack["flags"]) > 0:
                        g.config.reqs.submit_flags({"flags":attack["flags"]})
                    g.attack_storage.mark_as_sent(1)
                    if len(attack["flags"]) > 0:
                        qprint(f"Submitted {len(attack["flags"])} flags as manual submissions.")
        except Exception as e:
            qprint(f"[bold yellow]Can't submit flags due missing config")
    except KeyboardInterrupt:
        pass
    finally:
        shutdown()
        for _ in once_in_a_period(0.1):
            if g.print_queue.empty():
                g.print_queue.close()
        


def start_xploit(config: ClientConfig, shared_dict:dict, print_queue: Queue, pool_size:int=50, path:str="./", submit_pool_timeout:int=3, server_status_refresh_period: int = 5, exit_event: threading.Event = None, restart_event: threading.Event = None) -> threading.Thread:
    g.config = config
    g.pool_size = pool_size
    g.shared_memory = shared_dict
    g.submit_pool_timeout = submit_pool_timeout
    g.server_status_refresh_period = server_status_refresh_period
    g.exploit_config = ExploitConfig.read(path)
    g.print_queue = print_queue
    g.attack_storage = AttackStorage(config.client_id)
    g.instance_storage = InstanceStorage()
    g.exit_event = exit_event if exit_event else threading.Event()
    g.restart_event = restart_event if restart_event else threading.Event()
    
    thr = threading.Thread(target=xploit, args=(path,))
    thr.start()
    return thr
