"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRecordManager = void 0;
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const events = require("aws-cdk-lib/aws-events");
const events_targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambda_es = require("aws-cdk-lib/aws-lambda-event-sources");
const sqs = require("aws-cdk-lib/aws-sqs");
const customresources = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * An event-driven serverless app to maintain a list of public ips in a Route 53
 * hosted zone.
 */
class TaskRecordManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Poison pills go here.
        const deadLetterQueue = new sqs.Queue(this, 'EventsDL', {
            retentionPeriod: aws_cdk_lib_1.Duration.days(14),
        });
        // Time limit for processing queue items - we set the lambda time limit to
        // this value as well.
        const eventsQueueVisibilityTimeout = aws_cdk_lib_1.Duration.seconds(30);
        // This queue lets us batch together ecs task state events. This is useful
        // for when when we would be otherwise bombarded by them.
        const eventsQueue = new sqs.Queue(this, 'EventsQueue', {
            deadLetterQueue: {
                maxReceiveCount: 500,
                queue: deadLetterQueue,
            },
            visibilityTimeout: eventsQueueVisibilityTimeout,
        });
        // Storage for task and record set information.
        const recordsTable = new dynamodb.Table(this, 'Records', {
            partitionKey: {
                name: 'cluster_service',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // Put the cluster's task state changes events into the queue.
        const runningEventRule = new events.Rule(this, 'RuleRunning', {
            eventPattern: {
                source: ['aws.ecs'],
                detailType: ['ECS Task State Change'],
                detail: {
                    clusterArn: [props.service.cluster.clusterArn],
                    lastStatus: ['RUNNING'],
                    desiredStatus: ['RUNNING'],
                },
            },
            targets: [
                new events_targets.SqsQueue(eventsQueue),
            ],
        });
        const stoppedEventRule = new events.Rule(this, 'RuleStopped', {
            eventPattern: {
                source: ['aws.ecs'],
                detailType: ['ECS Task State Change'],
                detail: {
                    clusterArn: [props.service.cluster.clusterArn],
                    lastStatus: ['STOPPED'],
                    desiredStatus: ['STOPPED'],
                },
            },
            targets: [
                new events_targets.SqsQueue(eventsQueue),
            ],
        });
        // Shared codebase for the lambdas.
        const code = lambda.Code.fromAsset(path.join(__dirname, '..', '..', '..', 'lambda', 'assign-public-ip'), {
            exclude: [
                '.coverage',
                '*.pyc',
                '.idea',
            ],
        });
        // Fully qualified domain name of the record
        const recordFqdn = aws_cdk_lib_1.Fn.join('.', [props.dnsRecordName, props.dnsZone.zoneName]);
        // Allow access to manage a zone's records.
        const dnsPolicyStatement = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'route53:ChangeResourceRecordSets',
                'route53:ListResourceRecordSets',
            ],
            resources: [props.dnsZone.hostedZoneArn],
        });
        // This function consumes events from the event queue and does the work of
        // querying task IP addresses and creating, updating record sets. When there
        // are zero tasks, it deletes the record set.
        const eventHandler = new lambda.Function(this, 'EventHandler', {
            code: code,
            handler: 'index.queue_handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: eventsQueueVisibilityTimeout,
            // Single-concurrency to prevent a race to set the RecordSet
            reservedConcurrentExecutions: 1,
            environment: {
                HOSTED_ZONE_ID: props.dnsZone.hostedZoneId,
                RECORD_NAME: recordFqdn,
                RECORDS_TABLE: recordsTable.tableName,
                CLUSTER_ARN: props.service.cluster.clusterArn,
                SERVICE_NAME: props.service.serviceName,
            },
            events: [
                new lambda_es.SqsEventSource(eventsQueue),
            ],
            initialPolicy: [
                // Look up task IPs
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['ec2:DescribeNetworkInterfaces'],
                    resources: ['*'],
                }),
                dnsPolicyStatement,
            ],
        });
        recordsTable.grantReadWriteData(eventHandler);
        // The lambda for a custom resource provider that deletes dangling record
        // sets when the stack is deleted.
        const cleanupResourceProviderHandler = new lambda.Function(this, 'CleanupResourceProviderHandler', {
            code: code,
            handler: 'index.cleanup_resource_handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            initialPolicy: [
                dnsPolicyStatement,
            ],
        });
        const cleanupResourceProvider = new customresources.Provider(this, 'CleanupResourceProvider', {
            onEventHandler: cleanupResourceProviderHandler,
        });
        const cleanupResource = new aws_cdk_lib_1.CustomResource(this, 'Cleanup', {
            serviceToken: cleanupResourceProvider.serviceToken,
            properties: {
                HostedZoneId: props.dnsZone.hostedZoneId,
                RecordName: recordFqdn,
            },
        });
        // Prime the event queue with a message so that changes to dns config are
        // quickly applied.
        const primingSdkCall = {
            service: 'SQS',
            action: 'sendMessage',
            parameters: {
                QueueUrl: eventsQueue.queueUrl,
                DelaySeconds: 10,
                MessageBody: '{ "prime": true }',
                // Add the hosted zone id and record name so that priming occurs with
                // dns config updates.
                MessageAttributes: {
                    HostedZoneId: { DataType: 'String', StringValue: props.dnsZone.hostedZoneId },
                    RecordName: { DataType: 'String', StringValue: props.dnsRecordName },
                },
            },
            physicalResourceId: customresources.PhysicalResourceId.fromResponse('MessageId'),
        };
        const primingCall = new customresources.AwsCustomResource(this, 'PrimingCall', {
            onCreate: primingSdkCall,
            onUpdate: primingSdkCall,
            policy: customresources.AwsCustomResourcePolicy.fromStatements([
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['sqs:SendMessage'],
                    resources: [eventsQueue.queueArn],
                }),
            ]),
        });
        // Send the priming call after the handler is created/updated.
        primingCall.node.addDependency(eventHandler);
        // Ensure that the cleanup resource is deleted last (so it can clean up)
        props.service.taskDefinition.node.addDependency(cleanupResource);
        // Ensure that the event rules are created first so we can catch the first
        // state transitions.
        props.service.taskDefinition.node.addDependency(runningEventRule);
        props.service.taskDefinition.node.addDependency(stoppedEventRule);
    }
}
exports.TaskRecordManager = TaskRecordManager;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFzay1yZWNvcmQtbWFuYWdlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9leHRlbnNpb25zL2Fzc2lnbi1wdWJsaWMtaXAvdGFzay1yZWNvcmQtbWFuYWdlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsNkNBQTBFO0FBQzFFLHFEQUFxRDtBQUVyRCxpREFBaUQ7QUFDakQsaUVBQWlFO0FBQ2pFLDJDQUEyQztBQUMzQyxpREFBaUQ7QUFDakQsa0VBQWtFO0FBRWxFLDJDQUEyQztBQUMzQyxnRUFBZ0U7QUFDaEUsMkNBQXVDO0FBUXZDOzs7R0FHRztBQUNILE1BQWEsaUJBQWtCLFNBQVEsc0JBQVM7SUFDOUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNyRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLHdCQUF3QjtRQUN4QixNQUFNLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN0RCxlQUFlLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1NBQ25DLENBQUMsQ0FBQztRQUVILDBFQUEwRTtRQUMxRSxzQkFBc0I7UUFDdEIsTUFBTSw0QkFBNEIsR0FBRyxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUUxRCwwRUFBMEU7UUFDMUUseURBQXlEO1FBQ3pELE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ3JELGVBQWUsRUFBRTtnQkFDZixlQUFlLEVBQUUsR0FBRztnQkFDcEIsS0FBSyxFQUFFLGVBQWU7YUFDdkI7WUFDRCxpQkFBaUIsRUFBRSw0QkFBNEI7U0FDaEQsQ0FBQyxDQUFDO1FBRUgsK0NBQStDO1FBQy9DLE1BQU0sWUFBWSxHQUFHLElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3ZELFlBQVksRUFBRTtnQkFDWixJQUFJLEVBQUUsaUJBQWlCO2dCQUN2QixJQUFJLEVBQUUsUUFBUSxDQUFDLGFBQWEsQ0FBQyxNQUFNO2FBQ3BDO1lBQ0QsV0FBVyxFQUFFLFFBQVEsQ0FBQyxXQUFXLENBQUMsZUFBZTtZQUNqRCxhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO1NBQ3JDLENBQUMsQ0FBQztRQUVILDhEQUE4RDtRQUM5RCxNQUFNLGdCQUFnQixHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzVELFlBQVksRUFBRTtnQkFDWixNQUFNLEVBQUUsQ0FBQyxTQUFTLENBQUM7Z0JBQ25CLFVBQVUsRUFBRSxDQUFDLHVCQUF1QixDQUFDO2dCQUNyQyxNQUFNLEVBQUU7b0JBQ04sVUFBVSxFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO29CQUM5QyxVQUFVLEVBQUUsQ0FBQyxTQUFTLENBQUM7b0JBQ3ZCLGFBQWEsRUFBRSxDQUFDLFNBQVMsQ0FBQztpQkFDM0I7YUFDRjtZQUNELE9BQU8sRUFBRTtnQkFDUCxJQUFJLGNBQWMsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO2FBQ3pDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUM1RCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLENBQUMsU0FBUyxDQUFDO2dCQUNuQixVQUFVLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQztnQkFDckMsTUFBTSxFQUFFO29CQUNOLFVBQVUsRUFBRSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQztvQkFDOUMsVUFBVSxFQUFFLENBQUMsU0FBUyxDQUFDO29CQUN2QixhQUFhLEVBQUUsQ0FBQyxTQUFTLENBQUM7aUJBQzNCO2FBQ0Y7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsSUFBSSxjQUFjLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQzthQUN6QztTQUNGLENBQUMsQ0FBQztRQUVILG1DQUFtQztRQUNuQyxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsa0JBQWtCLENBQUMsRUFBRTtZQUN2RyxPQUFPLEVBQUU7Z0JBQ1AsV0FBVztnQkFDWCxPQUFPO2dCQUNQLE9BQU87YUFDUjtTQUNGLENBQUMsQ0FBQztRQUVILDRDQUE0QztRQUM1QyxNQUFNLFVBQVUsR0FBRyxnQkFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUUvRSwyQ0FBMkM7UUFDM0MsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDakQsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1Asa0NBQWtDO2dCQUNsQyxnQ0FBZ0M7YUFDakM7WUFDRCxTQUFTLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQztTQUN6QyxDQUFDLENBQUM7UUFFSCwwRUFBMEU7UUFDMUUsNEVBQTRFO1FBQzVFLDZDQUE2QztRQUM3QyxNQUFNLFlBQVksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUM3RCxJQUFJLEVBQUUsSUFBSTtZQUNWLE9BQU8sRUFBRSxxQkFBcUI7WUFDOUIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxPQUFPLEVBQUUsNEJBQTRCO1lBQ3JDLDREQUE0RDtZQUM1RCw0QkFBNEIsRUFBRSxDQUFDO1lBQy9CLFdBQVcsRUFBRTtnQkFDWCxjQUFjLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZO2dCQUMxQyxXQUFXLEVBQUUsVUFBVTtnQkFDdkIsYUFBYSxFQUFFLFlBQVksQ0FBQyxTQUFTO2dCQUNyQyxXQUFXLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsVUFBVTtnQkFDN0MsWUFBWSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVzthQUN4QztZQUNELE1BQU0sRUFBRTtnQkFDTixJQUFJLFNBQVMsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDO2FBQzFDO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLG1CQUFtQjtnQkFDbkIsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO29CQUN4QixPQUFPLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQztvQkFDMUMsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNqQixDQUFDO2dCQUNGLGtCQUFrQjthQUNuQjtTQUNGLENBQUMsQ0FBQztRQUNILFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUU5Qyx5RUFBeUU7UUFDekUsa0NBQWtDO1FBQ2xDLE1BQU0sOEJBQThCLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxnQ0FBZ0MsRUFBRTtZQUNqRyxJQUFJLEVBQUUsSUFBSTtZQUNWLE9BQU8sRUFBRSxnQ0FBZ0M7WUFDekMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzVCLGFBQWEsRUFBRTtnQkFDYixrQkFBa0I7YUFDbkI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLHVCQUF1QixHQUFHLElBQUksZUFBZSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUU7WUFDNUYsY0FBYyxFQUFFLDhCQUE4QjtTQUMvQyxDQUFDLENBQUM7UUFFSCxNQUFNLGVBQWUsR0FBRyxJQUFJLDRCQUFjLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMxRCxZQUFZLEVBQUUsdUJBQXVCLENBQUMsWUFBWTtZQUNsRCxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWTtnQkFDeEMsVUFBVSxFQUFFLFVBQVU7YUFDdkI7U0FDRixDQUFDLENBQUM7UUFFSCx5RUFBeUU7UUFDekUsbUJBQW1CO1FBQ25CLE1BQU0sY0FBYyxHQUErQjtZQUNqRCxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSxhQUFhO1lBQ3JCLFVBQVUsRUFBRTtnQkFDVixRQUFRLEVBQUUsV0FBVyxDQUFDLFFBQVE7Z0JBQzlCLFlBQVksRUFBRSxFQUFFO2dCQUNoQixXQUFXLEVBQUUsbUJBQW1CO2dCQUNoQyxxRUFBcUU7Z0JBQ3JFLHNCQUFzQjtnQkFDdEIsaUJBQWlCLEVBQUU7b0JBQ2pCLFlBQVksRUFBRSxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFO29CQUM3RSxVQUFVLEVBQUUsRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFO2lCQUNyRTthQUNGO1lBQ0Qsa0JBQWtCLEVBQUUsZUFBZSxDQUFDLGtCQUFrQixDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUM7U0FDakYsQ0FBQztRQUVGLE1BQU0sV0FBVyxHQUFHLElBQUksZUFBZSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDN0UsUUFBUSxFQUFFLGNBQWM7WUFDeEIsUUFBUSxFQUFFLGNBQWM7WUFDeEIsTUFBTSxFQUFFLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxjQUFjLENBQUM7Z0JBQzdELElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztvQkFDeEIsT0FBTyxFQUFFLENBQUMsaUJBQWlCLENBQUM7b0JBQzVCLFNBQVMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUM7aUJBQ2xDLENBQUM7YUFDSCxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsOERBQThEO1FBQzlELFdBQVcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRTdDLHdFQUF3RTtRQUN4RSxLQUFLLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ2pFLDBFQUEwRTtRQUMxRSxxQkFBcUI7UUFDckIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2xFLEtBQUssQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNwRSxDQUFDO0NBQ0Y7QUF2TEQsOENBdUxDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IEN1c3RvbVJlc291cmNlLCBEdXJhdGlvbiwgRm4sIFJlbW92YWxQb2xpY3kgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBkeW5hbW9kYiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZHluYW1vZGInO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MnO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgZXZlbnRzX3RhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxhbWJkYV9lcyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhLWV2ZW50LXNvdXJjZXMnO1xuaW1wb3J0ICogYXMgcm91dGU1MyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtcm91dGU1Myc7XG5pbXBvcnQgKiBhcyBzcXMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXNxcyc7XG5pbXBvcnQgKiBhcyBjdXN0b21yZXNvdXJjZXMgZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuZXhwb3J0IGludGVyZmFjZSBUYXNrUmVjb3JkTWFuYWdlclByb3BzIHtcbiAgc2VydmljZTogZWNzLkVjMlNlcnZpY2UgfCBlY3MuRmFyZ2F0ZVNlcnZpY2U7XG4gIGRuc1pvbmU6IHJvdXRlNTMuSUhvc3RlZFpvbmU7XG4gIGRuc1JlY29yZE5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBBbiBldmVudC1kcml2ZW4gc2VydmVybGVzcyBhcHAgdG8gbWFpbnRhaW4gYSBsaXN0IG9mIHB1YmxpYyBpcHMgaW4gYSBSb3V0ZSA1M1xuICogaG9zdGVkIHpvbmUuXG4gKi9cbmV4cG9ydCBjbGFzcyBUYXNrUmVjb3JkTWFuYWdlciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBUYXNrUmVjb3JkTWFuYWdlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIC8vIFBvaXNvbiBwaWxscyBnbyBoZXJlLlxuICAgIGNvbnN0IGRlYWRMZXR0ZXJRdWV1ZSA9IG5ldyBzcXMuUXVldWUodGhpcywgJ0V2ZW50c0RMJywge1xuICAgICAgcmV0ZW50aW9uUGVyaW9kOiBEdXJhdGlvbi5kYXlzKDE0KSxcbiAgICB9KTtcblxuICAgIC8vIFRpbWUgbGltaXQgZm9yIHByb2Nlc3NpbmcgcXVldWUgaXRlbXMgLSB3ZSBzZXQgdGhlIGxhbWJkYSB0aW1lIGxpbWl0IHRvXG4gICAgLy8gdGhpcyB2YWx1ZSBhcyB3ZWxsLlxuICAgIGNvbnN0IGV2ZW50c1F1ZXVlVmlzaWJpbGl0eVRpbWVvdXQgPSBEdXJhdGlvbi5zZWNvbmRzKDMwKTtcblxuICAgIC8vIFRoaXMgcXVldWUgbGV0cyB1cyBiYXRjaCB0b2dldGhlciBlY3MgdGFzayBzdGF0ZSBldmVudHMuIFRoaXMgaXMgdXNlZnVsXG4gICAgLy8gZm9yIHdoZW4gd2hlbiB3ZSB3b3VsZCBiZSBvdGhlcndpc2UgYm9tYmFyZGVkIGJ5IHRoZW0uXG4gICAgY29uc3QgZXZlbnRzUXVldWUgPSBuZXcgc3FzLlF1ZXVlKHRoaXMsICdFdmVudHNRdWV1ZScsIHtcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZToge1xuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDUwMCxcbiAgICAgICAgcXVldWU6IGRlYWRMZXR0ZXJRdWV1ZSxcbiAgICAgIH0sXG4gICAgICB2aXNpYmlsaXR5VGltZW91dDogZXZlbnRzUXVldWVWaXNpYmlsaXR5VGltZW91dCxcbiAgICB9KTtcblxuICAgIC8vIFN0b3JhZ2UgZm9yIHRhc2sgYW5kIHJlY29yZCBzZXQgaW5mb3JtYXRpb24uXG4gICAgY29uc3QgcmVjb3Jkc1RhYmxlID0gbmV3IGR5bmFtb2RiLlRhYmxlKHRoaXMsICdSZWNvcmRzJywge1xuICAgICAgcGFydGl0aW9uS2V5OiB7XG4gICAgICAgIG5hbWU6ICdjbHVzdGVyX3NlcnZpY2UnLFxuICAgICAgICB0eXBlOiBkeW5hbW9kYi5BdHRyaWJ1dGVUeXBlLlNUUklORyxcbiAgICAgIH0sXG4gICAgICBiaWxsaW5nTW9kZTogZHluYW1vZGIuQmlsbGluZ01vZGUuUEFZX1BFUl9SRVFVRVNULFxuICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIH0pO1xuXG4gICAgLy8gUHV0IHRoZSBjbHVzdGVyJ3MgdGFzayBzdGF0ZSBjaGFuZ2VzIGV2ZW50cyBpbnRvIHRoZSBxdWV1ZS5cbiAgICBjb25zdCBydW5uaW5nRXZlbnRSdWxlID0gbmV3IGV2ZW50cy5SdWxlKHRoaXMsICdSdWxlUnVubmluZycsIHtcbiAgICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgICBzb3VyY2U6IFsnYXdzLmVjcyddLFxuICAgICAgICBkZXRhaWxUeXBlOiBbJ0VDUyBUYXNrIFN0YXRlIENoYW5nZSddLFxuICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICBjbHVzdGVyQXJuOiBbcHJvcHMuc2VydmljZS5jbHVzdGVyLmNsdXN0ZXJBcm5dLFxuICAgICAgICAgIGxhc3RTdGF0dXM6IFsnUlVOTklORyddLFxuICAgICAgICAgIGRlc2lyZWRTdGF0dXM6IFsnUlVOTklORyddLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHRhcmdldHM6IFtcbiAgICAgICAgbmV3IGV2ZW50c190YXJnZXRzLlNxc1F1ZXVlKGV2ZW50c1F1ZXVlKSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBzdG9wcGVkRXZlbnRSdWxlID0gbmV3IGV2ZW50cy5SdWxlKHRoaXMsICdSdWxlU3RvcHBlZCcsIHtcbiAgICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgICBzb3VyY2U6IFsnYXdzLmVjcyddLFxuICAgICAgICBkZXRhaWxUeXBlOiBbJ0VDUyBUYXNrIFN0YXRlIENoYW5nZSddLFxuICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICBjbHVzdGVyQXJuOiBbcHJvcHMuc2VydmljZS5jbHVzdGVyLmNsdXN0ZXJBcm5dLFxuICAgICAgICAgIGxhc3RTdGF0dXM6IFsnU1RPUFBFRCddLFxuICAgICAgICAgIGRlc2lyZWRTdGF0dXM6IFsnU1RPUFBFRCddLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHRhcmdldHM6IFtcbiAgICAgICAgbmV3IGV2ZW50c190YXJnZXRzLlNxc1F1ZXVlKGV2ZW50c1F1ZXVlKSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICAvLyBTaGFyZWQgY29kZWJhc2UgZm9yIHRoZSBsYW1iZGFzLlxuICAgIGNvbnN0IGNvZGUgPSBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJy4uJywgJy4uJywgJ2xhbWJkYScsICdhc3NpZ24tcHVibGljLWlwJyksIHtcbiAgICAgIGV4Y2x1ZGU6IFtcbiAgICAgICAgJy5jb3ZlcmFnZScsXG4gICAgICAgICcqLnB5YycsXG4gICAgICAgICcuaWRlYScsXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgLy8gRnVsbHkgcXVhbGlmaWVkIGRvbWFpbiBuYW1lIG9mIHRoZSByZWNvcmRcbiAgICBjb25zdCByZWNvcmRGcWRuID0gRm4uam9pbignLicsIFtwcm9wcy5kbnNSZWNvcmROYW1lLCBwcm9wcy5kbnNab25lLnpvbmVOYW1lXSk7XG5cbiAgICAvLyBBbGxvdyBhY2Nlc3MgdG8gbWFuYWdlIGEgem9uZSdzIHJlY29yZHMuXG4gICAgY29uc3QgZG5zUG9saWN5U3RhdGVtZW50ID0gbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAncm91dGU1MzpDaGFuZ2VSZXNvdXJjZVJlY29yZFNldHMnLFxuICAgICAgICAncm91dGU1MzpMaXN0UmVzb3VyY2VSZWNvcmRTZXRzJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtwcm9wcy5kbnNab25lLmhvc3RlZFpvbmVBcm5dLFxuICAgIH0pO1xuXG4gICAgLy8gVGhpcyBmdW5jdGlvbiBjb25zdW1lcyBldmVudHMgZnJvbSB0aGUgZXZlbnQgcXVldWUgYW5kIGRvZXMgdGhlIHdvcmsgb2ZcbiAgICAvLyBxdWVyeWluZyB0YXNrIElQIGFkZHJlc3NlcyBhbmQgY3JlYXRpbmcsIHVwZGF0aW5nIHJlY29yZCBzZXRzLiBXaGVuIHRoZXJlXG4gICAgLy8gYXJlIHplcm8gdGFza3MsIGl0IGRlbGV0ZXMgdGhlIHJlY29yZCBzZXQuXG4gICAgY29uc3QgZXZlbnRIYW5kbGVyID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnRXZlbnRIYW5kbGVyJywge1xuICAgICAgY29kZTogY29kZSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5xdWV1ZV9oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzgsXG4gICAgICB0aW1lb3V0OiBldmVudHNRdWV1ZVZpc2liaWxpdHlUaW1lb3V0LFxuICAgICAgLy8gU2luZ2xlLWNvbmN1cnJlbmN5IHRvIHByZXZlbnQgYSByYWNlIHRvIHNldCB0aGUgUmVjb3JkU2V0XG4gICAgICByZXNlcnZlZENvbmN1cnJlbnRFeGVjdXRpb25zOiAxLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgSE9TVEVEX1pPTkVfSUQ6IHByb3BzLmRuc1pvbmUuaG9zdGVkWm9uZUlkLFxuICAgICAgICBSRUNPUkRfTkFNRTogcmVjb3JkRnFkbixcbiAgICAgICAgUkVDT1JEU19UQUJMRTogcmVjb3Jkc1RhYmxlLnRhYmxlTmFtZSxcbiAgICAgICAgQ0xVU1RFUl9BUk46IHByb3BzLnNlcnZpY2UuY2x1c3Rlci5jbHVzdGVyQXJuLFxuICAgICAgICBTRVJWSUNFX05BTUU6IHByb3BzLnNlcnZpY2Uuc2VydmljZU5hbWUsXG4gICAgICB9LFxuICAgICAgZXZlbnRzOiBbXG4gICAgICAgIG5ldyBsYW1iZGFfZXMuU3FzRXZlbnRTb3VyY2UoZXZlbnRzUXVldWUpLFxuICAgICAgXSxcbiAgICAgIGluaXRpYWxQb2xpY3k6IFtcbiAgICAgICAgLy8gTG9vayB1cCB0YXNrIElQc1xuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFsnZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICB9KSxcbiAgICAgICAgZG5zUG9saWN5U3RhdGVtZW50LFxuICAgICAgXSxcbiAgICB9KTtcbiAgICByZWNvcmRzVGFibGUuZ3JhbnRSZWFkV3JpdGVEYXRhKGV2ZW50SGFuZGxlcik7XG5cbiAgICAvLyBUaGUgbGFtYmRhIGZvciBhIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlciB0aGF0IGRlbGV0ZXMgZGFuZ2xpbmcgcmVjb3JkXG4gICAgLy8gc2V0cyB3aGVuIHRoZSBzdGFjayBpcyBkZWxldGVkLlxuICAgIGNvbnN0IGNsZWFudXBSZXNvdXJjZVByb3ZpZGVySGFuZGxlciA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ0NsZWFudXBSZXNvdXJjZVByb3ZpZGVySGFuZGxlcicsIHtcbiAgICAgIGNvZGU6IGNvZGUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguY2xlYW51cF9yZXNvdXJjZV9oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzgsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgaW5pdGlhbFBvbGljeTogW1xuICAgICAgICBkbnNQb2xpY3lTdGF0ZW1lbnQsXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY2xlYW51cFJlc291cmNlUHJvdmlkZXIgPSBuZXcgY3VzdG9tcmVzb3VyY2VzLlByb3ZpZGVyKHRoaXMsICdDbGVhbnVwUmVzb3VyY2VQcm92aWRlcicsIHtcbiAgICAgIG9uRXZlbnRIYW5kbGVyOiBjbGVhbnVwUmVzb3VyY2VQcm92aWRlckhhbmRsZXIsXG4gICAgfSk7XG5cbiAgICBjb25zdCBjbGVhbnVwUmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0NsZWFudXAnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGNsZWFudXBSZXNvdXJjZVByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgSG9zdGVkWm9uZUlkOiBwcm9wcy5kbnNab25lLmhvc3RlZFpvbmVJZCxcbiAgICAgICAgUmVjb3JkTmFtZTogcmVjb3JkRnFkbixcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBQcmltZSB0aGUgZXZlbnQgcXVldWUgd2l0aCBhIG1lc3NhZ2Ugc28gdGhhdCBjaGFuZ2VzIHRvIGRucyBjb25maWcgYXJlXG4gICAgLy8gcXVpY2tseSBhcHBsaWVkLlxuICAgIGNvbnN0IHByaW1pbmdTZGtDYWxsOiBjdXN0b21yZXNvdXJjZXMuQXdzU2RrQ2FsbCA9IHtcbiAgICAgIHNlcnZpY2U6ICdTUVMnLFxuICAgICAgYWN0aW9uOiAnc2VuZE1lc3NhZ2UnLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBRdWV1ZVVybDogZXZlbnRzUXVldWUucXVldWVVcmwsXG4gICAgICAgIERlbGF5U2Vjb25kczogMTAsXG4gICAgICAgIE1lc3NhZ2VCb2R5OiAneyBcInByaW1lXCI6IHRydWUgfScsXG4gICAgICAgIC8vIEFkZCB0aGUgaG9zdGVkIHpvbmUgaWQgYW5kIHJlY29yZCBuYW1lIHNvIHRoYXQgcHJpbWluZyBvY2N1cnMgd2l0aFxuICAgICAgICAvLyBkbnMgY29uZmlnIHVwZGF0ZXMuXG4gICAgICAgIE1lc3NhZ2VBdHRyaWJ1dGVzOiB7XG4gICAgICAgICAgSG9zdGVkWm9uZUlkOiB7IERhdGFUeXBlOiAnU3RyaW5nJywgU3RyaW5nVmFsdWU6IHByb3BzLmRuc1pvbmUuaG9zdGVkWm9uZUlkIH0sXG4gICAgICAgICAgUmVjb3JkTmFtZTogeyBEYXRhVHlwZTogJ1N0cmluZycsIFN0cmluZ1ZhbHVlOiBwcm9wcy5kbnNSZWNvcmROYW1lIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBjdXN0b21yZXNvdXJjZXMuUGh5c2ljYWxSZXNvdXJjZUlkLmZyb21SZXNwb25zZSgnTWVzc2FnZUlkJyksXG4gICAgfTtcblxuICAgIGNvbnN0IHByaW1pbmdDYWxsID0gbmV3IGN1c3RvbXJlc291cmNlcy5Bd3NDdXN0b21SZXNvdXJjZSh0aGlzLCAnUHJpbWluZ0NhbGwnLCB7XG4gICAgICBvbkNyZWF0ZTogcHJpbWluZ1Nka0NhbGwsXG4gICAgICBvblVwZGF0ZTogcHJpbWluZ1Nka0NhbGwsXG4gICAgICBwb2xpY3k6IGN1c3RvbXJlc291cmNlcy5Bd3NDdXN0b21SZXNvdXJjZVBvbGljeS5mcm9tU3RhdGVtZW50cyhbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogWydzcXM6U2VuZE1lc3NhZ2UnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtldmVudHNRdWV1ZS5xdWV1ZUFybl0sXG4gICAgICAgIH0pLFxuICAgICAgXSksXG4gICAgfSk7XG5cbiAgICAvLyBTZW5kIHRoZSBwcmltaW5nIGNhbGwgYWZ0ZXIgdGhlIGhhbmRsZXIgaXMgY3JlYXRlZC91cGRhdGVkLlxuICAgIHByaW1pbmdDYWxsLm5vZGUuYWRkRGVwZW5kZW5jeShldmVudEhhbmRsZXIpO1xuXG4gICAgLy8gRW5zdXJlIHRoYXQgdGhlIGNsZWFudXAgcmVzb3VyY2UgaXMgZGVsZXRlZCBsYXN0IChzbyBpdCBjYW4gY2xlYW4gdXApXG4gICAgcHJvcHMuc2VydmljZS50YXNrRGVmaW5pdGlvbi5ub2RlLmFkZERlcGVuZGVuY3koY2xlYW51cFJlc291cmNlKTtcbiAgICAvLyBFbnN1cmUgdGhhdCB0aGUgZXZlbnQgcnVsZXMgYXJlIGNyZWF0ZWQgZmlyc3Qgc28gd2UgY2FuIGNhdGNoIHRoZSBmaXJzdFxuICAgIC8vIHN0YXRlIHRyYW5zaXRpb25zLlxuICAgIHByb3BzLnNlcnZpY2UudGFza0RlZmluaXRpb24ubm9kZS5hZGREZXBlbmRlbmN5KHJ1bm5pbmdFdmVudFJ1bGUpO1xuICAgIHByb3BzLnNlcnZpY2UudGFza0RlZmluaXRpb24ubm9kZS5hZGREZXBlbmRlbmN5KHN0b3BwZWRFdmVudFJ1bGUpO1xuICB9XG59XG4iXX0=