"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecs = require("aws-cdk-lib/aws-ecs");
const constructs_1 = require("constructs");
const extension_interfaces_1 = require("./extensions/extension-interfaces");
/**
 * This Service construct serves as a Builder class for an ECS service. It
 * supports various extensions and keeps track of any mutating state, allowing
 * it to build up an ECS service progressively.
 */
class Service extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        /**
         * The flag to track if auto scaling policies have been configured
         * for the service.
         */
        this.autoScalingPoliciesEnabled = false;
        /**
         * The list of URLs associated with this service.
         */
        this.urls = {};
        this.scope = scope;
        this.id = id;
        this.environment = props.environment;
        this.vpc = props.environment.vpc;
        this.cluster = props.environment.cluster;
        this.capacityType = props.environment.capacityType;
        this.serviceDescription = props.serviceDescription;
        // Check to make sure that the user has actually added a container
        const containerextension = this.serviceDescription.get('service-container');
        if (!containerextension) {
            throw new Error(`Service '${this.id}' must have a Container extension`);
        }
        // First set the scope for all the extensions
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].prehook(this, this.scope);
            }
        }
        // At the point of preparation all extensions have been defined on the service
        // so give each extension a chance to now add hooks to other extensions if
        // needed
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].addHooks();
            }
        }
        // Give each extension a chance to mutate the task def creation properties
        let taskDefProps = {
            // Default CPU and memory
            cpu: '256',
            memory: '512',
            // Allow user to pre-define the taskRole so that it can be used in resource policies that may
            // be defined before the ECS service exists in a CDK application
            taskRole: props.taskRole,
            // Ensure that the task definition supports both EC2 and Fargate
            compatibility: ecs.Compatibility.EC2_AND_FARGATE,
        };
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                taskDefProps = this.serviceDescription.extensions[extensions].modifyTaskDefinitionProps(taskDefProps);
            }
        }
        // Now that the task definition properties are assembled, create it
        this.taskDefinition = new ecs.TaskDefinition(this.scope, `${this.id}-task-definition`, taskDefProps);
        // Now give each extension a chance to use the task definition
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].useTaskDefinition(this.taskDefinition);
            }
        }
        // Now that all containers are created, give each extension a chance
        // to bake its dependency graph
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].resolveContainerDependencies();
            }
        }
        // Give each extension a chance to mutate the service props before
        // service creation
        let serviceProps = {
            cluster: this.cluster,
            taskDefinition: this.taskDefinition,
            minHealthyPercent: 100,
            maxHealthyPercent: 200,
            desiredCount: (_b = props.desiredCount) !== null && _b !== void 0 ? _b : 1,
        };
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                serviceProps = this.serviceDescription.extensions[extensions].modifyServiceProps(serviceProps);
            }
        }
        // If a maxHealthyPercent and desired count has been set while minHealthyPercent == 100% then we
        // need to do some failsafe checking to ensure that the maxHealthyPercent
        // actually allows a rolling deploy. Otherwise it is possible to end up with
        // blocked deploys that can take no action because minHealtyhPercent == 100%
        // prevents running, healthy tasks from being stopped, but a low maxHealthyPercent
        // can also prevents new parallel tasks from being started.
        if (serviceProps.maxHealthyPercent && serviceProps.desiredCount && serviceProps.minHealthyPercent && serviceProps.minHealthyPercent == 100) {
            if (serviceProps.desiredCount == 1) {
                // If there is one task then we must allow max percentage to be at
                // least 200% for another replacement task to be added
                serviceProps = {
                    ...serviceProps,
                    maxHealthyPercent: Math.max(200, serviceProps.maxHealthyPercent),
                };
            }
            else if (serviceProps.desiredCount <= 3) {
                // If task count is 2 or 3 then max percent must be at least 150% to
                // allow one replacement task to be launched at a time.
                serviceProps = {
                    ...serviceProps,
                    maxHealthyPercent: Math.max(150, serviceProps.maxHealthyPercent),
                };
            }
            else {
                // For anything higher than 3 tasks set max percent to at least 125%
                // For 4 tasks this will allow exactly one extra replacement task
                // at a time, for any higher task count it will allow 25% of the tasks
                // to be replaced at a time.
                serviceProps = {
                    ...serviceProps,
                    maxHealthyPercent: Math.max(125, serviceProps.maxHealthyPercent),
                };
            }
        }
        // Set desiredCount to `undefined` if auto scaling is configured for the service
        if (props.autoScaleTaskCount || this.autoScalingPoliciesEnabled) {
            serviceProps = {
                ...serviceProps,
                desiredCount: undefined,
            };
        }
        // Now that the service props are determined we can create
        // the service
        if (this.capacityType === extension_interfaces_1.EnvironmentCapacityType.EC2) {
            this.ecsService = new ecs.Ec2Service(this.scope, `${this.id}-service`, serviceProps);
        }
        else if (this.capacityType === extension_interfaces_1.EnvironmentCapacityType.FARGATE) {
            this.ecsService = new ecs.FargateService(this.scope, `${this.id}-service`, serviceProps);
        }
        else {
            throw new Error(`Unknown capacity type for service ${this.id}`);
        }
        // Create the auto scaling target and configure target tracking policies after the service is created
        if (props.autoScaleTaskCount) {
            this.scalableTaskCount = this.ecsService.autoScaleTaskCount({
                maxCapacity: props.autoScaleTaskCount.maxTaskCount,
                minCapacity: props.autoScaleTaskCount.minTaskCount,
            });
            if (props.autoScaleTaskCount.targetCpuUtilization) {
                const targetCpuUtilizationPercent = props.autoScaleTaskCount.targetCpuUtilization;
                this.scalableTaskCount.scaleOnCpuUtilization(`${this.id}-target-cpu-utilization-${targetCpuUtilizationPercent}`, {
                    targetUtilizationPercent: targetCpuUtilizationPercent,
                });
                this.enableAutoScalingPolicy();
            }
            if (props.autoScaleTaskCount.targetMemoryUtilization) {
                const targetMemoryUtilizationPercent = props.autoScaleTaskCount.targetMemoryUtilization;
                this.scalableTaskCount.scaleOnMemoryUtilization(`${this.id}-target-memory-utilization-${targetMemoryUtilizationPercent}`, {
                    targetUtilizationPercent: targetMemoryUtilizationPercent,
                });
                this.enableAutoScalingPolicy();
            }
        }
        // Now give all extensions a chance to use the service
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].useService(this.ecsService);
            }
        }
        // Error out if the auto scaling target is created but no scaling policies have been configured
        if (this.scalableTaskCount && !this.autoScalingPoliciesEnabled) {
            throw Error(`The auto scaling target for the service '${this.id}' has been created but no auto scaling policies have been configured.`);
        }
    }
    /**
     * Tell extensions from one service to connect to extensions from
     * another sevice if they have implemented a hook for it.
     *
     * @param service
     */
    connectTo(service, connectToProps) {
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].connectToService(service, connectToProps);
            }
        }
    }
    /**
     * This method adds a new URL for the service. This allows extensions to
     * submit a URL for the service. For example, a load balancer might add its
     * URL, or App Mesh can add its DNS name for the service.
     *
     * @param urlName - The identifier name for this URL
     * @param url - The URL itself.
     */
    addURL(urlName, url) {
        this.urls[urlName] = url;
    }
    /**
     * Retrieve a URL for the service. The URL must have previously been
     * stored by one of the URL providing extensions.
     *
     * @param urlName - The URL to look up.
     */
    getURL(urlName) {
        if (!this.urls[urlName]) {
            throw new Error(`Unable to find a URL with name '${urlName}'`);
        }
        return this.urls[urlName];
    }
    /**
     * This helper method is used to set the `autoScalingPoliciesEnabled` attribute
     * whenever an auto scaling policy is configured for the service.
     */
    enableAutoScalingPolicy() {
        this.autoScalingPoliciesEnabled = true;
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.Service", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,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