"""Main Kiso task implementations."""

# ruff: noqa: ARG001
from __future__ import annotations

import copy
import json
import logging
import shutil
import subprocess
from collections import Counter, defaultdict
from dataclasses import fields
from functools import wraps
from ipaddress import IPv4Interface, IPv6Interface, ip_address
from pathlib import Path
from typing import TYPE_CHECKING, Any, Callable, TypeVar

import enoslib as en
import yaml
from dacite import Config, from_dict
from enoslib.objects import DefaultNetwork, Host, Networks, Roles
from enoslib.task import Environment, enostask
from jsonschema_pyref import ValidationError, validate
from rich.console import Console

import kiso.constants as const
from kiso import display, edge, utils
from kiso.configuration import Deployment, Kiso, Software
from kiso.errors import KisoError
from kiso.log import get_process_pool_executor
from kiso.schema import SCHEMA
from kiso.version import __version__

if TYPE_CHECKING:
    from os import PathLike

    from enoslib.infra.enos_chameleonedge.objects import ChameleonDevice
    from enoslib.infra.provider import Provider

    from kiso.configuration import ExperimentTypes


T = TypeVar("T")

PROVIDER_MAP: dict[str, tuple[Callable[[dict], Any], Callable[..., Any]]] = {}

log = logging.getLogger("kiso")

console = Console()

if hasattr(en, "Vagrant"):
    log.debug("Vagrant provider is available")
    PROVIDER_MAP["vagrant"] = (en.VagrantConf.from_dictionary, en.Vagrant)
if hasattr(en, "CBM"):
    log.debug("Chameleon Bare Metal provider is available")
    from enoslib.infra.enos_openstack.utils import source_credentials_from_rc_file

    PROVIDER_MAP["chameleon"] = (en.CBMConf.from_dictionary, en.CBM)
if hasattr(en, "ChameleonEdge"):
    log.debug("Chameleon Edge provider is available")

    PROVIDER_MAP["chameleon-edge"] = (
        en.ChameleonEdgeConf.from_dictionary,
        en.ChameleonEdge,
    )
if hasattr(en, "Fabric"):
    log.debug("FABRIC provider is available")
    from enoslib.infra.enos_fabric.utils import (
        source_credentials_from_rc_file as source_fabric_credentials_from_rc_file,
    )
    from fabrictestbed_extensions.fablib.fablib import FablibManager as fablib_manager

    PROVIDER_MAP["fabric"] = (en.FabricConf.from_dictionary, en.Fabric)


def validate_config(func: Callable[..., T]) -> Callable[..., T]:
    """Decorator to validate the experiment configuration against a predefined schema.

    Validates the experiment configuration by checking it against the Kiso experiment
    configuration schema. Supports configuration passed as a dictionary or a file path.

    :param func: The function to be decorated, which will receive the experiment
    configuration
    :type func: Callable[..., T]
    :return: A wrapped function that validates the configuration before executing the
    original function
    :rtype: Callable[..., T]
    :raises ValidationError: if the configuration is invalid
    """

    @wraps(func)
    def wrapper(experiment_config: PathLike | dict, *args: Any, **kwargs: Any) -> T:  # noqa: ANN401
        log.debug("Check Kiso experiment configuration")
        if isinstance(experiment_config, dict):
            config = experiment_config
            wd = Path.cwd().resolve()
        else:
            wd = Path(experiment_config).parent.resolve()
            with Path(experiment_config).open() as _experiment_config:
                config = yaml.safe_load(_experiment_config)

        try:
            validate(_replace_labels_key_with_roles_key(config), SCHEMA)
            # Convert the JSON configuration to a :py:class:`dataclasses.dataclass`
            config = from_dict(Kiso, config, Config(convert_key=_to_snake_case))
        except ValidationError:
            log.exception("Invalid Kiso experiment config <%s>", experiment_config)
            raise

        log.debug("Kiso experiment configuration is valid")
        return func(config, *args, wd=wd, **kwargs)

    return wrapper


def _replace_labels_key_with_roles_key(experiment_config: Kiso | dict) -> dict:
    """Replace labels with roles in the experiment configuration."""
    experiment_config = copy.deepcopy(experiment_config)
    sites = (
        experiment_config["sites"]
        if isinstance(experiment_config, dict)
        else experiment_config.sites
    )
    for site in sites:
        for machine in site["resources"]["machines"]:
            machine["roles"] = machine["labels"]
            del machine["labels"]

        for network in site["resources"].get("networks", []):
            if isinstance(network, str):
                continue

            network["roles"] = network["labels"]
            del network["labels"]

    return experiment_config


@validate_config
def check(experiment_config: Kiso, **kwargs: dict) -> None:
    """Check the experiment configuration for various validation criteria.

    This function performs multiple validation checks on the experiment configuration,
    including:
    - Verifying vagrant site constraints
    - Validating label definitions
    - Checking docker and HTCondor configurations
    - Ensuring proper node configurations
    - Validating input file locations
    - Performing EnOSlib platform checks

    :param experiment_config: The experiment configuration dictionary
    :type experiment_config: Kiso
    :param kwargs: Additional keyword arguments
    :type kwargs: dict
    """
    console.rule("[bold green]Check experiment configuration[/bold green]")
    log.debug("Check only one vagrant site is present in the experiment")
    label_to_machines: Roles = _get_defined_machines(experiment_config)

    _check_software(experiment_config.software, label_to_machines)
    _check_deployed_software(experiment_config.deployment, label_to_machines)
    _check_experiments(experiment_config, label_to_machines)

    log.debug("Check EnOSlib")
    en.MOTD = en.INFO = ""
    en.check(platform_filter=["Vagrant", "Fabric", "Chameleon", "ChameleonEdge"])


def _get_defined_machines(experiment_config: Kiso) -> Roles:
    """Get the defined machines from the experiment configuration.

    Extracts and counts labels defined in the sites section of the experiment
    configuration. Validates that only one Vagrant site is present and generates
    additional label variants.

    :param experiment_config: Configuration dictionary containing site and resource
    definitions
    :type experiment_config: Kiso
    :raises ValueError: If multiple Vagrant sites are detected
    :return: A counter of defined labels with their counts
    :rtype: Roles
    """
    vagrant_sites = 0
    def_labels: Counter = Counter()
    label_to_machines: Roles = defaultdict(set)

    for site_index, site in enumerate(experiment_config.sites):
        if site["kind"] == "vagrant":
            vagrant_sites += 1

        for machine_index, machine in enumerate(site["resources"]["machines"]):
            def_labels.update({site["kind"]: machine.get("number", 1)})

            for label in machine["labels"]:
                def_labels.update({label: machine.get("number", 1)})

            for index in range(machine.get("number", 1)):
                machine_key = Host(
                    f"site-{site_index}-machine-{machine_index}-index-{index}"
                )
                label_to_machines[site["kind"]].add(machine_key)

                for label in machine["labels"]:
                    label_to_machines[label].add(machine_key)

    else:
        if vagrant_sites > 1:
            raise ValueError("Multiple vagrant sites are not supported")

        extra_labels = {}
        for label, count in def_labels.items():
            machines = list(label_to_machines[label])
            for index in range(1, count + 1):
                extra_labels[f"kiso.{label}.{index}"] = 1
                label_to_machines[f"kiso.{label}.{index}"].add(machines[index - 1])

    return label_to_machines


def _check_software(softwares: Software, label_to_machines: dict[str, set]) -> None:
    """Check software configuration."""
    if softwares is None:
        return

    for software in fields(Software):
        config = getattr(softwares, software.name, None)
        if config is None:
            continue

        # Get the `name` of the software
        name = software.name

        # Locate the EntryPoint for the software `name` and load it
        installer = utils.get_software(name)

        # Instantiate the installer class. The installer class to use is defined in the
        # installer's `INSTALLER` attribute
        obj = installer.INSTALLER(
            config,  # Software configuration
            console=console,  # Console object to output experiment progress
            log=logging.getLogger(f"kiso.software.{name}"),  # Logger object to use
        )
        obj.check(label_to_machines)


def _check_deployed_software(
    deployments: Deployment, label_to_machines: dict[str, set]
) -> None:
    """Check software deployment configuration."""
    if deployments is None:
        return

    for deployment in fields(Deployment):
        config = getattr(deployments, deployment.name, None)
        if config is None:
            continue

        # Get the `name` of the software
        name = deployment.name

        # Locate the EntryPoint for the software `name` and load it
        installer = utils.get_deployment(name)

        # Instantiate the installer class. The installer class to use is defined in the
        # installer's `INSTALLER` attribute
        obj = installer.INSTALLER(
            config,  # Deployment configuration
            console=console,  # Console object to output experiment progress
            log=logging.getLogger(f"kiso.deployment.{name}"),  # Logger object to use
        )
        obj.check(label_to_machines)


def _check_experiments(
    experiment_config: Kiso, label_to_machines: dict[str, set]
) -> None:
    """Check software deployment configuration."""
    experiments = experiment_config.experiments
    if experiments is None:
        return

    variables = copy.deepcopy(experiment_config.variables, {})
    for index, experiment in enumerate(experiments):
        # Get the `kind` of experiment
        kind = experiment.kind

        # Locate the EntryPoint for the runner `kind` of experiment and load it
        runner_cfg = utils.get_runner(kind)

        # Instantiate the runner class. The runner class to use is defined in the
        # runner's `RUNNER` attribute
        runner = runner_cfg.RUNNER(
            experiment,
            index,
            console=console,  # Console object to output experiment progress
            log=logging.getLogger("kiso.experiment.pegasus"),  # Logger object to use
            variables=variables,  # Variables defined globally for the experiment
        )

        runner.check(experiment_config, label_to_machines)


@validate_config
@enostask(new=True, symlink=False)
def up(
    experiment_config: Kiso,
    force: bool = False,
    env: Environment = None,
    **kwargs: Any,  # noqa: ANN401
) -> None:
    """Create and set up resources for running an experiment.

    Initializes the experiment environment, sets up working directories, and prepares
    infrastructure by initializing sites, installing Docker, Apptainer, and HTCondor
    across specified labels.

    :param experiment_config: Configuration dictionary defining experiment parameters
    :type experiment_config: Kiso
    :param force: Force recreation of resources, defaults to False
    :type force: bool, optional
    :param env: Optional environment context for the experiment, defaults to None
    :type env: Environment, optional
    """
    console.rule(
        "[bold green]Create and set up resources for the experiments[/bold green]"
    )
    env["version"] = __version__
    env["wd"] = str(kwargs.get("wd", Path.cwd()))
    env["remote_wd"] = str(Path("~kiso") / Path(env["wd"]).name)

    experiment_config = _replace_labels_key_with_roles_key(experiment_config)

    _init_sites(experiment_config, env, force)
    _install_commons(env)
    _install_software(experiment_config, env)
    _install_deployed_software(experiment_config, env)


def _init_sites(
    experiment_config: Kiso, env: Environment, force: bool = False
) -> tuple[list[Provider], Roles, Networks]:
    """Initialize sites for an experiment.

    Initializes and configures sites from the experiment configuration using parallel
    processing.
    Performs the following key tasks:
    - Initializes providers for each site concurrently
    - Aggregates labels and networks from initialized sites
    - Extends labels with daemon-to-site mappings
    - Determines public IP requirements
    - Associates floating IPs and selects preferred IPs for nodes

    :param experiment_config: Configuration dictionary containing site definitions
    :type experiment_config: Kiso
    :param env: Environment context for the experiment
    :type env: Environment
    :param force: Force recreation of resources, defaults to False
    :type force: bool, optional
    :return: A tuple of providers, labels, and networks for the experiment
    :rtype: tuple[list[Provider], Roles, Networks]
    """
    log.debug("Initializing sites")

    providers = []
    labels = Roles()
    networks = Networks()

    with get_process_pool_executor() as executor:
        futures = [
            executor.submit(_init_site, site_index, site, force)
            for site_index, site in enumerate(experiment_config.sites)
        ]

        for future in futures:
            provider, _labels, _networks = future.result()

            providers.append(provider)
            labels.extend(_labels)
            networks.extend(_networks)

    daemon_to_site = _extend_labels(experiment_config, labels)
    is_public_ip_required = _is_public_ip_required(daemon_to_site)
    env["is_public_ip_required"] = is_public_ip_required

    for node in labels.all():
        # TODO(mayani): Remove the floating ip assignment code after it has been
        # implemented into the EnOSlib ChameleonEdge provider
        _associate_floating_ip(node, is_public_ip_required)

        ip = _get_best_ip(
            node,
            is_public_ip_required
            and (node.extra["is_submit"] or node.extra["is_central_manager"]),
        )
        node.extra["kiso_preferred_ip"] = ip

    providers = en.Providers(providers)
    env["providers"] = providers
    env["labels"] = labels
    env["networks"] = networks

    return providers, labels, networks


def _init_site(
    index: int, site: dict[Any, Any], force: bool = False
) -> tuple[Provider, Roles, Networks]:
    """Initialize a site for provisioning resources.

    Configures and initializes a site based on its provider type, handling specific
    requirements for different cloud providers like Chameleon. Performs the following
    key tasks:
    - Validates the site's provider type
    - Configures exposed ports for containers
    - Initializes provider resources and networks
    - Adds metadata to nodes about their provisioning context
    - Handles region-specific configurations

    :param index: The index of the site in the configuration
    :type index: int
    :param site: Site configuration dictionary
    :type site: dict[Any, Any]
    :param force: Force recreation of resources, defaults to False
    :type force: bool, optional
    :raises TypeError: If an invalid site provider type is specified
    :return: A tuple containing the provider, labels, and networks for the site
    :rtype: tuple[Provider, Roles, Networks]
    """
    kind = site["kind"]
    if kind not in PROVIDER_MAP:
        raise TypeError(f"Invalid site.type <{kind}> for site <{index}>")

    # There is no firewall on ChameleonEdge containers, but to reach HTCondor
    # daemons the port(s) still need to be exposed
    if kind == "chameleon-edge":
        for container in site["resources"]["machines"]:
            container = container["container"]
            exposed_ports = set(container.get("exposed_ports", []))
            exposed_ports.add(str(const.HTCONDOR_PORT))
            # exposed_ports.add(str(const.SSHD_PORT))
            container["exposed_ports"] = list(exposed_ports)

    conf = PROVIDER_MAP[kind][0](site)
    provider = PROVIDER_MAP[kind][1](conf)

    _labels, _networks = provider.init(force_deploy=force)
    _deduplicate_hosts(_labels)
    _labels[kind] = _labels.all()
    _networks[kind] = _networks.all()

    # For Chameleon site, the region name is important as each region will act like
    # a different site
    region_name = kind
    if kind.startswith("chameleon"):
        region_name = _get_region_name(site["rc_file"])
        _labels[region_name] = _labels.all()
        _networks[region_name] = _networks.all()

    # To each node we add a tag to identify what site/region it was provisioned on
    for node in _labels.all():
        # ChameleonDevice object does not have an attribute named extra
        if kind == "chameleon-edge":
            attr = "extra"
            setattr(node, attr, {})
        elif kind == "chameleon":
            # Used to copy this file to Chameleon VMs, so we cna use the Openstack
            # client to get a floating IP
            node.extra["rc_file"] = str(Path(conf.rc_file).resolve())

        node.extra["kind"] = kind
        node.extra["site"] = region_name

    if kind != "chameleon-edge":
        _labels = en.sync_info(_labels, _networks)
    else:
        # Because zunclient.v1.containers.Container is not pickleable
        provider.client.concrete_resources = []

    return provider, _labels, _networks


def _deduplicate_hosts(labels: Roles) -> None:
    """Deduplicate_hosts _summary_.

    _extended_summary_

    :param labels: _description_
    :type labels: Roles
    """
    dedup = {}
    for _, nodes in labels.items():
        update = set()
        for node in nodes:
            if node not in dedup:
                dedup[node] = node
            else:
                update.add(dedup[node])

        for node in update:
            nodes.remove(node)

        nodes.extend(update)


def _get_region_name(rc_file: str) -> str | None:
    """Extract the OpenStack region name from a given RC file.

    Parses the provided RC file to find the OS_REGION_NAME environment variable
    and returns its value. Raises a ValueError if the region name cannot be found.

    :param rc_file: Path to the OpenStack RC file containing environment variables
    :type rc_file: str
    :raises ValueError: If OS_REGION_NAME is not found in the RC file
    :return: The name of the OpenStack region
    :rtype: str | None
    """
    region_name = None
    with Path(rc_file).open() as env_file:
        for env_var in env_file:
            if "OS_REGION_NAME" in env_var:
                parts = env_var.split("=")
                region_name = parts[1].strip("\n\"'")
                break
        else:
            raise ValueError(f"Unable to get region name from the rc_file <{rc_file}>")

    return region_name


def _extend_labels(experiment_config: Kiso, labels: Roles) -> dict[str, set]:
    """Extend labels for an experiment configuration by adding unique labels and flags to nodes.

    Processes the given labels and experiment configuration to:
    - Create unique labels for each node based on their original label
    - Add flags to nodes indicating their HTCondor daemon types (central manager,
    submit, execute, personal)
    - Add flags for container technologies (Docker, Apptainer)
    - Track the sites where different HTCondor daemon types are located

    :param experiment_config: Configuration dictionary for the experiment
    :type experiment_config: Kiso
    :param labels: Dictionary of labels and their associated nodes
    :type labels: Roles
    :return: A mapping of HTCondor daemon types to their sites
    :rtype: dict[str, set]
    """  # noqa: E501
    extra: dict[str, set] = defaultdict(set)
    daemon_to_site = defaultdict(set)
    central_manager_labels, submit_labels, execute_labels, personal_labels = (
        _get_condor_daemon_labels(experiment_config)
    )

    for label, nodes in labels.items():
        is_central_manager = label in central_manager_labels
        is_submit = label in submit_labels
        is_execute = label in execute_labels
        is_personal = label in personal_labels
        for index, node in enumerate(nodes, 1):
            # EnOSlib resources.machines.number can be greater than 1, so we add the
            # host with a new unique label of the form kiso.<label>.<index>
            _label = f"kiso.{label}.{index}"
            extra[_label].add(node)

            # To each node we add flags to identify what HTCondor daemons will run on
            # the node
            node.extra["is_central_manager"] = (
                node.extra.get("is_central_manager", False) or is_central_manager
            )
            node.extra["is_submit"] = node.extra.get("is_submit", False) or is_submit
            node.extra["is_execute"] = node.extra.get("is_execute", False) or is_execute
            node.extra["is_personal"] = (
                node.extra.get("is_personal", False) or is_personal
            )

            site = [node.extra["site"]]
            if is_execute:
                daemon_to_site["execute"].update(site)
            if is_submit:
                daemon_to_site["submit"].update(site)
            if is_central_manager:
                daemon_to_site["central-manager"].update(site)

    labels.update(extra)

    return daemon_to_site


def _is_public_ip_required(daemon_to_site: dict[str, set]) -> bool:
    """Determine if a public IP address is required for the HTCondor cluster configuration.

    Checks if public IP addresses are needed based on the distribution of HTCondor
    daemons
    across different sites. A public IP is required under the following conditions:
    - Execute nodes are spread across multiple sites
    - Submit nodes are spread across multiple sites
    - Execute and submit nodes are on different sites
    - Submit nodes are on a different site from the central manager

    :param daemon_to_site: A dictionary mapping HTCondor daemon types to their sites
    :type daemon_to_site: dict[str, set]
    :return: True if a public IP is required, False otherwise
    :rtype: bool
    """  # noqa: E501
    is_public_ip_required = False
    central_manager = daemon_to_site["central-manager"]
    submit = daemon_to_site["submit"]
    execute = daemon_to_site["execute"]

    # A public IP is required if,
    # 1. If execute nodes are on multiple sites
    # 2. If submit nodes are on multiple sites
    # 3. If all execute nodes and submit nodes are on one site, but not the same one
    # 4. If submit nodes are on one site, but not the same one as the central manager
    if (central_manager or submit or execute) and (
        len(execute) > 1
        or len(submit) > 1
        or execute != submit
        or submit - central_manager
    ):
        is_public_ip_required = True

    return is_public_ip_required


def _associate_floating_ip(
    node: Host | ChameleonDevice, is_public_ip_required: bool = False
) -> None:
    """Associate a floating IP address to a node based on specific conditions.

    Determines whether to assign a floating IP to a node depending on its label and
    type. Supports different cloud providers and testbed types with specific IP
    assignment strategies.

    :param node: The node to potentially assign a floating IP to
    :type node: Host | ChameleonDevice
    :param is_public_ip_required: Flag indicating if a public IP is needed, defaults
    to False
    :type is_public_ip_required: bool, optional
    :raises NotImplementedError: If floating IP assignment is not supported for a
    specific testbed
    :raises KisoError: If assigning a public IP is unsupported
    :raises ValueError: If an unsupported site type is encountered
    """
    if is_public_ip_required and (
        node.extra["is_central_manager"] or node.extra["is_submit"]
    ):
        kind = node.extra["kind"]
        if kind == "chameleon":
            _associate_floating_ip_chameleon(node)
        elif kind == "chameleon-edge":
            _associate_floating_ip_edge(node)
        elif kind == "fabric":
            _associate_floating_ip_fabric(node)
        elif kind == "vagrant":
            raise KisoError("Assigning public IPs to Vagrant VMs is not supported")
        else:
            raise ValueError(f"Unknown site type {kind}", kind)


def _associate_floating_ip_chameleon(node: Host) -> None:
    """Associate a floating IP address with a Chameleon node.

    Retrieves or creates a floating IP for a Chameleon node using the OpenStack CLI.
    Handles cases where a node may already have a floating IP or requires a new one.
    Logs debug information during the IP association process.

    :param node: The Chameleon node to associate a floating IP with
    :type node: Host
    :raises ValueError: If the OpenStack CLI is not found or the server cannot be
    located
    """
    with source_credentials_from_rc_file(node.extra["rc_file"]):
        ip = None
        cli = shutil.which("openstack")
        if cli is None:
            raise ValueError("Could not locate the openstack client")

        try:
            cli = str(cli)

            log.debug("Get the Chameleon node's id")
            # Get the node information so we can extract the server id
            server = subprocess.run(  # noqa: S603
                [cli, "server", "show", node.alias, "-f", "json"],
                capture_output=True,
                check=True,
            )
            _server = json.loads(server.stdout.decode("utf-8"))

            log.debug("Check if the node already has a floating IP")
            # Determine if the node has a floating IP
            for _, addresses in _server["addresses"].items():
                for address in addresses:
                    if not ip_address(address).is_private:
                        ip = address

            if ip is None:
                log.debug("Check for any unused floating ips")
                # Check for any unused floating ip
                all_floating_ips = subprocess.run(  # noqa: S603
                    [cli, "floating", "ip", "list", "-f", "json"],
                    capture_output=True,
                    check=True,
                )
                _floating_ips = json.loads(all_floating_ips.stdout.decode("utf-8"))
                for floating_ip in _floating_ips:
                    # If an unused floating ip is available, use it
                    if (
                        floating_ip["Fixed IP Address"] is None
                        and floating_ip["Port"] is None
                    ):
                        _floating_ip = {"name": floating_ip["Floating IP Address"]}
                else:
                    log.debug("Request a new floating ip")
                    # Request a new floating ip
                    floating_ip = subprocess.run(  # noqa: S603
                        [cli, "floating", "ip", "create", "public", "-f", "json"],
                        capture_output=True,
                        check=True,
                    )
                    _floating_ip = json.loads(floating_ip.stdout.decode("utf-8"))

                log.debug("Associate the floating ip with the node")
                # Associate the floating ip with the node
                _associate_floating_ip = subprocess.run(  # noqa: S603
                    [
                        cli,
                        "server",
                        "add",
                        "floating",
                        "ip",
                        _server["id"],
                        _floating_ip["name"],
                    ],
                    capture_output=True,
                    check=True,
                )
                ip = _floating_ip["name"]
                log.debug(
                    "Floating IP <%s> associated with the node <%s>, status <%d>",
                    ip,
                    node.alias,
                    _associate_floating_ip,
                )

                floating_ips = node.extra.get("floating-ips", [])
                floating_ips.append(ip)
                node.extra["floating-ips"] = floating_ips
                log.debug("Floating IPs <%s>", floating_ips)
        except Exception as e:
            raise ValueError(f"Server <{node.alias}> not found") from e


def _associate_floating_ip_edge(node: ChameleonDevice) -> None:
    """Associate a floating IP address with a Chameleon Edge device.

    Attempts to retrieve an existing floating IP from /etc/floating-ip. If no IP is
    found, a new floating IP is associated with the device and saved to
    /etc/floating-ip.

    :param node: The Chameleon device to associate a floating IP with
    :type node: ChameleonDevice
    :raises: Potential exceptions from associate_floating_ip() method
    """
    # TODO(mayani): Handle error raised when user exceeds the floating IP usage
    # TODO(mayani): Handle error raised when IP can't be assigned as all are used up
    # Chameleon Edge API does not have a method to get the associated floating
    # IP, if one was already associated with the container
    status = edge._execute(node, "cat /etc/floating-ip")
    if status.rc == 0:
        log.debug("Floating IP already associated with the device")
        ip = status.stdout.strip()
    else:
        ip = node.associate_floating_ip()
        edge._execute(node, f"echo {ip} > /etc/floating-ip")

    log.debug("Floating IP associated with the device %s", ip)
    floating_ips = node.extra.get("floating-ips", [])
    floating_ips.append(ip)
    node.extra["floating-ips"] = floating_ips


def _associate_floating_ip_fabric(node: Host) -> None:
    """Associate a floating IP address with a Chameleon node.

    Retrieves or creates a floating IP for a Chameleon node using the OpenStack CLI.
    Handles cases where a node may already have a floating IP or requires a new one.
    Logs debug information during the IP association process.

    :param node: The Chameleon node to associate a floating IP with
    :type node: Host
    :raises ValueError: If the OpenStack CLI is not found or the server cannot be
    located
    """
    with source_fabric_credentials_from_rc_file(node.extra["rc_file"]):
        try:
            fablib = fablib_manager()
            fabric_slice = fablib.get_slice(name=node.extra["slice"])
            fabric_node = fabric_slice.get_node(name=node.extra["name"])
            stdout, _stderr = fabric_node.execute("cat /etc/floating-ip")
            if len(stdout.strip()):
                log.debug("Floating IP already associated with the device")
                ip = stdout.strip()
            else:
                submit = False
                network_name = f"{node.extra['name']}-public-network"
                nic_name = "public-nic"
                try:
                    component = fabric_node.get_component(name=nic_name)
                except Exception:
                    log.debug(
                        "Adding NIC_Basic component to FABRIC node <%s>",
                        fabric_node.get_management_ip(),
                    )
                    component = fabric_node.add_component(
                        model="NIC_Basic", name=nic_name
                    )
                    submit = True
                interface = component.get_interfaces()[0]

                if not fabric_slice.get_network(name=network_name):
                    log.debug(
                        "Adding IPv4Ext L3 Network to FABRIC node <%s>",
                        fabric_node.get_management_ip(),
                    )
                    fabric_slice.add_l3network(
                        name=network_name, interfaces=[interface], type="IPv4Ext"
                    )
                    submit = True
                if submit:
                    fabric_slice.submit()

                fabric_slice = fablib.get_slice(name=node.extra["slice"])
                network = fabric_slice.get_network(name=network_name)
                ip = network.get_available_ips()
                log.debug(
                    "Available IPs for FABRIC node <%s>, are <%s>",
                    fabric_node.get_management_ip(),
                    ip,
                )
                network.make_ip_publicly_routable(ipv4=[str(ip[0])])
                fabric_slice.submit()

                fabric_slice = fablib.get_slice(name=node.extra["slice"])
                network = fabric_slice.get_network(name=network_name)
                fabric_node = fabric_slice.get_node(name=node.extra["name"])
                interface = fabric_node.get_interface(network_name=network_name)
                ip = network.get_public_ips()[0]
                interface.ip_addr_add(addr=ip, subnet=network.get_subnet())
                # _stdout, _stderr = fabric_node.execute(
                #     f"sudo ip route add 0.0.0.0/0 via {network.get_gateway()}"
                # )
                fabric_node.execute(f"echo {ip} | sudo tee /etc/floating-ip")

            log.debug("Floating IP associated with the device %s", ip)
            floating_ips = node.extra.get("floating-ips", [])
            floating_ips.append(ip)
            node.extra["floating-ips"] = floating_ips

        except Exception as e:
            raise ValueError(
                f"Error occurred assigning public IP to FABRIC node <{node.alias}>"
            ) from e


def _get_best_ip(
    machine: Host | ChameleonDevice, is_public_ip_required: bool = False
) -> str:
    """Get the best IP address for a given machine.

    Selects an IP address based on priority, filtering out multicast, reserved,
    loopback, and link-local addresses. Supports both Host and ChameleonDevice
    types. Optionally enforces returning a public IP address.

    :param machine: The machine to get an IP address for
    :type machine: Host | ChameleonDevice
    :param is_public_ip_required: Whether a public IP is required, defaults to False
    :type is_public_ip_required: bool, optional
    :return: The selected IP address as a string
    :rtype: str
    :raises ValueError: If a public IP is required but not available
    """
    addresses = []
    # Vagrant Host
    # net_devices={
    #   NetDevice(
    #       name='eth1',
    #       addresses={
    #           IPAddress(
    #               network=None,
    #               ip=IPv6Interface('fe80::a00:27ff:fe6f:87e4/64')),
    #           IPAddress(
    #               network=<enoslib.infra.enos_vagrant.provider.VagrantNetwork ..,
    #               ip=IPv4Interface('172.16.255.243/24'))
    #   ..
    #   )
    # }
    #
    # Chameleon Host
    # net_devices={
    #   NetDevice(
    #     name='eno12419',
    #     addresses=set()),
    #   NetDevice(
    #     name='enp161s0f1',
    #     addresses=set()),
    #   NetDevice(
    #     name='enp161s0f0',
    #     addresses={
    #         IPAddress(
    #             network=<enoslib.infra.enos_openstack.objects.OSNetwork ..>,
    #             ip=IPv4Interface('10.52.3.205/22')
    #         ),
    #         IPAddress(
    #             network=None,
    #             ip=IPv6Interface('fe80::3680:dff:feed:50f4/64'))}
    #         ),
    #   NetDevice(
    #     name='eno8403',
    #     addresses=set()
    #   ),
    #   NetDevice(
    #     name='lo',
    #     addresses={
    #         IPAddress(network=None, ip=IPv4Interface('127.0.0.1/8')),
    #         IPAddress(network=None, ip=IPv6Interface('::1/128'))}),
    #   NetDevice(
    #     name='eno8303',
    #     addresses=set()
    #   ),
    #   NetDevice(
    #     name='eno12399',
    #     addresses=set()
    #   ),
    #   NetDevice(
    #     name='eno12429',
    #     addresses=set()
    #   ),
    #   NetDevice(
    #     name='eno12409',
    #     addresses=set()
    #   )
    # )
    # Chameleon Edge Host
    # Fabric Host
    if isinstance(machine, Host):
        for net_device in machine.net_devices:
            for address in net_device.addresses:
                if isinstance(address.network, DefaultNetwork) and isinstance(
                    address.ip, (IPv4Interface, IPv6Interface)
                ):
                    ip = address.ip.ip
                    if (
                        ip.is_multicast
                        or ip.is_reserved
                        or ip.is_loopback
                        or ip.is_link_local
                    ):
                        continue

                    priority = 1 if ip.is_private else 0
                    addresses.append((address.ip.ip, priority))
    else:
        address = ip_address(machine.address)
        priority = 1 if address.is_private else 0
        addresses.append((address, priority))

    for address in machine.extra.get("floating-ips", []):
        ip = ip_address(address)
        if ip.is_multicast or ip.is_reserved or ip.is_loopback or ip.is_link_local:
            continue

        priority = 1 if ip.is_private else 0
        addresses.append((ip, priority))

    addresses = sorted(addresses, key=lambda v: v[1])
    preferred_ip, priority = addresses[0]
    if is_public_ip_required is True and priority == 1:
        # TODO(mayani): We should not use gateway IP as it could be the same for
        # multiple VMs. Here we should just raise an error
        preferred_ip = machine.extra.get("gateway")
        if preferred_ip is None:
            raise ValueError(
                f"Machine <{machine.name}> does not have a public IP address"
            )

        preferred_ip = ip_address(preferred_ip)

    return str(preferred_ip)


def _get_condor_daemon_labels(
    experiment_config: Kiso,
) -> tuple[set[str], set[str], set[str], set[str]]:
    """Get labels for different HTCondor daemon types from an experiment configuration.

    Parses the HTCondor configuration to extract labels for central manager, submit,
    execute, and personal daemon types. Validates daemon types and raises an error for
    invalid types.

    :param experiment_config: Dictionary containing HTCondor cluster configuration
    :type experiment_config: Kiso
    :raises ValueError: If an invalid HTCondor daemon type is encountered
    :return: Tuple of label sets for central manager, submit, execute, and personal
    daemons
    :rtype: tuple[set[str], set[str], set[str], set[str]]
    """
    condor_cluster = (
        experiment_config.deployment and experiment_config.deployment.htcondor
    )
    central_manager_labels = set()
    submit_labels = set()
    execute_labels = set()
    personal_labels = set()

    if condor_cluster:
        for config in condor_cluster:
            if config.kind[0] == "c":  # central-manager
                central_manager_labels.update(config.labels)
            elif config.kind[0] == "s":  # submit
                submit_labels.update(config.labels)
            elif config.kind[0] == "e":  # execute
                execute_labels.update(config.labels)
            elif config.kind[0] == "p":  # personal
                personal_labels.update(config.labels)
            else:
                raise ValueError(
                    f"Invalid HTCondor daemon <{config.kind}> in configuration"
                )

    return central_manager_labels, submit_labels, execute_labels, personal_labels


def _install_commons(env: Environment) -> None:
    """Install components needed to run a Kiso experiment.

    1. Disable SELinux on EL-based systems.
    2. Disable Firewall.
    3. Install dependencies, like sudo, curl, etc.
    4. Create a kiso group and a user.
    5. Allow passwordless sudo for kiso.
    6. Copy .ssh dir to ~kiso/.ssh dir.

    :param env: Environment context for the installation
    :type env: Environment
    """
    log.debug("Install Commons")
    console.rule("[bold green]Installing Commons[/bold green]")

    labels = env["labels"]
    # Special case here. Do not pass (labels, labels) to split_labels. Since the Roles
    # object is like a dictionary, so labels - labels["<key>"] and
    # labels & labels["<key>"] doesn't work.
    vms, containers = utils.split_labels(labels.all(), labels)
    results = []

    if vms:
        results.extend(
            utils.run_ansible(
                [Path(__file__).parent / "commons/main.yml"],
                roles=vms,
            )
        )

    if containers:
        for container in containers:
            results.append(
                utils.run_script(
                    container,
                    Path(__file__).parent / "commons/init.sh",
                    "--no-dry-run",
                )
            )

    display.commons(console, results)


def _install_software(experiment_config: Kiso, env: Environment) -> None:
    """Install software on specified labels in an experiment configuration."""
    softwares = experiment_config.software
    if softwares is None:
        return

    for software in fields(Software):
        config = getattr(softwares, software.name, None)
        if config is None:
            continue

        # Get the `name` of the software
        name = software.name

        # Locate the EntryPoint for the software `name` and load it
        installer = utils.get_software(name)

        # Instantiate the installer class. The installer class to use is defined in the
        # installer's `INSTALLER` attribute
        obj = installer.INSTALLER(
            config,  # Software configuration
            console=console,  # Console object to output experiment progress
            log=logging.getLogger(f"kiso.software.{name}"),  # Logger object to use
        )
        obj(env)


def _install_deployed_software(experiment_config: Kiso, env: Environment) -> None:
    """Install software for deployments on specified labels in an experiment configuration."""  # noqa: E501
    deployments = experiment_config.deployment
    if deployments is None:
        return

    for deployment in fields(Deployment):
        config = getattr(deployments, deployment.name, None)
        if config is None:
            continue

        # Get the `name` of the deployment
        name = deployment.name

        # Locate the EntryPoint for the software `name` and load it
        installer = utils.get_deployment(name)

        # Instantiate the installer class. The installer class to use is defined in the
        # installer's `INSTALLER` attribute
        obj = installer.INSTALLER(
            config,  # Deployment configuration
            console=console,  # Console object to output experiment progress
            log=logging.getLogger(f"kiso.deployment.{name}"),  # Logger object to use
        )
        obj(env)


@validate_config
@enostask()
def run(
    experiment_config: Kiso,
    force: bool = False,
    env: Environment = None,
    **kwargs: Any,  # noqa: ANN401
) -> None:
    """Run the defined experiments.

    Executes a series of experiments by performing the following steps:
    - Copies experiment directory to remote labels
    - Executes experiment

    :param experiment_config: Configuration dictionary containing experiment details
    :type experiment_config: Kiso
    :param force: Force rerunning of experiments, defaults to False
    :type force: bool, optional
    :param env: Environment configuration containing providers, labels, and networks
    :type env: Environment, optional
    :param kwargs: Additional keyword arguments
    :type kwargs: dict
    """
    log.debug("Run Kiso experiments")
    console.rule("[bold green]Run experiments[/bold green]")

    experiments = experiment_config.experiments
    variables = copy.deepcopy(experiment_config.variables, {})
    env.setdefault("experiments", {})
    if force is True:
        env["experiments"] = {}

    _copy_experiment_dir(env)
    for experiment_index, experiment in enumerate(experiments):
        env["experiments"].setdefault(experiment_index, {})
        _run_experiments(experiment_index, experiment, variables, env)


def _copy_experiment_dir(env: Environment) -> None:
    """Copy experiment directory to remote labels.

    Copies the experiment directory from the local working directory to the remote
    working directory for specified submit node labels. Supports copying to both virtual
    machines and containers.

    :param env: Environment configuration containing labels and working directory
    information
    :type env: Environment
    :raises Exception: If directory copy fails for any label
    """
    log.debug("Copy experiment directory to remote nodes")
    console.print("Copying experiment directory to remote nodes")

    labels = env["labels"]
    # Special case here. Do not pass (labels, labels) to split_labels. Since the Roles
    # object is like a dictionary, so labels - labels["<key>"] and
    # labels & labels["<key>"] doesn't work.
    vms, containers = utils.split_labels(labels.all(), labels)

    try:
        kiso_state = env["experiments"]
        if kiso_state.get("copy-experiment-directory") == const.STATUS_OK:
            return
        kiso_state["copy-experiment-directory"] = const.STATUS_STARTED
        src = Path(env["wd"])
        dst = Path(env["remote_wd"]).parent
        if vms:
            with utils.actions(roles=vms, strategy="free") as p:
                p.shell(
                    "rsync -auzv -e 'ssh {{ansible_ssh_common_args}} "
                    "-p {{ansible_port}} -i {{ansible_ssh_private_key_file}}' "
                    f"{src} kiso@{{{{ansible_host}}}}:{dst}",
                    delegate_to="localhost",
                )
        if containers:
            for container in containers:
                edge.upload(container, src, dst, user=const.KISO_USER)
    except Exception:
        kiso_state["copy-experiment-directory"] = const.STATUS_FAILED
        raise
    else:
        kiso_state["copy-experiment-directory"] = const.STATUS_OK


def _run_experiments(
    index: int, experiment: ExperimentTypes, variables: dict, env: Environment
) -> None:
    """Run multiple workflow instances for a specific experiment.

    Generates and executes workflows for each instance of an experiment.

    :param index: The overall experiment index
    :type index: int
    :param experiment: Configuration dictionary for the experiment
    :type experiment: dict
    :param env: Environment context containing workflow and execution details
    :type env: Environment
    """
    # Get the `kind` of experiment
    kind = experiment.kind

    # Locate the EntryPoint for the runner `kind` of experiment and load it
    runner_cfg = utils.get_runner(kind)

    # Instantiate the runner class. The runner class to use is defined in the
    # runner's `RUNNER` attribute
    runner = runner_cfg.RUNNER(
        experiment,
        index,
        console=console,  # Console object to output experiment progress
        log=logging.getLogger("kiso.experiment.pegasus"),  # Logger object to use
        variables=variables,  # Variables defined globally for the experiment
    )

    # Run the experiment
    runner(
        env["wd"],  # Local experiment working directory
        env["remote_wd"],  # Remote experiment working directory
        env["resultdir"],  # Local results directory
        env["labels"],  # Provisioned resources
        env["experiments"][index],  # Store to maintain the state of the experiment
    )


def _to_snake_case(key: str) -> str:
    return "-".join(key.split("_"))


@validate_config
@enostask()
def down(experiment_config: Kiso, env: Environment = None, **kwargs: dict) -> None:
    """Destroy the resources provisioned for the experiments.

    This function is responsible for tearing down and cleaning up resources
    associated with an experiment configuration using the specified providers.

    :param experiment_config: Configuration dictionary for the experiment
    :type experiment_config: Kiso
    :param env: Environment object containing provider information
    :type env: Environment, optional
    :param kwargs: Additional keyword arguments
    :type kwargs: dict
    """
    log.debug("Destroy the resources provisioned for the experiments")
    console.rule(
        "[bold green]Destroy resources created for the experiments[/bold green]"
    )

    if "providers" not in env:
        log.debug("No providers found, skipping")
        console.rule(
            "No providers found. Either resources were not provisioned or the output "
            "directory was removed"
        )
        return

    providers = env["providers"]
    providers.destroy()
