from __future__ import annotations

import hashlib
from typing import TYPE_CHECKING

import pytest

from dissect.util.compression import lznt1

if TYPE_CHECKING:
    from pytest_benchmark.fixture import BenchmarkFixture


@pytest.mark.parametrize(
    ("data", "digest"),
    [
        pytest.param(
            "38b08846232000204720410010a24701a045204400084501507900c045200524"
            "138805b4024a44ef0358028c091601484500be009e000401189000",
            "4c306957685911d930cbc28c12baeefa87616d850b7b569fc629a4d551f5d28f",
            id="basic",
        ),
        pytest.param(
            "36b3004c6f72656d2069700073756d20646f6c6f00722073697420616d006574"
            "20636f6e736500637465747572206100646970697363696e006720656c69742e"
            "2000517569737175652000666175636962757300206578207361706900656e20"
            "7669746165002070656c6c656e740a65024a7300c6706c616308657261007649"
            "6e206940642063757273006e6d8069207072657469007f36740032001164005a"
            "017c7661006c6c69732e205465046d7000166c656f206500752061656e65616e"
            "002073656420646961006d2075726e6120740100226f722e2050756c2076696e"
            "617200837661126d0034667200b2696c6c0c6120007d000f6e6563200300db00"
            "09626962656e64810075656765737461803310496163750038206d6100737361"
            "206e69736c8100056c6573756164021f40696e69612069006967006572206e75"
            "6e632000706f73756572652e202055742068802672651672009e007670801076"
            "656c0820636c802520617074018084207461636974694020736f63696f808a2e"
            "902041642080a56f72805fa46f7200a66e74809a72017d84756200356e6f7374"
            "800d00696e636570746f73002068696d656e6165e06f732e0a0a7fdbbf6dbf6d"
            "fbbf6dbf6d758176c17b80cbbf6d7fdbb3c0e47fdb7320bf6dbf6d20bf6dce73"
            "bf6dbf6dbf6d7175bf6dbf6def9fa49fa49fa49fa46cc0979fa49fa43c76699f"
            "a49fa49fa49fa47373f99fa420749fa49fa47fdb7fdb7fdbff7fdb7fdb7fdb7f"
            "db7fdb7fdbbf6de0757fbf6dbf6dbf6dbf6dbf6dbf6d2f8965fe742f896f1b2f"
            "892f892f89b0352f89f7df362f892f89702f89d0052f892f89ff2f892f89118d"
            "2f892f892f89df362f89ffdf362f89df369fa49fa49fa49fa49fa4ff9fa49fa4"
            "2f899fa49fa49fa46f1b6f1bf9bf6d6d20df369fa49fa49fa49fa4fb9fa49fa4"
            "69f0519fa42f899fa40fc03f0fc00fc0b0150fc00fc00fc02e207f0fc00fc0e0"
            "940fc00fc00fc00fc067ff0fc0001d0fc00fc0bf6d0fc00fc0f097ff2f89c0bd"
            "0fc00fc081039fa40fc00fc0ff7fdb4f524f524f524f524f52e0134f52df4f52"
            "4f524f524f527fdb73df36df36cfb01a4f524f527fdb6e647fdb7fdbfb7fdb7f"
            "db6edf36df36df36df36df363fdf369fa490c8df36df36eaf635b30073206475"
            "69732063006f6e76616c6c6973002e2054656d70757300206c656f2065752000"
            "61656e65616e2073006564206469616d204075726e61207400886f00722e2050"
            "756c7669006e61722076697661026d00686672696e6769806c6c61206c616300"
            "1e806e6563206d6574001200626962656e64756d802065676573746100ce0649"
            "004200e0206d6173734061206e69736c000a6c206573756164023e696e406961"
            "20696e7400307200206e756e6320706f0073756572652e205508742068004d72"
            "6572692274009a6d70650082656c0820636c004b2061707400656e7420746163"
            "6900746920736f63696f007371752e20416420206c69746f7200bf6f729c7175"
            "01230138007d75620035106e6f7374800d696e63006570746f73206869006d65"
            "6e61656f732e000a0a4c6f72656d20086970730060646f6c6f68722073803c61"
            "006d819c7308656374007272206164206970697363808120653100352e205180"
            "ad8034206620617563696200836578802073617069656e009900746165207065"
            "6c6c9500446500506581642070009c00657261742e20496e0020696420637572"
            "7301801b6d692070726574fa69803f74001980087fdbbf6dbf6d9f008ebf6dbf"
            "6dbf6d7fdb73207fdbfdbf6d73407fdf36bf6dbf6dbf6dbf6df9bf6d6163bf6d"
            "bf6d9fa49fa4a00c7f9fa49fa49fa49fa49fa49fa49fa467799fa476699fa49f"
            "a47fdb7fdb737e207fdb7fdb7fdb7fdbe1d67fdb74fd40cb7281d37fdbbf6dbf"
            "6dbf6dbf6de7bf6dbf6ddf367320bf6dbf6d2f89ffd0722f892f896f1b2f892f"
            "892f892f89ff2f892f892f892f892f896f1b2f892f89f92f8974756f1b6f1b6f"
            "1b6f1b6f1bff6f1b9fa49fa49fa49fa49fa49fa49fa4ff9fa49fa49fa4df36bf"
            "6d9fa49fa4df36ff9fa49fa49fa49fa44f529fa44f52507aff2f899fa49fa40f"
            "c00fc00fc00fc0e232ff106d2f894f520fc00fc00fc00fc00fc0bf0fc00fc04f"
            "52bf6d0fc00fc065df36ff40bc0fc02f890fc02f894f527fdba034ff10177fdb"
            "7fdb7fdb7fdb7fdb7fdb7fdbcfd0cb7fdb2f897fdb73737fdb7fdbcf7fdb0fc0"
            "7fdb9fa465743156df365fdf36df36df36df367fdb6d40f36cde6cf019bf6dbf"
            "6dbf6d70eff6b0040f2f89eff6eff67adb1fb3007375616461206c610063696e"
            "696120696e007465676572206e75006e6320706f7375650072652e2055742068"
            "00656e647265726974202073656d7000847665006c20636c6173732000617074"
            "656e742074010084746920736f6369006f7371752e20416400206c69746f7261"
            "20190008717501460170636f6e84756200d46e6f7374003600696e636570746f"
            "73002068696d656e6165006f732e0a0a4c6f7200656d20697073756d0020646f"
            "6c6f722073410079616d657420003e7300656374657475722040616469706973"
            "00b967042065006a2e2051756901007a65206661756369006275732065782073"
            "00617069656e207669007461652070656c6c4d008865022500c9207000f96500"
            "7261742e20496e208069642063757273801b006d692070726574696d803f7400"
            "19800864002d013e7600616c6c69732e2054030083000b6c656f206575802061"
            "656e65616e808d0064206469616d20750c726e007b00116f722e200050756c76"
            "696e617211804176616d001a667269606e67696c6c82bb80076e18656320806d"
            "80046269624700b4803a80c2737461803349486163750038206d00b56120206e"
            "69736c00056c65fb7fdbbf6d6ebf6dbf6dbf6dbf6d7fdbff7fdbbf6dbf6dbf6d"
            "bf6dbf6dbf6dbf6df9bf6d7373bf6d9fa49fa49fa49fa46f9fa49fa49fa49fa4"
            "209fa49fa476fe699fa49fa49fa47fdb7fdb40337fdb3f7fdb7fdbe0d37fdb40"
            "677fdb7320df7fdbbf6d6f1bbf6d6f1b63a051bf6d7e20bf6dbf6d2f894f522f"
            "892f8969bf2f89101b2f892f896f1b2f89202f89ff2f892f892f8960972f892f"
            "892f892f89be202f892f892f892f892f89652f89ff4f529fa49fa49fa49fa49f"
            "a4df369fa4fb9fa49fa4746f1bbf6d9fa49fa4df36e79fa4df364f526d708032"
            "6f1b6f1bff6f1b6f1b6f1b6f1bbf6d80790fc00fc0ff0fc04f520fc00fc00fc0"
            "0fc00fc0df36ffdf36df36df36df36bf6d0fc00fc06f1b6c73650fc00fc0200f"
            "c00fc067f76f1b6f1b6f1b6edf36df36df36df36ffdf36df36df36df36df367f"
            "dbbf6d7fdbff7fdb7fdbbf6d2f89806bdf366f1b7fdbfe736f1b7fdbbf6dbf6d"
            "bf6dbf6dbf6d3fbf6dbf6deff6f161bf6db86d0bb300732e0a0a4c6f7265006d"
            "20697073756d2000646f6c6f72207369007420616d65742063006f6e73656374"
            "6574007572206164697069007363696e6720656c0069742e2051756973007175"
            "6520666175630069627573206578200073617069656e20760069746165207065"
            "6ca06c656e7465024a7300c680706c6163657261007600496e20696420637504"
            "727300376d6920707268657469007f74003200116403005a017c76616c6c6973"
            "402e2054656d7000166c00656f206575206165006e65616e2073656400206469"
            "616d207572106e61207400226f722e002050756c76696e612272008376616d00"
            "346672c100b2696c6c6120007d000f306e65632000db000962691062656e6480"
            "3a65676508737461803349616375010038206d6173736120106e69736c00056c"
            "657308756164021f696e69610420690069676572206e00756e6320706f737500"
            "6572652e2055742062688026726572009e00767081801076656c20636c802510"
            "206170748084207461006369746920736f6304696f808a2e204164204980a56f"
            "72805f6f7200a66e4a74809a72017d756200356e086f7374800d696e63650070"
            "746f732068696de0656e61656f7fdb804fbf6dffbf6dbf6dbf6dbf6d7fdb7fdb"
            "407dbf6dfbbf6dbf6d20bf6dbf6dbf6d2069bf6d7fbf6dbf6d81759fa49fa4c0"
            "48409a6dfe209fa49fa49fa49fa49fa49fa49fa4efe0b49fa49fa494a4001f00"
            "3f043f04ff3f043f043f043f043f043f041f021f01ff1f011f011f011f011f01"
            "1f011f011f01ff1f011f011f011f011f011f011f011f01ff1f011f011f011f01"
            "1f011f011f011f01ff1f011f011f011f011f011f011f011f01ff1f011f011f01"
            "1f011f011f011f011f01ff1f011f011f011f011f011f011f011f01ff1f011f01"
            "1f011f011f011f011f011f01ff1f011f011f011f011f011f011f011f01ff1f01"
            "1f011f011f011f011f011f011f01ff1f011f011f011f011f011f011f011f01ff"
            "1f011f011f011f011f011f011f011f01ff1f011f011f011f011f011f011f011f"
            "01ff1f011f011f011f011f011f011f011f017f1f011f011f011f011f011f011b"
            "0103b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03"
            "b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03b002"
            "00fc0f03b00200fc0f0000000000000000000000000000000000000000000000",
            "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529",
            id="large",
        ),
    ],
)
def test_lznt1_decompress(data: str, digest: str) -> None:
    assert hashlib.sha256(lznt1.decompress(bytes.fromhex(data)).rstrip(b"\x00")).hexdigest() == digest


@pytest.mark.benchmark
def test_benchmark_lznt1_decompress(benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex(
        "38b08846232000204720410010a24701a045204400084501507900c045200524"
        "138805b4024a44ef0358028c091601484500be009e000401189000"
    )
    assert benchmark(lznt1.decompress, buf) == (
        b"F# F# G A A G F# E D D E F# F# E E F# F# G A A "
        b"G F# E D D E F# E D D E E F# D E F# G F# D E F# "
        b"G F# E D E A F# F# G A A G F# E D D E F# E D D\x00"
    )


@pytest.mark.benchmark
def test_benchmark_large_lznt1_decompress(benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex(
        "36b3004c6f72656d2069700073756d20646f6c6f00722073697420616d006574"
        "20636f6e736500637465747572206100646970697363696e006720656c69742e"
        "2000517569737175652000666175636962757300206578207361706900656e20"
        "7669746165002070656c6c656e740a65024a7300c6706c616308657261007649"
        "6e206940642063757273006e6d8069207072657469007f36740032001164005a"
        "017c7661006c6c69732e205465046d7000166c656f206500752061656e65616e"
        "002073656420646961006d2075726e6120740100226f722e2050756c2076696e"
        "617200837661126d0034667200b2696c6c0c6120007d000f6e6563200300db00"
        "09626962656e64810075656765737461803310496163750038206d6100737361"
        "206e69736c8100056c6573756164021f40696e69612069006967006572206e75"
        "6e632000706f73756572652e202055742068802672651672009e007670801076"
        "656c0820636c802520617074018084207461636974694020736f63696f808a2e"
        "902041642080a56f72805fa46f7200a66e74809a72017d84756200356e6f7374"
        "800d00696e636570746f73002068696d656e6165e06f732e0a0a7fdbbf6dbf6d"
        "fbbf6dbf6d758176c17b80cbbf6d7fdbb3c0e47fdb7320bf6dbf6d20bf6dce73"
        "bf6dbf6dbf6d7175bf6dbf6def9fa49fa49fa49fa46cc0979fa49fa43c76699f"
        "a49fa49fa49fa47373f99fa420749fa49fa47fdb7fdb7fdbff7fdb7fdb7fdb7f"
        "db7fdb7fdbbf6de0757fbf6dbf6dbf6dbf6dbf6dbf6d2f8965fe742f896f1b2f"
        "892f892f89b0352f89f7df362f892f89702f89d0052f892f89ff2f892f89118d"
        "2f892f892f89df362f89ffdf362f89df369fa49fa49fa49fa49fa4ff9fa49fa4"
        "2f899fa49fa49fa46f1b6f1bf9bf6d6d20df369fa49fa49fa49fa4fb9fa49fa4"
        "69f0519fa42f899fa40fc03f0fc00fc0b0150fc00fc00fc02e207f0fc00fc0e0"
        "940fc00fc00fc00fc067ff0fc0001d0fc00fc0bf6d0fc00fc0f097ff2f89c0bd"
        "0fc00fc081039fa40fc00fc0ff7fdb4f524f524f524f524f52e0134f52df4f52"
        "4f524f524f527fdb73df36df36cfb01a4f524f527fdb6e647fdb7fdbfb7fdb7f"
        "db6edf36df36df36df36df363fdf369fa490c8df36df36eaf635b30073206475"
        "69732063006f6e76616c6c6973002e2054656d70757300206c656f2065752000"
        "61656e65616e2073006564206469616d204075726e61207400886f00722e2050"
        "756c7669006e61722076697661026d00686672696e6769806c6c61206c616300"
        "1e806e6563206d6574001200626962656e64756d802065676573746100ce0649"
        "004200e0206d6173734061206e69736c000a6c206573756164023e696e406961"
        "20696e7400307200206e756e6320706f0073756572652e205508742068004d72"
        "6572692274009a6d70650082656c0820636c004b2061707400656e7420746163"
        "6900746920736f63696f007371752e20416420206c69746f7200bf6f729c7175"
        "01230138007d75620035106e6f7374800d696e63006570746f73206869006d65"
        "6e61656f732e000a0a4c6f72656d20086970730060646f6c6f68722073803c61"
        "006d819c7308656374007272206164206970697363808120653100352e205180"
        "ad8034206620617563696200836578802073617069656e009900746165207065"
        "6c6c9500446500506581642070009c00657261742e20496e0020696420637572"
        "7301801b6d692070726574fa69803f74001980087fdbbf6dbf6d9f008ebf6dbf"
        "6dbf6d7fdb73207fdbfdbf6d73407fdf36bf6dbf6dbf6dbf6df9bf6d6163bf6d"
        "bf6d9fa49fa4a00c7f9fa49fa49fa49fa49fa49fa49fa467799fa476699fa49f"
        "a47fdb7fdb737e207fdb7fdb7fdb7fdbe1d67fdb74fd40cb7281d37fdbbf6dbf"
        "6dbf6dbf6de7bf6dbf6ddf367320bf6dbf6d2f89ffd0722f892f896f1b2f892f"
        "892f892f89ff2f892f892f892f892f896f1b2f892f89f92f8974756f1b6f1b6f"
        "1b6f1b6f1bff6f1b9fa49fa49fa49fa49fa49fa49fa4ff9fa49fa49fa4df36bf"
        "6d9fa49fa4df36ff9fa49fa49fa49fa44f529fa44f52507aff2f899fa49fa40f"
        "c00fc00fc00fc0e232ff106d2f894f520fc00fc00fc00fc00fc0bf0fc00fc04f"
        "52bf6d0fc00fc065df36ff40bc0fc02f890fc02f894f527fdba034ff10177fdb"
        "7fdb7fdb7fdb7fdb7fdb7fdbcfd0cb7fdb2f897fdb73737fdb7fdbcf7fdb0fc0"
        "7fdb9fa465743156df365fdf36df36df36df367fdb6d40f36cde6cf019bf6dbf"
        "6dbf6d70eff6b0040f2f89eff6eff67adb1fb3007375616461206c610063696e"
        "696120696e007465676572206e75006e6320706f7375650072652e2055742068"
        "00656e647265726974202073656d7000847665006c20636c6173732000617074"
        "656e742074010084746920736f6369006f7371752e20416400206c69746f7261"
        "20190008717501460170636f6e84756200d46e6f7374003600696e636570746f"
        "73002068696d656e6165006f732e0a0a4c6f7200656d20697073756d0020646f"
        "6c6f722073410079616d657420003e7300656374657475722040616469706973"
        "00b967042065006a2e2051756901007a65206661756369006275732065782073"
        "00617069656e207669007461652070656c6c4d008865022500c9207000f96500"
        "7261742e20496e208069642063757273801b006d692070726574696d803f7400"
        "19800864002d013e7600616c6c69732e2054030083000b6c656f206575802061"
        "656e65616e808d0064206469616d20750c726e007b00116f722e200050756c76"
        "696e617211804176616d001a667269606e67696c6c82bb80076e18656320806d"
        "80046269624700b4803a80c2737461803349486163750038206d00b56120206e"
        "69736c00056c65fb7fdbbf6d6ebf6dbf6dbf6dbf6d7fdbff7fdbbf6dbf6dbf6d"
        "bf6dbf6dbf6dbf6df9bf6d7373bf6d9fa49fa49fa49fa46f9fa49fa49fa49fa4"
        "209fa49fa476fe699fa49fa49fa47fdb7fdb40337fdb3f7fdb7fdbe0d37fdb40"
        "677fdb7320df7fdbbf6d6f1bbf6d6f1b63a051bf6d7e20bf6dbf6d2f894f522f"
        "892f8969bf2f89101b2f892f896f1b2f89202f89ff2f892f892f8960972f892f"
        "892f892f89be202f892f892f892f892f89652f89ff4f529fa49fa49fa49fa49f"
        "a4df369fa4fb9fa49fa4746f1bbf6d9fa49fa4df36e79fa4df364f526d708032"
        "6f1b6f1bff6f1b6f1b6f1b6f1bbf6d80790fc00fc0ff0fc04f520fc00fc00fc0"
        "0fc00fc0df36ffdf36df36df36df36bf6d0fc00fc06f1b6c73650fc00fc0200f"
        "c00fc067f76f1b6f1b6f1b6edf36df36df36df36ffdf36df36df36df36df367f"
        "dbbf6d7fdbff7fdb7fdbbf6d2f89806bdf366f1b7fdbfe736f1b7fdbbf6dbf6d"
        "bf6dbf6dbf6d3fbf6dbf6deff6f161bf6db86d0bb300732e0a0a4c6f7265006d"
        "20697073756d2000646f6c6f72207369007420616d65742063006f6e73656374"
        "6574007572206164697069007363696e6720656c0069742e2051756973007175"
        "6520666175630069627573206578200073617069656e20760069746165207065"
        "6ca06c656e7465024a7300c680706c6163657261007600496e20696420637504"
        "727300376d6920707268657469007f74003200116403005a017c76616c6c6973"
        "402e2054656d7000166c00656f206575206165006e65616e2073656400206469"
        "616d207572106e61207400226f722e002050756c76696e612272008376616d00"
        "346672c100b2696c6c6120007d000f306e65632000db000962691062656e6480"
        "3a65676508737461803349616375010038206d6173736120106e69736c00056c"
        "657308756164021f696e69610420690069676572206e00756e6320706f737500"
        "6572652e2055742062688026726572009e00767081801076656c20636c802510"
        "206170748084207461006369746920736f6304696f808a2e204164204980a56f"
        "72805f6f7200a66e4a74809a72017d756200356e086f7374800d696e63650070"
        "746f732068696de0656e61656f7fdb804fbf6dffbf6dbf6dbf6dbf6d7fdb7fdb"
        "407dbf6dfbbf6dbf6d20bf6dbf6dbf6d2069bf6d7fbf6dbf6d81759fa49fa4c0"
        "48409a6dfe209fa49fa49fa49fa49fa49fa49fa4efe0b49fa49fa494a4001f00"
        "3f043f04ff3f043f043f043f043f043f041f021f01ff1f011f011f011f011f01"
        "1f011f011f01ff1f011f011f011f011f011f011f011f01ff1f011f011f011f01"
        "1f011f011f011f01ff1f011f011f011f011f011f011f011f01ff1f011f011f01"
        "1f011f011f011f011f01ff1f011f011f011f011f011f011f011f01ff1f011f01"
        "1f011f011f011f011f011f01ff1f011f011f011f011f011f011f011f01ff1f01"
        "1f011f011f011f011f011f011f01ff1f011f011f011f011f011f011f011f01ff"
        "1f011f011f011f011f011f011f011f01ff1f011f011f011f011f011f011f011f"
        "01ff1f011f011f011f011f011f011f011f017f1f011f011f011f011f011f011b"
        "0103b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03"
        "b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03b00200fc0f03b002"
        "00fc0f03b00200fc0f0000000000000000000000000000000000000000000000"
    )
    assert hashlib.sha256(benchmark(lznt1.decompress, buf).rstrip(b"\x00")).hexdigest() == (
        "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529"
    )
