from __future__ import annotations

import hashlib
from typing import TYPE_CHECKING

import pytest

from dissect.util.compression import lzxpress_huffman

if TYPE_CHECKING:
    from pytest_benchmark.fixture import BenchmarkFixture


@pytest.mark.parametrize(
    ("data", "digest"),
    [
        pytest.param(
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000030230000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0200000000000000000000000000002000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "a8dc0000ff2601",
            "d9f5aeb06abebb3be3f38adec9a2e3b94228d52193be923eb4e24c9b56ee0930",
            id="basic",
        ),
        pytest.param(
            "0000000000080000000000000000000003000000000000060000000000000000"
            "9000000080000900990088000000000040573578480055547545440708000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "0800000000000000000000000000000000000000000000000800000000000000"
            "0700000000000000870800000000000086000000000000000600000000000000"
            "0800000000000080000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000"
            "e3fee146f3725c711a5bb00ee612300569ce27062a4157b9b7a26c30b0b1caff"
            "49bfd1238b265cca601f95e93a2c21ac361968b1463fe47d92cd49879edf162c"
            "93a2efa7a88b8cce002fbc2315fb6f77deef6ba9ec2b8d3cd9f370ec840c512c"
            "9a83aa8a4e5c04991e77b0354dffabbab826d5092e2f353d35c390b5f76f6360"
            "c54185cf79c9ac161edce165a33729c5573647170c746faf628795f2fafaa0b2"
            "295e3af32d0d9c419d3c871d0f9cce5dd2d106c793ef33c04b61606ec74c50e0"
            "a28487c217c5d20f1f768bca632c405cbac6c94f733e344cdef2c3b66cc7a156"
            "61ccaac7b2dc3e1c1edeb61fe0feff25350000",
            "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529",
            id="large",
        ),
    ],
)
def test_lzxpress_huffman_decompress(data: str, digest: str) -> None:
    assert hashlib.sha256(lzxpress_huffman.decompress(bytes.fromhex(data))).hexdigest() == digest


@pytest.mark.benchmark
def test_benchmark_lzxpress_huffman_decompress(benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex(
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000030230000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0200000000000000000000000000002000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "a8dc0000ff2601"
    )
    assert benchmark(lzxpress_huffman.decompress, buf) == b"abc" * 100


@pytest.mark.benchmark
def test_benchmark_large_lzxpress_huffman_decompress(benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex(
        "0000000000080000000000000000000003000000000000060000000000000000"
        "9000000080000900990088000000000040573578480055547545440708000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0800000000000000000000000000000000000000000000000800000000000000"
        "0700000000000000870800000000000086000000000000000600000000000000"
        "0800000000000080000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "e3fee146f3725c711a5bb00ee612300569ce27062a4157b9b7a26c30b0b1caff"
        "49bfd1238b265cca601f95e93a2c21ac361968b1463fe47d92cd49879edf162c"
        "93a2efa7a88b8cce002fbc2315fb6f77deef6ba9ec2b8d3cd9f370ec840c512c"
        "9a83aa8a4e5c04991e77b0354dffabbab826d5092e2f353d35c390b5f76f6360"
        "c54185cf79c9ac161edce165a33729c5573647170c746faf628795f2fafaa0b2"
        "295e3af32d0d9c419d3c871d0f9cce5dd2d106c793ef33c04b61606ec74c50e0"
        "a28487c217c5d20f1f768bca632c405cbac6c94f733e344cdef2c3b66cc7a156"
        "61ccaac7b2dc3e1c1edeb61fe0feff25350000",
    )
    assert (
        hashlib.sha256(benchmark(lzxpress_huffman.decompress, buf)).hexdigest()
        == "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529"
    )
