from __future__ import annotations

import random
from dataclasses import dataclass
from typing import Any, TYPE_CHECKING

if TYPE_CHECKING:
    from ...protocols import ScreenProtocol, AsciiQuariumProtocol
    from ...screen_compat import Screen
else:
    from ...screen_compat import Screen

# Maximum bubble lifetime to prevent memory leaks
MAX_BUBBLE_LIFETIME = 10.0


@dataclass
class Bubble:
    """Simple bubble entity providing animated atmospheric effects.

    Bubbles are lightweight visual effects generated by fish and other entities
    to enhance the aquatic atmosphere. They provide simple upward-floating
    animation with randomized visual appearance and automatic cleanup.

    Key Features:
        **Upward Motion**: Bubbles continuously float upward at a fixed speed,
        creating a natural underwater bubble effect.

        **Visual Variety**: Random character selection (".", "o", "O") provides
        visual variety and gives the impression of different bubble sizes.

        **Automatic Cleanup**: Bubbles have a maximum lifetime to prevent
        memory leaks and ensure performance stability.

        **Lightweight Design**: Minimal state and computation for high-frequency
        spawning without performance impact.

    Architecture:
        The Bubble class is designed as a simple data structure with minimal
        logic for maximum performance. It implements the Actor protocol for
        consistent integration with the entity system.

    Lifecycle:
        1. **Creation**: Spawned by fish or other entities at specific coordinates
        2. **Animation**: Continuous upward movement with lifetime tracking
        3. **Rendering**: Random character display with cyan coloring
        4. **Cleanup**: Automatic removal when lifetime expires or off-screen

    Performance:
        Bubbles are designed for high-frequency creation and destruction (multiple
        per second per fish). The implementation prioritizes minimal memory usage
        and computational overhead over complex visual effects.

    Attributes:
        x (int): Horizontal position (column) on screen
        y (int): Vertical position (row) on screen
        lifetime (float): Age of bubble in seconds (for cleanup tracking)

    Constants:
        MAX_BUBBLE_LIFETIME (float): Maximum age before automatic cleanup (10.0 seconds)

    Example:
        >>> # Create bubble at fish position
        >>> bubble = Bubble(x=25, y=15)
        >>>
        >>> # Update in game loop
        >>> bubble.update(dt=0.016, screen=screen, app=aquarium)
        >>> bubble.draw(screen)
        >>>
        >>> # Check if still active
        >>> if not bubble.active:
        ...     # Remove from bubble list
        ...     bubbles.remove(bubble)

    See Also:
        - Fish: Primary entity that generates bubbles
        - Actor: Base protocol for entity system integration
        - Entity System Documentation: docs/ENTITY_SYSTEM.md
    """

    x: int
    y: int
    lifetime: float = 0

    @property
    def active(self) -> bool:
        """Check if bubble is still active (hasn't exceeded max lifetime)"""
        return self.lifetime < MAX_BUBBLE_LIFETIME

    def update(self, dt: float, screen: "Screen", app: "AsciiQuariumProtocol") -> None:
        self.lifetime += dt
        self.y -= max(1, int(10 * dt))

    def draw(self, screen: Screen):
        if 0 <= self.y < screen.height:
            bubble_char = random.choice([".", "o", "O"])
            screen.print_at(bubble_char, self.x, self.y, colour=Screen.COLOUR_CYAN)
