import logging

from functools import wraps
from inspect import signature
from typing import Literal

import numba as nb
import numpy as np
import pandas as pd
import pytensor.tensor as pt
import sympy as sp

from pytensor.graph.basic import Apply
from pytensor.graph.op import Op
from pytensor.tensor import TensorVariable
from scipy import linalg

from gEconpy.classes.containers import SymbolDictionary
from gEconpy.classes.time_aware_symbol import TimeAwareSymbol
from gEconpy.model.compile import BACKENDS, compile_function
from gEconpy.numbaf.overloads import nb_ordqz
from gEconpy.solvers.gensys import _gensys_setup
from gEconpy.utilities import eq_to_ss, get_name, simplify_matrix

_log = logging.getLogger(__name__)


def override_dummy_wrapper(f, param_name="not_loglin_variable"):
    """
    Wrap a function to map a parameter name to a _Dummy argument in a sympy lambdify generated function.

    To have a 1d array input to a sympy lambdify function, it is necessary to use an IndexBase. IndexBase,
    unfortunately, always ends up as a Dummy value when lambdified. This wrapper finds a single dummy value
    in a function signature, and automatically maps the parameter name to it.

    Parameters
    ----------
    f: Callable
        Function generated by sympy.lambidfy, with exactly one dummy variable
    param_name: str
        Named arugment that will be mapped to the dummy in the wrapped function

    Returns
    -------
    inner: Callable
        Same as f, with a keyword argument "param_name" that maps to the Dummy input

    """
    sig = signature(f)
    f_inputs = list(sig.parameters.keys())

    # If the parameter is already in the function signature, we're copacetic and don't need to wrap the function
    if param_name in f_inputs:
        return f

    dummies = [x for x in f_inputs if x.startswith("_Dummy")]
    assert len(dummies) == 1

    @wraps(f)
    def inner(*args, **kwargs):
        loglin = kwargs.pop(param_name)
        kwargs[dummies[0]] = loglin

        return f(*args, **kwargs)

    return inner


def make_all_variable_time_combinations(
    variables,
) -> tuple[list[TimeAwareSymbol], list[TimeAwareSymbol], list[TimeAwareSymbol]]:
    """
    Shift a list of TimeAwareSymbols to create all possible lags, current, and lead variables.

    Parameters
    ----------
    variables: List[TimeAwareSymbol]
        List of variables to shift.

    Returns
    -------
    lags: List[TimeAwareSymbol]
        List of variables shifted to t-1.
    now: List[TimeAwareSymbol]
        List of variables at time t.
    leads: List[TimeAwareSymbol]
        List of variables shifted to t+1.
    """
    # Set all variables to time t, remove duplicates, and sort by base name.
    now = list({x.set_t(0) for x in variables})
    now = sorted(now, key=lambda x: x.base_name)

    # Create lags and leads by shifting the time of the variables.
    lags = [x.step_backward() for x in now]
    leads = [x.step_forward() for x in now]

    return lags, now, leads


def linearize_model(
    variables: list[TimeAwareSymbol],
    equations: list[sp.Expr],
    shocks: list[TimeAwareSymbol],
    order=1,
) -> tuple[list[sp.Matrix, ...], sp.Symbol]:
    r"""
    Log-linearize a model around its steady state.

    Parameters
    ----------
    variables: List[TimeAwareSymbol]
        List of all variables in the model, expressed at time t

    equations: List[sp.Expr]
        List of equations that define the model.

    shocks: List[TimeAwareSymbol]
        List of exogenous shocks in the model.

    order: int, default 1
        Order of the linear approximation of the model. Currently only order = 1 is supported.

    Returns
    -------
    Fs: List[sp.Matrix]
        List of matrices representing the log-linearized model.

    not_loglin_variables: sp.Symbol
        A special symbol created by the function that allows transformation between the log-linear and non-log-linear
        representations of the model. See the Notes for details.

    Notes
    -----
    Convert the non-linear model to its log-linear approximation using a first-order Taylor expansion around the
    deterministic steady state. The specific method of log-linearization is taken from ..[1]

    .. math::
        F_1 T y_{t-1} + F_2 @ T @ y_t + F_3 @ T @ y_{t+1} + F4 \varepsilon_t = 0

    Each of F1, F2, F3, and F4 are the Jacobian matrices of the model equations with respect to the variables at time
    t-1, t, t+1, and the exogenous shocks, respectively.

    The T matrix requires special note. It is a diagonal matrix with either the steady state value of the variable or
    1, depending on whether the variable is log-linearized or not. Specifically:

    .. math::
        T = \text{Diagonal}(y_{ss}^{1 - \text{not_loglin_variable})

    Where :math:`\text{not_loglin_variable}` is a vector whose :math:`i`-th value is zero if the :math:`i`-th variable
    is log-linearized, and one otherwise. The :math:`T` matrix arises from application of the chain rule. When a
    variable is assumed to be represented in logs, it is entered into all model equations as :math:`\exp(y)` (indeed,
    Dynare requires the research to do exactly this). Having made this substitution, the partial derivative of a model
    equation :math:`f(exp(x))` with respect to :math:`x` is:

    .. math::
       \frac{\partial f}{\partial x_ss} f(exp(x_ss)) = f'(exp(x_ss)) \cdot exp(x_ss)

    Since we interpret the variable :math:`y_{ss}` as (implicitly) being in logs, this simplifies to
    :math:`f'(y_ss) \cdot y_{ss}`. On the other hand, if we are not log-linearizing the variable, the partial derivative
    is simply :math:`f'(y_ss)`. By setting the value of the exponent to 1 or 0, we can obtain the correct value of the
    derivative for each equation, with respect to each variable.

    Evaluating the matrix multiplications between each :math:`F` matrix and the :math:`T` matrix, we obtain the
    following simplified expression:

    .. math::
        A y_{t-1} + B y_t + C y_{t+1} + D \varepsilon = 0

    Matrices A, B, C, and D are returned by this function.

    References
    ----------
    [1] gEcon User's Guide, page 54, equation 9.9.
    """
    if order != 1:
        raise NotImplementedError("Only order = 1 linearization is currently implemented.")

    ss_variables = [x.to_ss() for x in variables]
    lags, now, leads = make_all_variable_time_combinations(variables)

    eq_vec = sp.Matrix(equations)
    A, B, C = (eq_to_ss(eq_vec.jacobian(var_group)) for var_group in [lags, now, leads])
    A, B, C = (simplify_matrix(x) for x in [A, B, C])
    D = eq_to_ss(eq_vec.jacobian(shocks)) if shocks else sp.ZeroMatrix(*eq_vec.shape)

    not_loglin_var = sp.IndexedBase("not_loglin_variable", shape=(len(variables),))
    T = sp.diag(*[ss_var ** (1 - not_loglin_var[i]) for i, ss_var in enumerate(ss_variables)])

    if shocks:
        shock_subs = {x.to_ss(): 0 for x in shocks}
        A, B, C, D = (x.subs(shock_subs) for x in [A, B, C, D])

    Fs = [A @ T, B @ T, C @ T, D]

    return Fs, not_loglin_var


def make_not_loglin_flags(
    variables: list[TimeAwareSymbol],
    calibrated_params: list[sp.Symbol],
    steady_state: SymbolDictionary[str, float],
    log_linearize: bool = True,
    not_loglin_variables: list[str] | None = None,
    loglin_negative_ss: bool = False,
    verbose: bool = True,
):
    if not_loglin_variables is None:
        not_loglin_variables = []
    if not log_linearize:
        return np.ones(len(variables))

    vars_and_calibrated = variables + calibrated_params
    var_names = [get_name(x, base_name=True) for x in vars_and_calibrated]
    unknown_not_login = set(not_loglin_variables) - set(var_names)

    if len(unknown_not_login) > 0:
        raise ValueError(
            f"The following variables were requested not to be log-linearized, but are unknown to the model: "
            f"{', '.join(unknown_not_login)}"
        )

    if verbose and len(not_loglin_variables) > 0:
        _log.warning(
            f"The following variables will not be log-linearized at the user's request: {not_loglin_variables}"
        )

    n_variables = len(vars_and_calibrated)
    not_loglin_flags = np.zeros(n_variables)

    for i, name in enumerate(var_names):
        not_loglin_flags[i] = name in not_loglin_variables

    FLOAT_ZEROS = 1e-8
    ss_values = np.array(list(steady_state.values()))
    ss_zeros = np.abs(ss_values) < FLOAT_ZEROS
    ss_negative = ss_values < 0.0

    if np.any(ss_zeros):
        zero_idxs = np.flatnonzero(ss_zeros)
        zero_vars = [vars_and_calibrated[i] for i in zero_idxs]
        if verbose:
            _log.warning(
                f"The following variables had steady-state values close to zero and will not be log-linearized:"
                f"{[get_name(x) for x in zero_vars]}"
            )

        not_loglin_flags[ss_zeros] = 1

    if np.any(ss_negative) and not loglin_negative_ss:
        neg_idxs = np.flatnonzero(ss_negative)
        neg_vars = [vars_and_calibrated[i] for i in neg_idxs]
        if verbose:
            _log.warning(
                f"The following variables had negative steady-state values and will not be log-linearized:"
                f"{[get_name(x) for x in neg_vars]}"
            )

        not_loglin_flags[neg_idxs] = 1

    return not_loglin_flags


def compile_linearized_system(
    equations: list[sp.Expr],
    variables: list[sp.Symbol | TimeAwareSymbol],
    param_dict: SymbolDictionary[sp.Symbol, float | sp.Expr],
    deterministic_dict: SymbolDictionary[sp.Symbol, sp.Expr],
    calib_dict: SymbolDictionary[sp.Symbol, float | sp.Expr],
    shocks: list[TimeAwareSymbol],
    backend: BACKENDS = "numpy",
    return_symbolic: bool = False,
    cache: dict | None = None,
    **kwargs,
):
    """
    Compile a function that evaluates the linearized system of equations.

    Parameters
    ----------
    equations
    variables
    param_dict
    deterministic_dict
    calib_dict
    shocks
    backend
    return_symbolic
    cache
    kwargs

    Returns
    -------
    f_linearze: Callable
        Function that evaluates the linearized system of equations.
    cache: dict
        Dictionary mapping sympy symbols to pytensor tensors. Empty if backend is not pytensor
    """
    cache = {} if cache is None else cache

    ss_variables = [x.to_ss() for x in variables]

    parameters = list((param_dict | deterministic_dict).to_sympy().keys())
    parameters = [x for x in parameters if x not in calib_dict.to_sympy()]
    calib_params = list(calib_dict.to_sympy().keys())

    outputs, not_loglin_var = linearize_model(variables, equations, shocks)
    inputs = ss_variables + calib_params + parameters + [not_loglin_var]

    f_linearize, cache = compile_function(
        inputs,
        outputs,
        backend=backend,
        cache=cache,
        return_symbolic=return_symbolic,
        **kwargs,
    )

    return f_linearize, cache


def residual_norms(B, C, D, Q, P, A_prime, R_prime, S_prime):
    norm_deterministic = linalg.norm(A_prime + B @ R_prime + C @ R_prime @ P)

    norm_stochastic = linalg.norm(B @ S_prime + C @ R_prime @ Q + D)

    return norm_deterministic, norm_stochastic


def statespace_to_gEcon_representation(A, T, R, tol):
    n_vars = T.shape[1]
    n_shocks = R.shape[1]

    state_var_idx = np.where(np.abs(T[np.argmax(np.abs(T), axis=0), np.arange(n_vars)]) >= tol)[0]
    state_var_mask = np.isin(np.arange(n_vars), state_var_idx)

    shock_idx = np.arange(n_shocks)

    PP = T.copy()
    PP[np.where(np.abs(PP) < tol)] = 0
    QQ = R.copy()
    QQ = QQ[:n_vars, :]
    QQ[np.where(np.abs(QQ) < tol)] = 0

    P = PP[state_var_mask, :][:, state_var_mask]
    Q = QQ[state_var_mask, :][:, shock_idx]
    R = PP[~state_var_mask, :][:, state_var_idx]
    S = QQ[~state_var_mask, :][:, shock_idx]

    A_prime = A[:, state_var_mask]
    R_prime = PP[:, state_var_mask]
    S_prime = QQ[:, shock_idx]

    return P, Q, R, S, A_prime, R_prime, S_prime


def check_perturbation_solution(A, B, C, D, T, R, tol=1e-8):
    P, Q, R, S, A_prime, R_prime, S_prime = statespace_to_gEcon_representation(A, T, R, tol)
    norm_deterministic, norm_stochastic = residual_norms(B, C, D, Q, P, A_prime, R_prime, S_prime)

    _log.info(f"Norm of deterministic part: {norm_deterministic:0.9f}")
    _log.info(f"Norm of stochastic part:    {norm_stochastic:0.9f}")


# TODO: Cannot cache this I think because of the call to ordqz -- test if this is true
@nb.njit(cache=False)
def _compute_solution_eigenvalues(A, B, C, D, tol=1e-8) -> np.array:
    Gamma_0, Gamma_1, _, _, _ = _gensys_setup(A, B, C, D, tol)

    # Using scipy instead of qzdiv appears to offer a huge speedup for nearly the same answer; some eigenvalues
    # have sign flip relative to qzdiv -- does it matter?
    A, B, alpha, beta, Q, Z = nb_ordqz(-Gamma_0, Gamma_1, sort="ouc", output="complex")

    eigenval = np.diag(B) / (np.diag(A) + tol)
    n_eigs = len(eigenval)

    eig = np.empty((n_eigs, 3))
    eig[:, 0] = np.abs(eigenval)
    eig[:, 1] = np.real(eigenval)
    eig[:, 2] = np.imag(eigenval)

    sorted_idx = np.argsort(eig[:, 0])
    return eig[sorted_idx, :]


def check_bk_condition(
    A,
    B,
    C,
    D,
    tol=1e-8,
    verbose=True,
    on_failure: Literal["raise", "ignore"] = "ignore",
    return_value: Literal["dataframe", "bool", None] = "dataframe",
) -> bool | pd.DataFrame | None:
    """
    Compute the generalized eigenvalues of system in the form presented in [1].

    Per [2], the number of unstable eigenvalues (:math:`|v| > 1`) should not be greater than the number of
    forward-looking variables. Failing this test suggests timing problems in the definition of the model.

    Parameters
    ----------
    A: np.ndarray
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to past variables
        values that are known when decision-making: those with t-1 subscripts.
    B: np.ndarray
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to variables that
        are observed when decision-making: those with t subscripts.
    C: np.ndarray
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to variables that
        enter in expectation when decision-making: those with t+1 subscripts.
    D: np.ndarray
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to exogenous shocks.
    verbose: bool, default: True
        Flag to print the results of the test, otherwise the eigenvalues are returned without comment.
    on_failure: str, default: 'ignore'
        Action to take if the Blanchard-Kahn condition is not satisfied. Valid values are 'ignore' and 'raise'.
    return_value: string, default: 'dataframe'
        Controls what is returned by the function. Valid values are 'dataframe', 'bool', and 'none'.
        If df, a dataframe containing eigenvalues is returned. If 'bool', a boolean indicating whether the BK
        condition is satisfied. If None, nothing is returned.
    tol: float, 1e-8
        Tolerance below which numerical values are considered zero

    Returns
    -------
    bk_result, bool or pd.DataFrame, optional.
        Return value requested. Datatype corresponds to what was requested in the ``return_value`` argument:

        - None, If return_value is 'none'
        - condition_satisfied, bool, if return_value is 'bool', returns True if the Blanchard-Kahn condition is
          satisfied, False otherwise.
        - Eigenvalues, pd.DataFrame, if return_value is 'df', returns a dataframe containing the real and imaginary
          components of the system's, eigenvalues, along with their modulus.
    """
    if return_value not in ["dataframe", "bool", None]:
        raise ValueError(f'Unknown return type "{return_value}"')

    n_forward = (np.abs(C).sum(axis=0) > tol).sum().astype(int)
    eig = pd.DataFrame(
        _compute_solution_eigenvalues(A, B, C, D, tol),
        columns=["Modulus", "Real", "Imaginary"],
    )
    n_greater_than_one = (eig["Modulus"] > 1).sum()
    condition_not_satisfied = n_forward != n_greater_than_one

    message = (
        f"Model solution has {n_greater_than_one} eigenvalues greater than one in modulus and {n_forward} "
        f"forward-looking variables. "
        f"\nBlanchard-Kahn condition is{' NOT' if condition_not_satisfied else ''} satisfied."
    )

    if condition_not_satisfied:
        if n_greater_than_one > n_forward:
            reason = "No stable solution (More unstable eigenvalues than forward-looking variables)"
        else:
            reason = "No unique solution (More forward-looking variables than unstable eigenvalues)"

        message += " " + reason

    if condition_not_satisfied and on_failure == "raise":
        raise ValueError(message)

    if verbose:
        _log.info(message)

    if return_value is None:
        return None
    if return_value == "dataframe":
        return eig
    return ~condition_not_satisfied


class BlanchardKahnCondition(Op):
    def __init__(self, tol=1e-8):
        self.tol = tol
        super().__init__()

    def make_node(self, A, B, C, D) -> Apply:
        inputs = list(map(pt.as_tensor, [A, B, C, D]))
        outputs = [
            pt.scalar("bk_flag", dtype=bool),
            pt.scalar("n_forward", dtype=int),
            pt.scalar("n_greater_than_one", dtype=int),
        ]

        return Apply(self, inputs, outputs)

    def perform(self, node: Apply, inputs: list[np.ndarray], outputs: list[list[None]]) -> None:
        A, B, C, D = inputs
        eig = check_bk_condition(A, B, C, D, tol=self.tol, verbose=False, return_value="dataframe")

        FLOAT_ZERO = 1e-8
        n_forward = (np.abs(C).sum(axis=0) > FLOAT_ZERO).sum().astype(int)
        n_greater_than_one = (eig["Modulus"] > 1).sum()

        condition_not_satisfied = n_forward != n_greater_than_one

        outputs[0][0] = np.array(condition_not_satisfied)
        outputs[1][0] = np.array(n_forward)
        outputs[2][0] = np.array(n_greater_than_one)


def check_bk_condition_pt(
    A: TensorVariable,
    B: TensorVariable,
    C: TensorVariable,
    D: TensorVariable,
    tol: float = 1e-8,
):
    """
    Check the Blanchard-Kahn condition for a model.

    Compute the generalized eigenvalues of system in the form presented in [1]. Per [2], the number of
    unstable eigenvalues (:math:`|v| > 1`) should not be greater than the number of forward-looking variables. Failing
    this test suggests timing problems in the definition of the model.

    Parameters
    ----------
    A: TensorVariable
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to past variables
        values that are known when decision-making: those with t-1 subscripts.
    B: TensorVariable
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to variables that
        are observed when decision-making: those with t subscripts.
    C: TensorVariable
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to variables that
        enter in expectation when decision-making: those with t+1 subscripts.
    D: TensorVariable
        Jacobian matrix of the DSGE system, evaluated at the steady state, taken with respect to exogenous shocks.
    tol: float, optional
        Tolerance below which numerical values are considered zero. Default is 1e-8.

    Returns
    -------
    bk_flag: bool
        True if the Blanchard-Kahn condition is satisfied, False otherwise.
    n_forward: int
        Number of forward-looking variables in the model.
    n_greater_than_one: int
        Number of eigenvalues greater than one in modulus.
    """
    return BlanchardKahnCondition(tol=tol)(A, B, C, D)
