# Disclaimer: Most tests in this file are generated by the VSCode Agent with Claude Sonnet 4.
import sys
from unittest.mock import patch

import numpy as np
import pytest

from debug_gym.gym.tools.tool import ToolCall
from debug_gym.gym.tools.toolbox import Toolbox
from debug_gym.llms import Human
from debug_gym.llms.human import (
    CommandParser,
    DynamicToolCommandCompleter,
    ToolCommandValidator,
)

# Import prompt_toolkit dependencies for testing, with fallback mocks
try:
    from prompt_toolkit.document import Document
    from prompt_toolkit.validation import ValidationError

    prompt_toolkit_available = sys.stdout.isatty()
except ImportError:
    # Create mock classes if prompt_toolkit is not available
    class ValidationError(Exception):
        def __init__(self, message="", cursor_position=0):
            super().__init__(message)
            self.message = message
            self.cursor_position = cursor_position

    class Document:
        def __init__(self, text="", cursor_position=None):
            self.text = text
            self.cursor_position = (
                cursor_position if cursor_position is not None else len(text)
            )

    prompt_toolkit_available = False


@pytest.fixture
def example_tools():
    """
    Build example tools for testing - includes various argument types.
    """
    return [
        {
            "id": "print-001",
            "name": "print",
            "description": "Print a message to stdout",
            "arguments": {
                "msg": {"type": ["string"], "description": "Message to print"}
            },
        },
        {
            "id": "sum-002",
            "name": "sum",
            "description": "Add two numbers",
            "arguments": {
                "a": {"type": ["number"], "description": "First number"},
                "b": {"type": ["number"], "description": "Second number"},
            },
        },
        {
            "id": "toggle-003",
            "name": "toggle",
            "description": "Toggle a boolean flag",
            "arguments": {
                "enabled": {"type": ["boolean"], "description": "Enable or disable"},
                "verbose": {
                    "type": ["boolean", "null"],
                    "description": "Optional verbose mode",
                },
            },
        },
    ]


@pytest.fixture
def simple_tools():
    """Simplified tools for basic testing"""
    return [
        {
            "id": "cmd1-001",
            "name": "cmd1",
            "description": "First command",
            "arguments": {
                "arg1": {"type": ["string"], "description": "First argument"}
            },
        },
        {
            "id": "cmd2-002",
            "name": "cmd2",
            "description": "Second command",
            "arguments": {
                "arg2": {"type": ["number"], "description": "Second argument"}
            },
        },
    ]


class TestCommandParser:
    """Tests for the CommandParser class"""

    def test_init(self):
        """Test CommandParser initialization"""
        parser = CommandParser()
        assert hasattr(parser, "patterns")
        assert "command" in parser.patterns
        assert "arg_name" in parser.patterns
        assert "boolean" in parser.patterns
        assert "number" in parser.patterns

    def test_parse_command_empty_input(self):
        """Test parsing empty input"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("")
        assert command is None
        assert args == {}
        assert len(errors) == 1
        assert errors[0][0] == "Empty input"

    def test_parse_command_simple_command(self):
        """Test parsing a simple command without arguments"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test_command")
        assert command == "test_command"
        assert args == {}
        assert errors == []

    def test_parse_command_invalid_command_name(self):
        """Test parsing invalid command names"""
        parser = CommandParser()

        # Test command starting with number
        command, args, errors = parser.parse_command("123invalid")
        assert command is None
        assert args == {}
        assert len(errors) == 1
        assert "Invalid command name" in errors[0][0]

        # Test command with special characters
        command, args, errors = parser.parse_command("invalid-command")
        assert command is None
        assert args == {}
        assert len(errors) == 1

    def test_parse_command_with_string_arguments(self):
        """Test parsing command with string arguments"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test arg1=value1 arg2=value2")
        assert command == "test"
        assert args == {"arg1": "value1", "arg2": "value2"}
        assert errors == []

    def test_parse_command_with_boolean_arguments(self):
        """Test parsing command with boolean arguments"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test flag1=true flag2=false")
        assert command == "test"
        assert args == {"flag1": True, "flag2": False}
        assert errors == []

    def test_parse_command_with_number_arguments(self):
        """Test parsing command with number arguments"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test int_arg=42 float_arg=3.14")
        assert command == "test"
        assert args == {"int_arg": 42, "float_arg": 3.14}
        assert errors == []

    def test_parse_command_with_quoted_strings(self):
        """Test parsing command with quoted string arguments"""
        parser = CommandParser()

        # Double quotes
        command, args, errors = parser.parse_command('test msg="hello world"')
        assert command == "test"
        assert args == {"msg": "hello world"}
        assert errors == []

        # Single quotes
        command, args, errors = parser.parse_command("test msg='hello world'")
        assert command == "test"
        assert args == {"msg": "hello world"}
        assert errors == []

    def test_parse_command_with_escaped_quotes(self):
        """Test parsing command with escaped quotes in strings"""
        parser = CommandParser()
        command, args, errors = parser.parse_command(r'test msg="hello \"world\""')
        assert command == "test"
        assert args == {"msg": 'hello "world"'}
        assert errors == []

    def test_parse_command_unclosed_string(self):
        """Test parsing command with unclosed string"""
        parser = CommandParser()
        command, args, errors = parser.parse_command('test msg="unclosed string')
        assert command == "test"
        assert len(errors) >= 1
        assert any("Unclosed string literal" in error[0] for error in errors)

    def test_parse_command_missing_equals(self):
        """Test parsing command with missing equals sign"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test arg1 value1")
        assert command == "test"
        assert len(errors) == 1
        assert "Expected '=' after argument name" in errors[0][0]

    def test_parse_command_missing_value(self):
        """Test parsing command with missing argument value"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test arg1=")
        assert command == "test"
        assert len(errors) == 1
        assert "Missing value for argument 'arg1'" in errors[0][0]

    def test_parse_command_invalid_argument_name(self):
        """Test parsing command with invalid argument names"""
        parser = CommandParser()
        command, args, errors = parser.parse_command("test 123invalid=value")
        assert command == "test"
        assert len(errors) == 1
        assert "Invalid argument name" in errors[0][0]

    def test_parse_command_complex_example(self):
        """Test parsing a complex command with mixed argument types"""
        parser = CommandParser()
        cmd = 'test name="John Doe" age=30 active=true score=95.5'
        command, args, errors = parser.parse_command(cmd)
        assert command == "test"
        assert args == {"name": "John Doe", "age": 30, "active": True, "score": 95.5}
        assert errors == []


@pytest.mark.skipif(not prompt_toolkit_available, reason="prompt_toolkit not available")
class TestToolCommandValidator:
    """Tests for the ToolCommandValidator class"""

    def test_init(self, example_tools):
        """Test ToolCommandValidator initialization"""
        validator = ToolCommandValidator(example_tools)
        assert validator.command_names == ["print", "sum", "toggle"]
        assert "print" in validator.command_args
        assert "sum" in validator.command_args
        assert isinstance(validator.parser, CommandParser)

    def test_validate_argument_valid_types(self, example_tools):
        """Test validate_argument with valid argument types"""
        validator = ToolCommandValidator(example_tools)

        # Test string argument
        arg_info = {"type": ["string"], "description": "Test"}
        is_valid, value, error = validator.validate_argument("test", "hello", arg_info)
        assert is_valid
        assert value == "hello"
        assert error is None

        # Test boolean argument
        arg_info = {"type": ["boolean"], "description": "Test"}
        is_valid, value, error = validator.validate_argument("test", True, arg_info)
        assert is_valid
        assert value is True
        assert error is None

        # Test number argument
        arg_info = {"type": ["number"], "description": "Test"}
        is_valid, value, error = validator.validate_argument("test", 42, arg_info)
        assert is_valid
        assert value == 42
        assert error is None

    def test_validate_argument_invalid_types(self, example_tools):
        """Test validate_argument with invalid argument types"""
        validator = ToolCommandValidator(example_tools)

        # Test wrong type
        arg_info = {"type": ["number"], "description": "Test"}
        is_valid, value, error = validator.validate_argument(
            "test", "not_a_number", arg_info
        )
        assert not is_valid
        assert value is None
        assert "Invalid value for test" in error

    def test_validate_argument_unknown_argument(self, example_tools):
        """Test validate_argument with unknown argument"""
        validator = ToolCommandValidator(example_tools)
        is_valid, value, error = validator.validate_argument("unknown", "value", None)
        assert not is_valid
        assert value is None
        assert "Unknown argument: unknown" in error

    def test_validate_empty_input(self, example_tools):
        """Test validation with empty input"""
        validator = ToolCommandValidator(example_tools)
        document = Document("")
        # Should not raise ValidationError for empty input
        validator.validate(document)

    def test_validate_unknown_command(self, example_tools):
        """Test validation with unknown command"""
        validator = ToolCommandValidator(example_tools)
        document = Document("unknown_command")

        with pytest.raises(ValidationError) as exc_info:
            validator.validate(document)
        assert "Unknown command: unknown_command" in str(exc_info.value)

    def test_validate_valid_command_no_args(self, example_tools):
        """Test validation with valid command and no arguments"""
        validator = ToolCommandValidator(example_tools)
        # Toggle has optional arguments (with null type)
        document = Document("toggle")

        with pytest.raises(ValidationError) as exc_info:
            validator.validate(document)
        # Should complain about missing mandatory argument
        assert "Missing mandatory argument 'enabled'" in str(exc_info.value)

    def test_validate_valid_command_with_args(self, example_tools):
        """Test validation with valid command and arguments"""
        validator = ToolCommandValidator(example_tools)
        document = Document("print msg=hello")
        # Should not raise ValidationError
        validator.validate(document)

    def test_validate_command_with_unknown_argument(self, example_tools):
        """Test validation with command containing unknown argument"""
        validator = ToolCommandValidator(example_tools)
        document = Document("print msg=hello unknown_arg=value")

        with pytest.raises(ValidationError) as exc_info:
            validator.validate(document)
        assert "Unknown argument 'unknown_arg' for command 'print'" in str(
            exc_info.value
        )

    def test_validate_command_with_invalid_argument_value(self, example_tools):
        """Test validation with command containing invalid argument value"""
        validator = ToolCommandValidator(example_tools)
        document = Document("sum a=not_a_number b=42")

        with pytest.raises(ValidationError) as exc_info:
            validator.validate(document)
        assert "Invalid value for a" in str(exc_info.value)

    def test_validate_command_with_optional_null_argument(self, example_tools):
        """Test validation with command that has optional (null) arguments"""
        validator = ToolCommandValidator(example_tools)
        # Toggle has 'verbose' as optional argument with null type
        document = Document("toggle enabled=true")
        # Should not raise ValidationError as verbose is optional
        validator.validate(document)


class TestDynamicToolCommandCompleter:
    """Tests for the DynamicToolCommandCompleter class"""

    def test_init(self, example_tools):
        """Test DynamicToolCommandCompleter initialization"""
        completer = DynamicToolCommandCompleter(example_tools)
        assert completer.command_names == ["print", "sum", "toggle"]
        assert "print" in completer.command_args
        assert "print" in completer.command_descriptions
        assert "print" in completer.command_arg_descriptions

    def test_get_used_args(self, simple_tools):
        """Test _get_used_args method"""
        completer = DynamicToolCommandCompleter(simple_tools)

        # No arguments
        used = completer._get_used_args([])
        assert used == []

        # Single argument
        used = completer._get_used_args(["arg1=value"])
        assert used == ["arg1"]

        # Multiple arguments
        used = completer._get_used_args(["arg1=value1", "arg2=value2"])
        assert used == ["arg1", "arg2"]

        # Mixed with non-argument parts
        used = completer._get_used_args(["arg1=value1", "not_an_arg", "arg2=value2"])
        assert used == ["arg1", "arg2"]

    def test_get_command_completions_empty_prefix(self, simple_tools):
        """Test command completions with empty prefix"""
        completer = DynamicToolCommandCompleter(simple_tools)
        completions = list(completer._get_command_completions(""))

        assert len(completions) == 2
        completion_texts = [c.text for c in completions]
        assert "cmd1" in completion_texts
        assert "cmd2" in completion_texts

    def test_get_command_completions_with_prefix(self, simple_tools):
        """Test command completions with prefix"""
        completer = DynamicToolCommandCompleter(simple_tools)
        completions = list(completer._get_command_completions("cmd1"))

        assert len(completions) == 1
        assert completions[0].text == "cmd1"
        assert completions[0].start_position == -4  # length of "cmd1"

    def test_get_argument_completions(self, simple_tools):
        """Test argument completions"""
        completer = DynamicToolCommandCompleter(simple_tools)
        completions = list(completer._get_argument_completions("cmd1", []))

        assert len(completions) == 1
        assert completions[0].text == "arg1="
        assert completions[0].start_position == 0

    def test_get_argument_completions_with_used_args(self, example_tools):
        """Test argument completions excluding already used arguments"""
        completer = DynamicToolCommandCompleter(example_tools)
        completions = list(completer._get_argument_completions("sum", ["a"]))

        assert len(completions) == 1
        assert completions[0].text == "b="

    def test_get_argument_name_completions(self, example_tools):
        """Test argument name completions with prefix"""
        completer = DynamicToolCommandCompleter(example_tools)
        completions = list(completer._get_argument_name_completions("sum", "a", []))

        assert len(completions) == 1
        assert completions[0].text == "a="
        assert completions[0].start_position == -1  # length of "a"

    def test_get_argument_value_completions_boolean(self, example_tools):
        """Test argument value completions for boolean type"""
        completer = DynamicToolCommandCompleter(example_tools)
        completions = list(
            completer._get_argument_value_completions("toggle", "enabled")
        )

        assert len(completions) == 2
        completion_texts = [c.text for c in completions]
        assert "true" in completion_texts
        assert "false" in completion_texts

    def test_get_argument_value_completions_non_boolean(self, example_tools):
        """Test argument value completions for non-boolean type"""
        completer = DynamicToolCommandCompleter(example_tools)
        completions = list(completer._get_argument_value_completions("print", "msg"))

        # Should return empty for non-boolean types
        assert len(completions) == 0

    def test_get_completions_empty_document(self, simple_tools):
        """Test get_completions with empty document"""
        completer = DynamicToolCommandCompleter(simple_tools)
        document = Document("")
        completions = list(completer.get_completions(document, None))

        # Should suggest all commands
        assert len(completions) == 2
        completion_texts = [c.text for c in completions]
        assert "cmd1" in completion_texts
        assert "cmd2" in completion_texts

    def test_get_completions_partial_command(self, simple_tools):
        """Test get_completions with partial command"""
        completer = DynamicToolCommandCompleter(simple_tools)
        document = Document("cmd1")
        completions = list(completer.get_completions(document, None))

        assert len(completions) == 1
        assert completions[0].text == "cmd1"

    def test_get_completions_command_with_space(self, simple_tools):
        """Test get_completions after command with space"""
        completer = DynamicToolCommandCompleter(simple_tools)
        document = Document("cmd1 ")
        completions = list(completer.get_completions(document, None))

        # Should suggest arguments
        assert len(completions) == 1
        assert completions[0].text == "arg1="

    def test_get_completions_unknown_command(self, simple_tools):
        """Test get_completions with unknown command"""
        completer = DynamicToolCommandCompleter(simple_tools)
        document = Document("unknown ")
        completions = list(completer.get_completions(document, None))

        # Should return no completions for unknown command
        assert len(completions) == 0

    def test_get_completions_partial_argument_name(self, example_tools):
        """Test get_completions with partial argument name"""
        completer = DynamicToolCommandCompleter(example_tools)
        document = Document("sum a")
        completions = list(completer.get_completions(document, None))

        assert len(completions) == 1
        assert completions[0].text == "a="

    def test_get_completions_argument_value(self, example_tools):
        """Test get_completions for argument value"""
        completer = DynamicToolCommandCompleter(example_tools)
        document = Document("toggle enabled=")
        completions = list(completer.get_completions(document, None))

        # Should suggest boolean values
        assert len(completions) == 2
        completion_texts = [c.text for c in completions]
        assert "true" in completion_texts
        assert "false" in completion_texts


@patch(
    (
        "prompt_toolkit.PromptSession.prompt"
        if prompt_toolkit_available
        else "builtins.input"
    ),
    lambda *args, **kwargs: 'pdb command="b 10"',
)
def test_human_integration(build_env_info):
    """Test Human LLM integration with command parsing"""
    # Always generate the same random toolcall id: "pdb-637469"
    np.random.seed(42)
    human = Human()
    messages = [{"role": "user", "content": "Hello"}]
    env_info = build_env_info(
        action_tool_call=ToolCall(
            id="pdb-637469", name="pdb", arguments={"command": "b 10"}
        ),
        tools=[Toolbox.get_tool("pdb"), Toolbox.get_tool("view")],
    )
    llm_response = human(messages, env_info.tools)
    # Human only uses the messages content
    assert llm_response.prompt == [{"role": "user", "content": "Hello"}]
    assert llm_response.response == 'pdb command="b 10"'
    assert llm_response.tool.name == "pdb"
    assert llm_response.tool.arguments == {"command": "b 10"}


def test_human_max_retries_proper_behavior(build_env_info):
    """Test Human LLM max retries - now works correctly after bug fix"""
    human = Human(max_retries=3)  # Allow 3 retries so the third valid input succeeds
    messages = [{"role": "user", "content": "Test message"}]
    env_info = build_env_info(
        tools=[Toolbox.get_tool("pdb"), Toolbox.get_tool("view")],
    )

    # Mock input to return invalid input twice, then valid input
    inputs = ["", "invalid command syntax", 'pdb command="help"']
    input_to_patch = (
        "prompt_toolkit.PromptSession.prompt"
        if prompt_toolkit_available
        else "builtins.input"
    )
    with patch(input_to_patch, side_effect=inputs):
        with patch("numpy.random.randint", return_value=np.array([1, 2, 3, 4])):
            llm_response = human(messages, env_info.tools)
            # Should succeed on the third try
            assert llm_response.tool.name == "pdb"
            assert llm_response.tool.arguments == {"command": "help"}


def test_human_max_retries_exceeded(build_env_info):
    """Test Human LLM when max retries is exceeded"""
    human = Human(max_retries=2)
    messages = [{"role": "user", "content": "Test message"}]
    env_info = build_env_info(
        tools=[Toolbox.get_tool("pdb"), Toolbox.get_tool("view")],
    )

    # Mock input to always return invalid input
    input_to_patch = (
        "prompt_toolkit.PromptSession.prompt"
        if prompt_toolkit_available
        else "builtins.input"
    )
    with patch(input_to_patch, return_value=""):
        with pytest.raises(
            ValueError, match="Maximum retries \\(2\\) reached without valid input"
        ):
            human(messages, env_info.tools)


def test_human_tokenize():
    """Test Human tokenization"""
    human = Human()
    tokens = human.tokenize("hello world test")
    assert tokens == ["hello", "world", "test"]


def test_human_count_tokens():
    """Test Human token counting"""
    human = Human()
    count = human.count_tokens("hello world test")
    assert count == 3


def test_human_define_tools():
    """Test Human tool definitions"""
    human = Human()
    mock_tools = [
        type(
            "MockTool",
            (),
            {
                "name": "test_tool",
                "arguments": {"arg1": "value1"},
                "description": "Test tool description",
            },
        )()
    ]

    # Mock random to get predictable IDs - need to return array
    with patch("numpy.random.randint", return_value=np.array([1, 2, 3, 4, 5, 6])):
        defined_tools = human.define_tools(mock_tools)

    assert len(defined_tools) == 1
    tool = defined_tools[0]
    assert tool["name"] == "test_tool"
    assert tool["arguments"] == {"arg1": "value1"}
    assert tool["description"] == "Test tool description"
    assert tool["id"] == "test_tool-123456"


@patch(
    (
        "prompt_toolkit.PromptSession.prompt"
        if prompt_toolkit_available
        else "builtins.input"
    ),
    lambda *args, **kwargs: "sum a=5 b=10",
)
def test_human_with_number_arguments(build_env_info):
    """Test Human with number arguments"""
    np.random.seed(42)
    human = Human()
    messages = [{"role": "user", "content": "Calculate sum"}]

    # Create a mock tool that accepts number arguments
    mock_tool = type(
        "MockTool",
        (),
        {
            "name": "sum",
            "arguments": {
                "a": {"type": ["number"], "description": "First number"},
                "b": {"type": ["number"], "description": "Second number"},
            },
            "description": "Add two numbers",
        },
    )()

    env_info = build_env_info(tools=[mock_tool])
    llm_response = human(messages, env_info.tools)

    assert llm_response.tool.name == "sum"
    assert llm_response.tool.arguments == {"a": 5, "b": 10}


@patch(
    (
        "prompt_toolkit.PromptSession.prompt"
        if prompt_toolkit_available
        else "builtins.input"
    ),
    lambda *args, **kwargs: "toggle enabled=true",
)
def test_human_with_boolean_arguments(build_env_info):
    """Test Human with boolean arguments"""
    np.random.seed(42)
    human = Human()
    messages = [{"role": "user", "content": "Toggle flag"}]

    # Create a mock tool that accepts boolean arguments
    mock_tool = type(
        "MockTool",
        (),
        {
            "name": "toggle",
            "arguments": {
                "enabled": {"type": ["boolean"], "description": "Enable flag"}
            },
            "description": "Toggle a boolean flag",
        },
    )()

    env_info = build_env_info(tools=[mock_tool])
    llm_response = human(messages, env_info.tools)

    assert llm_response.tool.name == "toggle"
    assert llm_response.tool.arguments == {"enabled": True}


def test_human_format_tool_call_history(build_env_info):
    """Test Human tool call history formatting"""
    human = Human()

    # Create mock response
    tool_call = ToolCall(id="test-123", name="test_tool", arguments={"arg": "value"})
    mock_response = type("MockResponse", (), {})()
    mock_response = [type("MockItem", (), {"tool": tool_call})()]

    # Create mock history info
    history_info = build_env_info(
        action_tool_call=ToolCall(
            id="test-123", name="test_tool", arguments={"arg": "value"}
        ),
        step_observation=type("MockObs", (), {"observation": "Test observation"})(),
    )

    formatted = human.format_tool_call_history(history_info, mock_response)

    assert len(formatted) == 2
    assert formatted[0]["role"] == "assistant"
    assert formatted[0]["content"][0]["type"] == "tool_use"
    assert formatted[1]["role"] == "user"
    assert formatted[1]["content"][0]["type"] == "tool_result"
