import json
import logging
import time
from dataclasses import dataclass
from typing import Dict, List, Optional, Tuple

import numpy as np
import pandas as pd
from scipy.integrate import solve_ivp
from scipy.optimize import curve_fit
from scipy.signal import find_peaks

from .integration.optics import delta_omega_xpm
from .utils import soft_logic

HBAR = 1.054_571_817e-34
C = 299_792_458.0
TWOPI = 2 * np.pi

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


@dataclass
class PhotonicMolecule:
    omega0: float = TWOPI * 193.5e12  # 1550 nm
    kappa_A: float = TWOPI * 0.39e9
    kappa_B: float = TWOPI * 0.39e9
    kappa_eA: float = TWOPI * 0.20e9
    kappa_eB: float = TWOPI * 0.20e9
    J: float = TWOPI * 1.5e9
    g_XPM: float = TWOPI * 2.5e9 / 1e-3  # rad/(s·W)
    chi: float = 0.0
    K: float = 0.0
    heater_efficiency: float = 10e-12  # m/W (10 pm/mW)
    thermal_time_constant: float = 10e-6
    delta_A0: float = 0.0
    delta_B0: float = 0.0

    # P1: physics-based XPM parameters and operating temperature (used in P2)
    T_op: float = 300.0  # K
    xpm_mode: str = "linear"  # "linear" (default) or "physics"
    n2: Optional[float] = None  # Kerr coefficient (m^2/W) for physics mode
    A_eff: float = 0.6e-12  # effective area (m^2)
    n_eff: float = 3.4  # effective index
    g_geom: float = 1.0  # geometry scaling for Δn -> detuning

    def eigenfrequencies(self) -> Tuple[complex, complex]:
        kappa_avg = (self.kappa_A + self.kappa_B) / 2
        kappa_diff = (self.kappa_A - self.kappa_B) / 2
        omega_plus = -1j * kappa_avg / 2 + np.sqrt(self.J**2 - (kappa_diff / 2) ** 2 + 0j)
        omega_minus = -1j * kappa_avg / 2 - np.sqrt(self.J**2 - (kappa_diff / 2) ** 2 + 0j)
        return omega_plus, omega_minus

    def transfer_matrix(
        self,
        omega: float,
        P_ctrl: float = 0.0,
        delta_bias_A: float = 0.0,
        delta_bias_B: float = 0.0,
    ) -> np.ndarray:
        delta_A = omega - self.omega0 + self.delta_A0 + delta_bias_A
        delta_A += self.xpm_detuning(P_ctrl)
        delta_B = omega - self.omega0 + self.delta_B0 + delta_bias_B
        j = 1j
        M = np.array(
            [
                [j * delta_A - self.kappa_A / 2, -1j * self.J],
                [-1j * self.J, j * delta_B - self.kappa_B / 2],
            ],
            dtype=complex,
        )
        return M

    def xpm_detuning(self, P_ctrl: float) -> float:
        """
        Compute XPM-induced detuning Δω for cavity A as a function of control power.
        Default: linear model (g_XPM * P_ctrl). Physics mode: use delta_omega_xpm().
        """
        if (self.xpm_mode or "").lower() == "physics":
            return float(
                delta_omega_xpm(
                    omega0=self.omega0,
                    T_op=self.T_op,
                    P_ctrl=P_ctrl,
                    n2=self.n2,
                    gXPM=None,  # prefer n2 path for physics mode
                    A_eff=self.A_eff,
                    n_eff=self.n_eff,
                    g_geom=self.g_geom,
                )
            )
        return float(self.g_XPM * P_ctrl)

    # EIT-induced frequency pull on cavity B
    def delta_omega_eit(self, Delta_p, g0, N, Omega_c, Delta_c, gamma_eg, gamma_rg, U_block):
        num = g0**2 * N
        denom_inner = (Delta_p + Delta_c) + 1j * gamma_rg + 1j * U_block / HBAR
        denom = (Delta_p + 1j * gamma_eg) - (abs(Omega_c) ** 2) / denom_inner
        return -num / denom  # complex pull

    def steady_state_response(
        self,
        omega: float,
        P_ctrl: float = 0.0,
        input_port: str = "A",
        eit_params: Optional[Dict] = None,
        clamp_imag_pull: bool = True,
    ) -> Dict[str, float]:
        M = self.transfer_matrix(omega, P_ctrl)
        if eit_params is not None:
            Delta_p = omega - self.omega0
            d_omega = self.delta_omega_eit(Delta_p, **eit_params)
            d_re, d_im = float(np.real(d_omega)), float(np.imag(d_omega))
            if clamp_imag_pull:
                M[1, 1] += 1j * d_re
            else:
                M[1, 1] += 1j * d_re - d_im / 2

        b = (
            np.array([np.sqrt(self.kappa_eA), 0], dtype=complex)
            if input_port == "A"
            else np.array([0, np.sqrt(self.kappa_eB)], dtype=complex)
        )
        try:
            a = np.linalg.solve(M, b)
        except np.linalg.LinAlgError:
            return {
                "T_through": 0.0,
                "T_drop": 0.0,
                "R_reflect": 1.0,
                "phase_through": 0.0,
                "phase_drop": 0.0,
            }

        if input_port == "A":
            s_through = 1 - np.sqrt(self.kappa_eA) * a[0]
            s_drop = np.sqrt(self.kappa_eB) * a[1]
        else:
            s_through = 1 - np.sqrt(self.kappa_eB) * a[1]
            s_drop = np.sqrt(self.kappa_eA) * a[0]
        Tt, Td = float(np.abs(s_through) ** 2), float(np.abs(s_drop) ** 2)
        Tt = max(min(Tt, 1.2), 0.0)  # Allow 20% overshoot for XPM effects
        Td = max(min(Td, 1.2), 0.0)  # Allow 20% overshoot for XPM effects
        return {
            "T_through": Tt,
            "T_drop": Td,
            "R_reflect": float(np.abs(1 - s_through - s_drop) ** 2),
            "phase_through": float(np.angle(s_through)),
            "phase_drop": float(np.angle(s_drop)),
        }


class DeviceCalibrator:
    def __init__(self, device: PhotonicMolecule):
        self.device = device
        self.calibration_data = {}

    def find_resonances(
        self, omega_span: np.ndarray, P_ctrl: float = 0.0
    ) -> Dict[str, List[float]]:
        transmission = [
            self.device.steady_state_response(om, P_ctrl)["T_through"] for om in omega_span
        ]
        transmission = np.array(transmission)
        peaks, _ = find_peaks(-transmission, prominence=0.1)
        if len(peaks) >= 2:
            omega_plus = omega_span[peaks[0]]
            omega_minus = omega_span[peaks[1]]
            splitting = abs(omega_plus - omega_minus)

            def lorentzian(w, w0, gamma, A, offset):
                return offset - A / (1 + ((w - w0) / (gamma / 2)) ** 2)

            Q_factors = []
            for peak in peaks[:2]:
                fit_range = slice(max(0, peak - 20), min(len(omega_span), peak + 20))
                popt, _ = curve_fit(
                    lorentzian,
                    omega_span[fit_range],
                    transmission[fit_range],
                    p0=[omega_span[peak], self.device.kappa_A, 0.5, 0.9],
                )
                w0, gamma = popt[0], popt[1]
                Q_factors.append(w0 / gamma)
            self.calibration_data["resonances"] = {
                "omega_plus": omega_plus,
                "omega_minus": omega_minus,
                "splitting_Hz": splitting / TWOPI,
                "J_extracted_Hz": splitting / TWOPI / 2,
                "Q_factors": Q_factors,
            }
            logger.info(f"Found splitting: {splitting/TWOPI/1e9:.2f} GHz")
        return self.calibration_data.get("resonances", {})

    def calibrate_XPM(self, omega_signal: float, P_ctrl_sweep: np.ndarray) -> float:
        shifts = []
        for P_ctrl in P_ctrl_sweep:
            response = self.device.steady_state_response(omega_signal, P_ctrl)
            shifts.append(response["T_through"])

        def linear_shift(P, g_XPM, offset):
            return offset + g_XPM * P

        popt, _ = curve_fit(linear_shift, P_ctrl_sweep, shifts)
        g_XPM_fitted = popt[0]
        self.calibration_data["g_XPM"] = g_XPM_fitted
        logger.info(f"Calibrated g_XPM: {g_XPM_fitted/TWOPI/1e9*1e-3:.2f} GHz/mW")
        return g_XPM_fitted

    def generate_truth_table(self, omega_signal: float, ctrl_powers: List[float]) -> pd.DataFrame:
        rows = []
        for P_ctrl in ctrl_powers:
            resp = self.device.steady_state_response(omega_signal, P_ctrl)
            rows.append(
                {
                    "P_ctrl_mW": P_ctrl * 1e3,
                    "T_through": resp["T_through"],
                    "T_drop": resp["T_drop"],
                    "phase_deg": np.degrees(resp["phase_through"]),
                }
            )
        df = pd.DataFrame(rows)
        thr = 0.5 * (df.T_through.max() + df.T_through.min())
        df["logic_out"] = (df.T_through > thr).astype(int)
        df.attrs["threshold"] = thr
        df.attrs["contrast_dB"] = 10 * np.log10(df.T_through.max() / df.T_through.min())
        return df


class ExperimentController:
    def __init__(self, device: PhotonicMolecule):
        self.device = device
        self.calibrator = DeviceCalibrator(device)
        self.results = {}

    def run_full_characterization(self) -> Dict:
        omega_span = np.linspace(
            self.device.omega0 - 10 * self.device.J, self.device.omega0 + 10 * self.device.J, 1000
        )
        resonances = self.calibrator.find_resonances(omega_span)
        if resonances:
            omega_signal = resonances["omega_minus"]
            sweep = np.linspace(0, 2e-3, 20)
            g_XPM = self.calibrator.calibrate_XPM(omega_signal, sweep)
            truth = self.calibrator.generate_truth_table(omega_signal, [0, 0.5e-3, 1e-3, 2e-3])
            E90 = self.find_switching_energy(omega_signal, target_contrast=0.9)
            self.results = {
                "resonances": resonances,
                "g_XPM_Hz_per_W": g_XPM / TWOPI,
                "truth_table": truth.to_dict(),
                "E90_pJ": E90 * 1e12,
                "contrast_dB": truth.attrs["contrast_dB"],
            }
        return self.results

    def find_switching_energy(
        self, omega_signal: float, target_contrast: float = 0.9, pulse_duration: float = 10e-9
    ) -> float:
        def contrast_vs_power(P_ctrl):
            on = self.device.steady_state_response(omega_signal, P_ctrl)
            off = self.device.steady_state_response(omega_signal, 0)
            return abs(on["T_through"] - off["T_through"]) / max(off["T_through"], 1e-12)

        P_low, P_high = 0.0, 10e-3
        for _ in range(60):
            P_mid = 0.5 * (P_low + P_high)
            if contrast_vs_power(P_mid) < target_contrast:
                P_low = P_mid
            else:
                P_high = P_mid
        return P_high * pulse_duration

    def test_cascade(
        self, 
        n_stages: int = 2, 
        base_P_ctrl_W: float = 1e-3,
        pulse_duration_s: float = 10e-9,
        threshold_mode: str = "hard", 
        beta: float = 25.0,
        fanout: int = 1,
        split_loss_db: float = 0.5
    ) -> Dict:
        """
        Simulate cascade outputs using full XPM physics with power scaling.
        - threshold_mode: "hard" returns binary logic_out (0/1) using thr=0.5.
                          "soft" returns logic_out_soft in (0,1) via sigmoid.
        - beta: slope parameter for soft threshold.
        - fanout: number of parallel outputs per gate (default=1)
        - split_loss_db: loss per split in dB (default=0.5 dB for realistic couplers)
        """
        results = {}
        
        # Calculate split efficiency for fanout>1
        split_efficiency = 1.0
        if fanout > 1:
            split_efficiency = 10 ** (-split_loss_db / 10)  # Power efficiency per split
    
        # Physics-based power scaling
        n2_reference = 1.5e-17  # AlGaAs baseline
        n2_actual = self.device.n2 if self.device.n2 else n2_reference
        power_scale = n2_reference / n2_actual if n2_actual != 0 else 1.0
    
        effective_P_ctrl_W = base_P_ctrl_W * power_scale
    
        # Truth tables for each logic gate (for reference):
        # AND: [0, 0, 0, 1] - Only true when both inputs are 1
        # OR:  [0, 1, 1, 1] - True when at least one input is 1
        # XOR: [0, 1, 1, 0] - True when inputs are different
    
        for logic in ["AND", "OR", "XOR"]:
            inputs = [(0, 0), (0, 1), (1, 0), (1, 1)]
            outputs = []
            output_details = []
    
            for i, (in1, in2) in enumerate(inputs):
                # Expected outputs for each logic gate:
                # AND[i]: [0, 0, 0, 1][i]
                # OR[i]:  [0, 1, 1, 1][i]
                # XOR[i]: [0, 1, 1, 0][i]
    
                # Simulate photonic implementation with physics-based power scaling
                if logic == "AND":
                    # AND gate: signal passes only if both inputs are high
                    # Use control power to modulate transmission
                    P_ctrl = (
                        in2 * effective_P_ctrl_W
                    )  # Control from second input with scaling
                    signal = float(in1)  # Signal from first input
    
                    # Pass through stages
                    for stage in range(n_stages):
                        resp = self.device.steady_state_response(
                            self.device.omega0, P_ctrl if stage == 0 else 0
                        )
                        # AND behavior: both inputs must be high for signal to pass
                        if in1 == 1 and in2 == 1:
                            signal *= float(resp["T_through"])
                        else:
                            signal *= 0.1  # Attenuate signal when not both high
    
                elif logic == "OR":
                    # OR gate: signal passes if either input is high
                    if in1 == 1 or in2 == 1:
                        signal = 1.0
                        # Use control to maintain high transmission with physics scaling
                        P_ctrl = max(in1, in2) * effective_P_ctrl_W
                        for stage in range(n_stages):
                            resp = self.device.steady_state_response(
                                self.device.omega0, P_ctrl if stage == 0 else 0
                            )
                            signal *= float(resp["T_through"])
                    else:
                        signal = 0.0
                        P_ctrl = 0.0
    
                elif logic == "XOR":
                    # XOR gate: signal passes only if inputs are different
                    if in1 != in2:
                        signal = 1.0
                        # Use differential control with physics scaling
                        P_ctrl = abs(in1 - in2) * effective_P_ctrl_W
                        for stage in range(n_stages):
                            resp = self.device.steady_state_response(
                                self.device.omega0, P_ctrl if stage == 0 else 0
                            )
                            signal *= float(resp["T_through"])
                    else:
                        signal = 0.0
                        P_ctrl = 0.0
    
                # Apply fanout splitting loss
                if fanout > 1:
                    signal *= (split_efficiency ** (fanout - 1))
                
                outputs.append(signal)
                output_details.append({
                    "inputs": (in1, in2), 
                    "P_ctrl": P_ctrl, 
                    "signal": signal,
                    "fanout": fanout
                })
    
            # Calculate fanout-adjusted metrics
            base_energy_fJ = effective_P_ctrl_W * pulse_duration_s * 1e15  # W * s * 1e15 = fJ
            fanout_adjusted_energy_fJ = base_energy_fJ * fanout  # Total energy for parallel ops
            
            # Calculate effective cascade depth with fanout
            if fanout > 1:
                # Depth reduction approximation: depth ~ original_depth / sqrt(fanout)
                effective_depth = max(1, int(n_stages / np.sqrt(fanout)))
            else:
                effective_depth = n_stages
    
            # Apply thresholding
            thr = 0.5
            if (threshold_mode or "").lower() == "soft":
                logic_soft = [float(soft_logic(o, thr, beta)) for o in outputs]
                results[logic] = {
                    "outputs": outputs,
                    "logic_out_soft": logic_soft,
                    "min_contrast_dB": 10 * np.log10(max(outputs) / max(min(outputs), 1e-12)),
                    "power_scale_factor": power_scale,
                    "effective_P_ctrl_mW": effective_P_ctrl_W * 1e3,
                    "details": output_details,
                    "fanout": fanout,
                    "split_loss_db": split_loss_db,
                    "split_efficiency": split_efficiency,
                    "base_energy_fJ": base_energy_fJ,
                    "fanout_adjusted_energy_fJ": fanout_adjusted_energy_fJ,
                    "effective_cascade_depth": effective_depth,
                }
            else:
                logic_out = [1 if o > thr else 0 for o in outputs]
                results[logic] = {
                    "outputs": outputs,
                    "logic_out": logic_out,
                    "min_contrast_dB": 10 * np.log10(max(outputs) / max(min(outputs), 1e-12)),
                    "power_scale_factor": power_scale,
                    "effective_P_ctrl_mW": effective_P_ctrl_W * 1e3,
                    "details": output_details,
                    "fanout": fanout,
                    "split_loss_db": split_loss_db,
                    "split_efficiency": split_efficiency,
                    "base_energy_fJ": base_energy_fJ,
                    "fanout_adjusted_energy_fJ": fanout_adjusted_energy_fJ,
                    "effective_cascade_depth": effective_depth,
                }
        return results


def generate_design_report(
    device: PhotonicMolecule, results: Dict, filename: str = "photonic_logic_report.json"
):
    report = {
        "timestamp": time.strftime("%Y-%m-%d %H:%M:%S"),
        "device_parameters": {
            "Q_factor": float(device.omega0 / device.kappa_A),
            "J_Hz": float(device.J / TWOPI),
            "kappa_Hz": float(device.kappa_A / TWOPI),
            "g_XPM_Hz_per_W": float(device.g_XPM / TWOPI),
            "wavelength_nm": C / (device.omega0 / TWOPI) * 1e9,
        },
        "performance_metrics": results,
        "design_criteria_met": {
            "splitting_resolved": results.get("resonances", {}).get("splitting_Hz", 0)
            > device.kappa_A / TWOPI,
            "XPM_sufficient": results.get("E90_pJ", float("inf")) < 5000,
            "contrast_adequate": results.get("contrast_dB", 0) > 15,
        },
    }
    with open(filename, "w") as f:
        json.dump(report, f, indent=2)
    logger.info(f"Report saved to {filename}")
    return report


# Optional: time-domain simulate helper
def simulate_pulse(device: PhotonicMolecule, omega: float, P_ctrl_t, t_span=(0, 10e-9), y0=None):
    def rhs(t, y):
        P_ctrl = P_ctrl_t(t)
        M = device.transfer_matrix(omega, P_ctrl)
        vec = M @ (y[0::2] + 1j * y[1::2]) + np.array([np.sqrt(device.kappa_eA), 0], complex)
        return np.array([vec[0].real, vec[0].imag, vec[1].real, vec[1].imag])

    y0 = np.zeros(4) if y0 is None else y0
    sol = solve_ivp(rhs, t_span, y0, max_step=1e-11)
    aA = sol.y[0] + 1j * sol.y[1]
    s_through = 1 - np.sqrt(device.kappa_eA) * aA
    return sol.t, np.abs(s_through) ** 2
