"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkDeps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver = require("semver");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Manages dependencies on the AWS CDK.
 */
class AwsCdkDeps extends component_1.Component {
    constructor(project, options) {
        var _b, _c, _d;
        super(project);
        this.cdkDependenciesAsDeps = (_b = options.cdkDependenciesAsDeps) !== null && _b !== void 0 ? _b : true;
        this.dependencyType = options.dependencyType;
        this._packageNames = this.packageNames();
        const framework = determineFrameworkVersion(options);
        this.cdkVersion = framework.range;
        this.cdkMajorVersion = framework.major;
        this.cdkMinimumVersion = framework.minimum;
        this.addFrameworkDependency(options);
        // assert/assertions library
        this.addV1AssertionLibraryDependency(options);
        // constructs library
        this.addConstructsDependency(options.constructsVersion);
        // user-defined v1 dependencies (will only fail in CDK v2 if these have values)
        this.addV1Dependencies(...((_c = options.cdkDependencies) !== null && _c !== void 0 ? _c : []));
        this.addV1DevDependencies(...((_d = options.cdkTestDependencies) !== null && _d !== void 0 ? _d : []));
    }
    /**
     * Adds dependencies to AWS CDK modules.
     *
     * The type of dependency is determined by the `dependencyType` option.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     */
    addV1Dependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error("addV1Dependencies() is not supported for CDK 2.x and above, use addDeps() or addPeerDeps() instead");
        }
        // this will add dependencies based on the type requested by the user
        // for libraries, this will be "peer" and for apps it will be "runtime"
        this.addV1DependenciesByType(this.dependencyType, ...deps);
        // add deps as runtime deps if `cdkDepsAsDeps` is true
        if (this.cdkDependenciesAsDeps) {
            this.addV1DependenciesByType(dependencies_1.DependencyType.RUNTIME, ...deps);
        }
    }
    /**
     * Adds AWS CDK modules as dev dependencies.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps fully qualified names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     */
    addV1DevDependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error("addV1DevDependencies() is not supported for CDK 2.x and above, use addDevDeps()/addTestDeps() instead");
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.BUILD, ...deps);
    }
    addConstructsDependency(requestedVersion) {
        var _b;
        if (requestedVersion && !semver.parse(requestedVersion)) {
            throw new Error(`"constructsVersion" cannot be parsed as a semver version: ${requestedVersion}`);
        }
        const defaultVersion = this.cdkMajorVersion === 1 ? "3.2.27" : "10.0.5";
        const versionRequirement = `^${requestedVersion !== null && requestedVersion !== void 0 ? requestedVersion : defaultVersion}`;
        const constructsMajorVersion = (_b = semver.minVersion(versionRequirement)) === null || _b === void 0 ? void 0 : _b.major;
        if (!constructsMajorVersion) {
            throw new Error(`Cannot determine major version of constructs version '${versionRequirement}'`);
        }
        switch (this.cdkMajorVersion) {
            case 1:
                if (constructsMajorVersion !== 3) {
                    throw new Error("AWS CDK 1.x requires constructs 3.x");
                }
                break;
            case 2:
                if (constructsMajorVersion !== 10) {
                    throw new Error("AWS CDK 2.x requires constructs 10.x");
                }
                break;
        }
        this.project.deps.addDependency(`${this._packageNames.constructs}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    /**
     * Adds a dependency on the AWS CDK framework (e.g. `@aws-cdk/core` for V1 or `aws-cdk-lib` for V1).
     */
    addFrameworkDependency(options) {
        switch (this.cdkMajorVersion) {
            case 1:
                this.addV1Dependencies(this._packageNames.coreV1);
                break;
            case 2:
                if (options.cdkDependencies !== undefined) {
                    throw new Error('cdkDependencies is not used for CDK 2.x. Use "peerDeps" or "deps" instead');
                }
                if (options.cdkDependenciesAsDeps !== undefined) {
                    throw new Error("cdkDependenciesAsDeps is not used for CDK 2.x");
                }
                if (options.cdkTestDependencies !== undefined) {
                    throw new Error('cdkTestDependencies is not used for CDK 2.x. Use "devDeps" or "testDeps" instead');
                }
                this.project.deps.addDependency(`${this._packageNames.coreV2}@${this.cdkVersion}`, this.dependencyType);
                break;
            default:
                throw new Error(`Unsupported AWS CDK major version ${this.cdkMajorVersion}.x`);
        }
    }
    addV1AssertionLibraryDependency(options) {
        var _b, _c;
        if (this.cdkMajorVersion !== 1) {
            if (options.cdkAssert !== undefined) {
                throw new Error("cdkAssert is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib");
            }
            if (options.cdkAssertions !== undefined) {
                throw new Error("cdkAssertion is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib");
            }
            return;
        }
        const testDeps = new Array();
        if (((_b = options.cdkAssert) !== null && _b !== void 0 ? _b : true) && this._packageNames.assert) {
            testDeps.push(this._packageNames.assert);
        }
        // @aws-cdk/assertions is only available starting v1.111.0
        if (semver.gte(this.cdkMinimumVersion, "1.111.0") &&
            ((_c = options.cdkAssertions) !== null && _c !== void 0 ? _c : true)) {
            testDeps.push(this._packageNames.assertions);
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.TEST, ...testDeps);
    }
    /**
     * Adds a set of dependencies with the user-specified dependency type.
     * @param deps The set of dependency specifications
     */
    addV1DependenciesByType(type, ...modules) {
        for (const module of modules) {
            this.project.deps.addDependency(`${module}@${this.cdkVersion}`, type);
        }
    }
}
exports.AwsCdkDeps = AwsCdkDeps;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkDeps[_a] = { fqn: "projen.awscdk.AwsCdkDeps", version: "0.52.53" };
function determineFrameworkVersion(options) {
    const ver = semver.parse(options.cdkVersion);
    if (!ver) {
        throw new Error(`"cdkVersion" cannot be parsed as a semver version: ${options.cdkVersion}`);
    }
    return {
        minimum: ver.format(),
        range: options.cdkVersionPinning
            ? options.cdkVersion
            : `^${options.cdkVersion}`,
        major: ver.major,
    };
}
//# sourceMappingURL=data:application/json;base64,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