import { Component } from "../component";
import { BundlingOptions } from "../javascript";
import { Project } from "../project";
import { AwsCdkDeps } from "./awscdk-deps";
/**
 * Common options for `LambdaFunction`. Applies to all functions in
 * auto-discovery.
 */
export interface LambdaFunctionCommonOptions {
    /**
     * The node.js version to target.
     *
     * @default Runtime.NODEJS_14_X
     */
    readonly runtime?: LambdaRuntime;
    /**
     * Bundling options for this AWS Lambda function.
     *
     * If not specified the default bundling options specified for the project
     * `Bundler` instance will be used.
     *
     * @default - defaults
     */
    readonly bundlingOptions?: BundlingOptions;
    /**
     * Whether to automatically reuse TCP connections when working with the AWS
     * SDK for JavaScript.
     *
     * This sets the `AWS_NODEJS_CONNECTION_REUSE_ENABLED` environment variable
     * to `1`.
     *
     * @see https://docs.aws.amazon.com/sdk-for-javascript/v2/developer-guide/node-reusing-connections.html
     *
     * @default true
     */
    readonly awsSdkConnectionReuse?: boolean;
}
/**
 * Options for `Function`.
 */
export interface LambdaFunctionOptions extends LambdaFunctionCommonOptions {
    /**
     * A path from the project root directory to a TypeScript file which contains
     * the AWS Lambda handler entrypoint (exports a `handler` function).
     *
     * This is relative to the root directory of the project.
     *
     * @example "src/subdir/foo.lambda.ts"
     */
    readonly entrypoint: string;
    /**
     * The name of the generated TypeScript source file. This file should also be
     * under the source tree.
     *
     * @default - The name of the entrypoint file, with the `-function.ts` suffix
     * instead of `.lambda.ts`.
     */
    readonly constructFile?: string;
    /**
     * The name of the generated `lambda.Function` subclass.
     *
     * @default - A pascal cased version of the name of the entrypoint file, with
     * the extension `Function` (e.g. `ResizeImageFunction`).
     */
    readonly constructName?: string;
    /**
     * AWS CDK dependency manager.
     */
    readonly cdkDeps: AwsCdkDeps;
}
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
export declare class LambdaFunction extends Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project: Project, options: LambdaFunctionOptions);
}
/**
 * The runtime for the AWS Lambda function.
 */
export declare class LambdaRuntime {
    /**
     * The aws-lambda.Runtime member name to use.
     */
    readonly functionRuntime: string;
    /**
     * The esbuild setting to use.
     */
    readonly esbuildTarget: string;
    /**
     * Node.js 10.x
     */
    static readonly NODEJS_10_X: LambdaRuntime;
    /**
     * Node.js 12.x
     */
    static readonly NODEJS_12_X: LambdaRuntime;
    /**
     * Node.js 14.x
     */
    static readonly NODEJS_14_X: LambdaRuntime;
    readonly esbuildPlatform = "node";
    private constructor();
}
