"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        var _c, _d, _e, _f;
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = (_c = options.runtime) !== null && _c !== void 0 ? _c : LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint === null || eslint === void 0 ? void 0 : eslint.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} extension`);
        }
        const basePath = path_1.join(path_1.dirname(entrypoint), path_1.basename(entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = (_d = options.constructFile) !== null && _d !== void 0 ? _d : `${basePath}-function.ts`;
        if (path_1.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = (_e = options.constructName) !== null && _e !== void 0 ? _e : case_1.pascal(path_1.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ["aws-sdk"],
            ...options.bundlingOptions,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path_1.join(project.outdir, bundle.outfile);
        const constructAbs = path_1.join(project.outdir, constructFile);
        const relativeOutfile = path_1.relative(path_1.dirname(constructAbs), path_1.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${convertToPosixPath(basePath)}.`);
        src.line(" */");
        src.open(`export class ${constructName} extends lambda.Function {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: lambda.Runtime.${runtime.functionRuntime},`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((_f = options.awsSdkConnectionReuse) !== null && _f !== void 0 ? _f : true) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.52.53" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The aws-lambda.Runtime member name to use.
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.52.53" };
/**
 * Node.js 10.x
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("NODEJS_10_X", "node10");
/**
 * Node.js 12.x
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("NODEJS_12_X", "node12");
/**
 * Node.js 14.x
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("NODEJS_14_X", "node14");
/**
 * Converts the given path string to posix if it wasn't already.
 */
function convertToPosixPath(p) {
    return p.split(path_1.sep).join(path_1.posix.sep);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLWZ1bmN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2F3c2Nkay9sYW1iZGEtZnVuY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwrQkFBOEU7QUFDOUUsK0JBQThCO0FBQzlCLDRDQUF5QztBQUN6Qyw4Q0FBaUU7QUFFakUsZ0RBQTRDO0FBRTVDLHlDQUFtRDtBQTJFbkQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FtQkc7QUFDSCxNQUFhLGNBQWUsU0FBUSxxQkFBUztJQUMzQzs7Ozs7T0FLRztJQUNILFlBQVksT0FBZ0IsRUFBRSxPQUE4Qjs7UUFDMUQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWYsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUNoQyxNQUFNLE9BQU8sR0FBRyxvQkFBTyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ1osTUFBTSxJQUFJLEtBQUssQ0FDYixtRUFBbUUsQ0FDcEUsQ0FBQztTQUNIO1FBRUQsTUFBTSxPQUFPLFNBQUcsT0FBTyxDQUFDLE9BQU8sbUNBQUksYUFBYSxDQUFDLFdBQVcsQ0FBQztRQUU3RCwwRUFBMEU7UUFDMUUsa0JBQWtCO1FBQ2xCLE1BQU0sTUFBTSxHQUFHLG1CQUFNLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2xDLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxZQUFZLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRTtRQUV6QyxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO1FBRXRDLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLGdDQUFxQixDQUFDLEVBQUU7WUFDL0MsTUFBTSxJQUFJLEtBQUssQ0FDYixHQUFHLFVBQVUsZ0JBQWdCLGdDQUFxQixZQUFZLENBQy9ELENBQUM7U0FDSDtRQUVELE1BQU0sUUFBUSxHQUFHLFdBQUksQ0FDbkIsY0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUNuQixlQUFRLENBQUMsVUFBVSxFQUFFLGdDQUFxQixDQUFDLENBQzVDLENBQUM7UUFDRixNQUFNLGFBQWEsU0FBRyxPQUFPLENBQUMsYUFBYSxtQ0FBSSxHQUFHLFFBQVEsY0FBYyxDQUFDO1FBRXpFLElBQUksY0FBTyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEtBQUssRUFBRTtZQUNwQyxNQUFNLElBQUksS0FBSyxDQUNiLHdCQUF3QixhQUFhLDZCQUE2QixDQUNuRSxDQUFDO1NBQ0g7UUFFRCxhQUFhO1FBQ2IsTUFBTSxhQUFhLFNBQ2pCLE9BQU8sQ0FBQyxhQUFhLG1DQUFJLGFBQU0sQ0FBQyxlQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxVQUFVLENBQUM7UUFDbkUsTUFBTSxTQUFTLEdBQUcsR0FBRyxhQUFhLE9BQU8sQ0FBQztRQUUxQyxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRTtZQUMzQyxNQUFNLEVBQUUsT0FBTyxDQUFDLGFBQWE7WUFDN0IsUUFBUSxFQUFFLE9BQU8sQ0FBQyxlQUFlO1lBQ2pDLFNBQVMsRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUN0QixHQUFHLE9BQU8sQ0FBQyxlQUFlO1NBQzNCLENBQUMsQ0FBQztRQUVILG1FQUFtRTtRQUNuRSx5RUFBeUU7UUFDekUsc0RBQXNEO1FBQ3RELE9BQU87UUFDUCw4RUFBOEU7UUFDOUUsdUVBQXVFO1FBQ3ZFLE1BQU0sVUFBVSxHQUFHLFdBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN4RCxNQUFNLFlBQVksR0FBRyxXQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQztRQUN6RCxNQUFNLGVBQWUsR0FBRyxlQUFRLENBQzlCLGNBQU8sQ0FBQyxZQUFZLENBQUMsRUFDckIsY0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUNwQixDQUFDO1FBRUYsTUFBTSxHQUFHLEdBQUcsSUFBSSx3QkFBVSxDQUFDLE9BQU8sRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNuRCxJQUFJLEdBQUcsQ0FBQyxNQUFNLEVBQUU7WUFDZCxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDOUI7UUFDRCxHQUFHLENBQUMsSUFBSSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFFMUMsSUFBSSxPQUFPLENBQUMsZUFBZSxLQUFLLENBQUMsRUFBRTtZQUNqQyxHQUFHLENBQUMsSUFBSSxDQUFDLGdEQUFnRCxDQUFDLENBQUM7WUFDM0QsR0FBRyxDQUFDLElBQUksQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1lBQ3ZELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1lBQ2pELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUM1QzthQUFNO1lBQ0wsR0FBRyxDQUFDLElBQUksQ0FBQyxtREFBbUQsQ0FBQyxDQUFDO1lBQzlELEdBQUcsQ0FBQyxJQUFJLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUNyRDtRQUVELEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNYLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEIsR0FBRyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsYUFBYSxFQUFFLENBQUMsQ0FBQztRQUMxQyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hCLEdBQUcsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLFNBQVMsbUNBQW1DLENBQUMsQ0FBQztRQUMzRSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2YsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ1gsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQixHQUFHLENBQUMsSUFBSSxDQUNOLDRDQUE0QyxrQkFBa0IsQ0FDNUQsUUFBUSxDQUNULEdBQUcsQ0FDTCxDQUFDO1FBQ0YsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQixHQUFHLENBQUMsSUFBSSxDQUFDLGdCQUFnQixhQUFhLDRCQUE0QixDQUFDLENBQUM7UUFDcEUsR0FBRyxDQUFDLElBQUksQ0FDTixxREFBcUQsU0FBUyxLQUFLLENBQ3BFLENBQUM7UUFDRixHQUFHLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsa0JBQWtCLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzlELEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDdEIsR0FBRyxDQUFDLElBQUksQ0FBQywyQkFBMkIsT0FBTyxDQUFDLGVBQWUsR0FBRyxDQUFDLENBQUM7UUFDaEUsR0FBRyxDQUFDLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLEdBQUcsQ0FBQyxJQUFJLENBQ04scURBQXFELGtCQUFrQixDQUNyRSxlQUFlLENBQ2hCLE1BQU0sQ0FDUixDQUFDO1FBQ0YsR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNqQixVQUFJLE9BQU8sQ0FBQyxxQkFBcUIsbUNBQUksSUFBSSxFQUFFO1lBQ3pDLEdBQUcsQ0FBQyxJQUFJLENBQ04sMEZBQTBGLENBQzNGLENBQUM7U0FDSDtRQUNELEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDZixHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWYsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUN6QixHQUFHLFFBQVEsZ0JBQWdCLGFBQWEsc0JBQXNCLGFBQWEsR0FBRyxDQUMvRSxDQUFDO1FBQ0YsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUN6QixHQUFHLFFBQVEsa0JBQWtCLE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBSSxHQUFHLENBQ3ZELENBQUM7UUFDRixJQUFJLE1BQU0sQ0FBQyxTQUFTLEVBQUU7WUFDcEIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUN6QixHQUFHLFFBQVEsd0JBQXdCLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQzVELENBQUM7U0FDSDtJQUNILENBQUM7O0FBdElILHdDQXVJQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGFBQWE7SUEyQnhCO0lBQ0U7O09BRUc7SUFDYSxlQUF1QjtJQUV2Qzs7T0FFRztJQUNhLGFBQXFCO1FBTHJCLG9CQUFlLEdBQWYsZUFBZSxDQUFRO1FBS3ZCLGtCQUFhLEdBQWIsYUFBYSxDQUFRO1FBWHZCLG9CQUFlLEdBQUcsTUFBTSxDQUFDO0lBWXRDLENBQUM7O0FBckNOLHNDQXNDQzs7O0FBckNDOztHQUVHO0FBQ29CLHlCQUFXLEdBQUcsSUFBSSxhQUFhLENBQ3BELGFBQWEsRUFDYixRQUFRLENBQ1QsQ0FBQztBQUVGOztHQUVHO0FBQ29CLHlCQUFXLEdBQUcsSUFBSSxhQUFhLENBQ3BELGFBQWEsRUFDYixRQUFRLENBQ1QsQ0FBQztBQUVGOztHQUVHO0FBQ29CLHlCQUFXLEdBQUcsSUFBSSxhQUFhLENBQ3BELGFBQWEsRUFDYixRQUFRLENBQ1QsQ0FBQztBQWlCSjs7R0FFRztBQUNILFNBQVMsa0JBQWtCLENBQUMsQ0FBUztJQUNuQyxPQUFPLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBRyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUN0QyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgYmFzZW5hbWUsIGRpcm5hbWUsIGV4dG5hbWUsIGpvaW4sIHJlbGF0aXZlLCBzZXAsIHBvc2l4IH0gZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IHBhc2NhbCB9IGZyb20gXCJjYXNlXCI7XG5pbXBvcnQgeyBDb21wb25lbnQgfSBmcm9tIFwiLi4vY29tcG9uZW50XCI7XG5pbXBvcnQgeyBCdW5kbGVyLCBCdW5kbGluZ09wdGlvbnMsIEVzbGludCB9IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcbmltcG9ydCB7IFNvdXJjZUNvZGUgfSBmcm9tIFwiLi4vc291cmNlLWNvZGVcIjtcbmltcG9ydCB7IEF3c0Nka0RlcHMgfSBmcm9tIFwiLi9hd3NjZGstZGVwc1wiO1xuaW1wb3J0IHsgVFlQRVNDUklQVF9MQU1CREFfRVhUIH0gZnJvbSBcIi4vaW50ZXJuYWxcIjtcblxuLyoqXG4gKiBDb21tb24gb3B0aW9ucyBmb3IgYExhbWJkYUZ1bmN0aW9uYC4gQXBwbGllcyB0byBhbGwgZnVuY3Rpb25zIGluXG4gKiBhdXRvLWRpc2NvdmVyeS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFGdW5jdGlvbkNvbW1vbk9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIG5vZGUuanMgdmVyc2lvbiB0byB0YXJnZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJ1bnRpbWUuTk9ERUpTXzE0X1hcbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU/OiBMYW1iZGFSdW50aW1lO1xuXG4gIC8qKlxuICAgKiBCdW5kbGluZyBvcHRpb25zIGZvciB0aGlzIEFXUyBMYW1iZGEgZnVuY3Rpb24uXG4gICAqXG4gICAqIElmIG5vdCBzcGVjaWZpZWQgdGhlIGRlZmF1bHQgYnVuZGxpbmcgb3B0aW9ucyBzcGVjaWZpZWQgZm9yIHRoZSBwcm9qZWN0XG4gICAqIGBCdW5kbGVyYCBpbnN0YW5jZSB3aWxsIGJlIHVzZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdHNcbiAgICovXG4gIHJlYWRvbmx5IGJ1bmRsaW5nT3B0aW9ucz86IEJ1bmRsaW5nT3B0aW9ucztcblxuICAvKipcbiAgICogV2hldGhlciB0byBhdXRvbWF0aWNhbGx5IHJldXNlIFRDUCBjb25uZWN0aW9ucyB3aGVuIHdvcmtpbmcgd2l0aCB0aGUgQVdTXG4gICAqIFNESyBmb3IgSmF2YVNjcmlwdC5cbiAgICpcbiAgICogVGhpcyBzZXRzIHRoZSBgQVdTX05PREVKU19DT05ORUNUSU9OX1JFVVNFX0VOQUJMRURgIGVudmlyb25tZW50IHZhcmlhYmxlXG4gICAqIHRvIGAxYC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2RrLWZvci1qYXZhc2NyaXB0L3YyL2RldmVsb3Blci1ndWlkZS9ub2RlLXJldXNpbmctY29ubmVjdGlvbnMuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBhd3NTZGtDb25uZWN0aW9uUmV1c2U/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBGdW5jdGlvbmAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhRnVuY3Rpb25PcHRpb25zIGV4dGVuZHMgTGFtYmRhRnVuY3Rpb25Db21tb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIEEgcGF0aCBmcm9tIHRoZSBwcm9qZWN0IHJvb3QgZGlyZWN0b3J5IHRvIGEgVHlwZVNjcmlwdCBmaWxlIHdoaWNoIGNvbnRhaW5zXG4gICAqIHRoZSBBV1MgTGFtYmRhIGhhbmRsZXIgZW50cnlwb2ludCAoZXhwb3J0cyBhIGBoYW5kbGVyYCBmdW5jdGlvbikuXG4gICAqXG4gICAqIFRoaXMgaXMgcmVsYXRpdmUgdG8gdGhlIHJvb3QgZGlyZWN0b3J5IG9mIHRoZSBwcm9qZWN0LlxuICAgKlxuICAgKiBAZXhhbXBsZSBcInNyYy9zdWJkaXIvZm9vLmxhbWJkYS50c1wiXG4gICAqL1xuICByZWFkb25seSBlbnRyeXBvaW50OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBnZW5lcmF0ZWQgVHlwZVNjcmlwdCBzb3VyY2UgZmlsZS4gVGhpcyBmaWxlIHNob3VsZCBhbHNvIGJlXG4gICAqIHVuZGVyIHRoZSBzb3VyY2UgdHJlZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgbmFtZSBvZiB0aGUgZW50cnlwb2ludCBmaWxlLCB3aXRoIHRoZSBgLWZ1bmN0aW9uLnRzYCBzdWZmaXhcbiAgICogaW5zdGVhZCBvZiBgLmxhbWJkYS50c2AuXG4gICAqL1xuICByZWFkb25seSBjb25zdHJ1Y3RGaWxlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZ2VuZXJhdGVkIGBsYW1iZGEuRnVuY3Rpb25gIHN1YmNsYXNzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgcGFzY2FsIGNhc2VkIHZlcnNpb24gb2YgdGhlIG5hbWUgb2YgdGhlIGVudHJ5cG9pbnQgZmlsZSwgd2l0aFxuICAgKiB0aGUgZXh0ZW5zaW9uIGBGdW5jdGlvbmAgKGUuZy4gYFJlc2l6ZUltYWdlRnVuY3Rpb25gKS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnN0cnVjdE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFXUyBDREsgZGVwZW5kZW5jeSBtYW5hZ2VyLlxuICAgKi9cbiAgcmVhZG9ubHkgY2RrRGVwczogQXdzQ2RrRGVwcztcbn1cblxuLyoqXG4gKiBHZW5lcmF0ZXMgYSBwcmUtYnVuZGxlZCBBV1MgTGFtYmRhIGZ1bmN0aW9uIGNvbnN0cnVjdCBmcm9tIGhhbmRsZXIgY29kZS5cbiAqXG4gKiBUbyB1c2UgdGhpcywgY3JlYXRlIGFuIEFXUyBMYW1iZGEgaGFuZGxlciBmaWxlIHVuZGVyIHlvdXIgc291cmNlIHRyZWUgd2l0aFxuICogdGhlIGAubGFtYmRhLnRzYCBleHRlbnNpb24gYW5kIGFkZCBhIGBMYW1iZGFGdW5jdGlvbmAgY29tcG9uZW50IHRvIHlvdXJcbiAqIHR5cGVzY3JpcHQgcHJvamVjdCBwb2ludGluZyB0byB0aGlzIGVudHJ5cG9pbnQuXG4gKlxuICogVGhpcyB3aWxsIGFkZCBhIHRhc2sgdG8geW91ciBcImNvbXBpbGVcIiBzdGVwIHdoaWNoIHdpbGwgdXNlIGBlc2J1aWxkYCB0b1xuICogYnVuZGxlIHRoZSBoYW5kbGVyIGNvZGUgaW50byB0aGUgYnVpbGQgZGlyZWN0b3J5LiBJdCB3aWxsIGFsc28gZ2VuZXJhdGUgYVxuICogZmlsZSBgc3JjL2Zvby1mdW5jdGlvbi50c2Agd2l0aCBhIGN1c3RvbSBBV1MgY29uc3RydWN0IGNhbGxlZCBgRm9vRnVuY3Rpb25gXG4gKiB3aGljaCBleHRlbmRzIGBAYXdzLWNkay9hd3MtbGFtYmRhLkZ1bmN0aW9uYCB3aGljaCBpcyBib3VuZCB0byB0aGUgYnVuZGxlZFxuICogaGFuZGxlIHRocm91Z2ggYW4gYXNzZXQuXG4gKlxuICogQGV4YW1wbGVcbiAqXG4gKiBuZXcgTGFtYmRhRnVuY3Rpb24obXlQcm9qZWN0LCB7XG4gKiAgIHNyY2RpcjogbXlQcm9qZWN0LnNyY2RpcixcbiAqICAgZW50cnlwb2ludDogJ3NyYy9mb28ubGFtYmRhLnRzJyxcbiAqIH0pO1xuICovXG5leHBvcnQgY2xhc3MgTGFtYmRhRnVuY3Rpb24gZXh0ZW5kcyBDb21wb25lbnQge1xuICAvKipcbiAgICogRGVmaW5lcyBhIHByZS1idW5kbGVkIEFXUyBMYW1iZGEgZnVuY3Rpb24gY29uc3RydWN0IGZyb20gaGFuZGxlciBjb2RlLlxuICAgKlxuICAgKiBAcGFyYW0gcHJvamVjdCBUaGUgcHJvamVjdCB0byB1c2VcbiAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9uc1xuICAgKi9cbiAgY29uc3RydWN0b3IocHJvamVjdDogUHJvamVjdCwgb3B0aW9uczogTGFtYmRhRnVuY3Rpb25PcHRpb25zKSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG5cbiAgICBjb25zdCBjZGtEZXBzID0gb3B0aW9ucy5jZGtEZXBzO1xuICAgIGNvbnN0IGJ1bmRsZXIgPSBCdW5kbGVyLm9mKHByb2plY3QpO1xuICAgIGlmICghYnVuZGxlcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIk5vIGJ1bmRsZXIgZm91bmQuIFBsZWFzZSBhZGQgYSBCdW5kbGVyIGNvbXBvbmVudCB0byB5b3VyIHByb2plY3QuXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgcnVudGltZSA9IG9wdGlvbnMucnVudGltZSA/PyBMYW1iZGFSdW50aW1lLk5PREVKU18xNF9YO1xuXG4gICAgLy8gYWxsb3cgTGFtYmRhIGhhbmRsZXIgY29kZSB0byBpbXBvcnQgZGV2LWRlcHMgc2luY2UgdGhleSBhcmUgb25seSBuZWVkZWRcbiAgICAvLyBkdXJpbmcgYnVuZGxpbmdcbiAgICBjb25zdCBlc2xpbnQgPSBFc2xpbnQub2YocHJvamVjdCk7XG4gICAgZXNsaW50Py5hbGxvd0RldkRlcHMob3B0aW9ucy5lbnRyeXBvaW50KTtcblxuICAgIGNvbnN0IGVudHJ5cG9pbnQgPSBvcHRpb25zLmVudHJ5cG9pbnQ7XG5cbiAgICBpZiAoIWVudHJ5cG9pbnQuZW5kc1dpdGgoVFlQRVNDUklQVF9MQU1CREFfRVhUKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgJHtlbnRyeXBvaW50fSBtdXN0IGhhdmUgYSAke1RZUEVTQ1JJUFRfTEFNQkRBX0VYVH0gZXh0ZW5zaW9uYFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBiYXNlUGF0aCA9IGpvaW4oXG4gICAgICBkaXJuYW1lKGVudHJ5cG9pbnQpLFxuICAgICAgYmFzZW5hbWUoZW50cnlwb2ludCwgVFlQRVNDUklQVF9MQU1CREFfRVhUKVxuICAgICk7XG4gICAgY29uc3QgY29uc3RydWN0RmlsZSA9IG9wdGlvbnMuY29uc3RydWN0RmlsZSA/PyBgJHtiYXNlUGF0aH0tZnVuY3Rpb24udHNgO1xuXG4gICAgaWYgKGV4dG5hbWUoY29uc3RydWN0RmlsZSkgIT09IFwiLnRzXCIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYENvbnN0cnVjdCBmaWxlIG5hbWUgXCIke2NvbnN0cnVjdEZpbGV9XCIgbXVzdCBoYXZlIGEgLnRzIGV4dGVuc2lvbmBcbiAgICAgICk7XG4gICAgfVxuXG4gICAgLy8gdHlwZSBuYW1lc1xuICAgIGNvbnN0IGNvbnN0cnVjdE5hbWUgPVxuICAgICAgb3B0aW9ucy5jb25zdHJ1Y3ROYW1lID8/IHBhc2NhbChiYXNlbmFtZShiYXNlUGF0aCkpICsgXCJGdW5jdGlvblwiO1xuICAgIGNvbnN0IHByb3BzVHlwZSA9IGAke2NvbnN0cnVjdE5hbWV9UHJvcHNgO1xuXG4gICAgY29uc3QgYnVuZGxlID0gYnVuZGxlci5hZGRCdW5kbGUoZW50cnlwb2ludCwge1xuICAgICAgdGFyZ2V0OiBydW50aW1lLmVzYnVpbGRUYXJnZXQsXG4gICAgICBwbGF0Zm9ybTogcnVudGltZS5lc2J1aWxkUGxhdGZvcm0sXG4gICAgICBleHRlcm5hbHM6IFtcImF3cy1zZGtcIl0sXG4gICAgICAuLi5vcHRpb25zLmJ1bmRsaW5nT3B0aW9ucyxcbiAgICB9KTtcblxuICAgIC8vIGNhbGN1bGF0ZSB0aGUgcmVsYXRpdmUgcGF0aCBiZXR3ZWVuIHRoZSBkaXJlY3RvcnkgY29udGFpbmluZyB0aGVcbiAgICAvLyBnZW5lcmF0ZWQgY29uc3RydWN0IHNvdXJjZSBmaWxlIHRvIHRoZSBkaXJlY3RvcnkgY29udGFpbmluZyB0aGUgYnVuZGxlXG4gICAgLy8gaW5kZXguanMgYnkgcmVzb2x2aW5nIHRoZW0gYXMgYWJzb2x1dGUgcGF0aHMgZmlyc3QuXG4gICAgLy8gZS5nOlxuICAgIC8vICAtIG91dGZpbGVBYnMgPT4gYC9wcm9qZWN0LW91dGRpci9hc3NldHMvZm9vL2Jhci9iYXovZm9vLWZ1bmN0aW9uL2luZGV4LmpzYFxuICAgIC8vICAtIGNvbnN0cnVjdEFicyA9PiBgL3Byb2plY3Qtb3V0ZGlyL3NyYy9mb28vYmFyL2Jhei9mb28tZnVuY3Rpb24udHNgXG4gICAgY29uc3Qgb3V0ZmlsZUFicyA9IGpvaW4ocHJvamVjdC5vdXRkaXIsIGJ1bmRsZS5vdXRmaWxlKTtcbiAgICBjb25zdCBjb25zdHJ1Y3RBYnMgPSBqb2luKHByb2plY3Qub3V0ZGlyLCBjb25zdHJ1Y3RGaWxlKTtcbiAgICBjb25zdCByZWxhdGl2ZU91dGZpbGUgPSByZWxhdGl2ZShcbiAgICAgIGRpcm5hbWUoY29uc3RydWN0QWJzKSxcbiAgICAgIGRpcm5hbWUob3V0ZmlsZUFicylcbiAgICApO1xuXG4gICAgY29uc3Qgc3JjID0gbmV3IFNvdXJjZUNvZGUocHJvamVjdCwgY29uc3RydWN0RmlsZSk7XG4gICAgaWYgKHNyYy5tYXJrZXIpIHtcbiAgICAgIHNyYy5saW5lKGAvLyAke3NyYy5tYXJrZXJ9YCk7XG4gICAgfVxuICAgIHNyYy5saW5lKFwiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcIik7XG5cbiAgICBpZiAoY2RrRGVwcy5jZGtNYWpvclZlcnNpb24gPT09IDEpIHtcbiAgICAgIHNyYy5saW5lKFwiaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1wiKTtcbiAgICAgIHNyYy5saW5lKFwiaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XCIpO1xuICAgICAgY2RrRGVwcy5hZGRWMURlcGVuZGVuY2llcyhcIkBhd3MtY2RrL2F3cy1sYW1iZGFcIik7XG4gICAgICBjZGtEZXBzLmFkZFYxRGVwZW5kZW5jaWVzKFwiQGF3cy1jZGsvY29yZVwiKTtcbiAgICB9IGVsc2Uge1xuICAgICAgc3JjLmxpbmUoXCJpbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XCIpO1xuICAgICAgc3JjLmxpbmUoXCJpbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcIik7XG4gICAgfVxuXG4gICAgc3JjLmxpbmUoKTtcbiAgICBzcmMubGluZShcIi8qKlwiKTtcbiAgICBzcmMubGluZShgICogUHJvcHMgZm9yICR7Y29uc3RydWN0TmFtZX1gKTtcbiAgICBzcmMubGluZShcIiAqL1wiKTtcbiAgICBzcmMub3BlbihgZXhwb3J0IGludGVyZmFjZSAke3Byb3BzVHlwZX0gZXh0ZW5kcyBsYW1iZGEuRnVuY3Rpb25PcHRpb25zIHtgKTtcbiAgICBzcmMuY2xvc2UoXCJ9XCIpO1xuICAgIHNyYy5saW5lKCk7XG4gICAgc3JjLmxpbmUoXCIvKipcIik7XG4gICAgc3JjLmxpbmUoXG4gICAgICBgICogQW4gQVdTIExhbWJkYSBmdW5jdGlvbiB3aGljaCBleGVjdXRlcyAke2NvbnZlcnRUb1Bvc2l4UGF0aChcbiAgICAgICAgYmFzZVBhdGhcbiAgICAgICl9LmBcbiAgICApO1xuICAgIHNyYy5saW5lKFwiICovXCIpO1xuICAgIHNyYy5vcGVuKGBleHBvcnQgY2xhc3MgJHtjb25zdHJ1Y3ROYW1lfSBleHRlbmRzIGxhbWJkYS5GdW5jdGlvbiB7YCk7XG4gICAgc3JjLm9wZW4oXG4gICAgICBgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiAke3Byb3BzVHlwZX0pIHtgXG4gICAgKTtcbiAgICBzcmMub3BlbihcInN1cGVyKHNjb3BlLCBpZCwge1wiKTtcbiAgICBzcmMubGluZShgZGVzY3JpcHRpb246ICcke2NvbnZlcnRUb1Bvc2l4UGF0aChlbnRyeXBvaW50KX0nLGApO1xuICAgIHNyYy5saW5lKFwiLi4ucHJvcHMsXCIpO1xuICAgIHNyYy5saW5lKGBydW50aW1lOiBsYW1iZGEuUnVudGltZS4ke3J1bnRpbWUuZnVuY3Rpb25SdW50aW1lfSxgKTtcbiAgICBzcmMubGluZShcImhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcIik7XG4gICAgc3JjLmxpbmUoXG4gICAgICBgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcke2NvbnZlcnRUb1Bvc2l4UGF0aChcbiAgICAgICAgcmVsYXRpdmVPdXRmaWxlXG4gICAgICApfScpKSxgXG4gICAgKTtcbiAgICBzcmMuY2xvc2UoXCJ9KTtcIik7XG4gICAgaWYgKG9wdGlvbnMuYXdzU2RrQ29ubmVjdGlvblJldXNlID8/IHRydWUpIHtcbiAgICAgIHNyYy5saW5lKFxuICAgICAgICBcInRoaXMuYWRkRW52aXJvbm1lbnQoJ0FXU19OT0RFSlNfQ09OTkVDVElPTl9SRVVTRV9FTkFCTEVEJywgJzEnLCB7IHJlbW92ZUluRWRnZTogdHJ1ZSB9KTtcIlxuICAgICAgKTtcbiAgICB9XG4gICAgc3JjLmNsb3NlKFwifVwiKTtcbiAgICBzcmMuY2xvc2UoXCJ9XCIpO1xuXG4gICAgdGhpcy5wcm9qZWN0LmxvZ2dlci52ZXJib3NlKFxuICAgICAgYCR7YmFzZVBhdGh9OiBjb25zdHJ1Y3QgXCIke2NvbnN0cnVjdE5hbWV9XCIgZ2VuZXJhdGVkIHVuZGVyIFwiJHtjb25zdHJ1Y3RGaWxlfVwiYFxuICAgICk7XG4gICAgdGhpcy5wcm9qZWN0LmxvZ2dlci52ZXJib3NlKFxuICAgICAgYCR7YmFzZVBhdGh9OiBidW5kbGUgdGFzayBcIiR7YnVuZGxlLmJ1bmRsZVRhc2submFtZX1cImBcbiAgICApO1xuICAgIGlmIChidW5kbGUud2F0Y2hUYXNrKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLnZlcmJvc2UoXG4gICAgICAgIGAke2Jhc2VQYXRofTogYnVuZGxlIHdhdGNoIHRhc2sgXCIke2J1bmRsZS53YXRjaFRhc2submFtZX1cImBcbiAgICAgICk7XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogVGhlIHJ1bnRpbWUgZm9yIHRoZSBBV1MgTGFtYmRhIGZ1bmN0aW9uLlxuICovXG5leHBvcnQgY2xhc3MgTGFtYmRhUnVudGltZSB7XG4gIC8qKlxuICAgKiBOb2RlLmpzIDEwLnhcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTk9ERUpTXzEwX1ggPSBuZXcgTGFtYmRhUnVudGltZShcbiAgICBcIk5PREVKU18xMF9YXCIsXG4gICAgXCJub2RlMTBcIlxuICApO1xuXG4gIC8qKlxuICAgKiBOb2RlLmpzIDEyLnhcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTk9ERUpTXzEyX1ggPSBuZXcgTGFtYmRhUnVudGltZShcbiAgICBcIk5PREVKU18xMl9YXCIsXG4gICAgXCJub2RlMTJcIlxuICApO1xuXG4gIC8qKlxuICAgKiBOb2RlLmpzIDE0LnhcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTk9ERUpTXzE0X1ggPSBuZXcgTGFtYmRhUnVudGltZShcbiAgICBcIk5PREVKU18xNF9YXCIsXG4gICAgXCJub2RlMTRcIlxuICApO1xuXG4gIHB1YmxpYyByZWFkb25seSBlc2J1aWxkUGxhdGZvcm0gPSBcIm5vZGVcIjtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKFxuICAgIC8qKlxuICAgICAqIFRoZSBhd3MtbGFtYmRhLlJ1bnRpbWUgbWVtYmVyIG5hbWUgdG8gdXNlLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBmdW5jdGlvblJ1bnRpbWU6IHN0cmluZyxcblxuICAgIC8qKlxuICAgICAqIFRoZSBlc2J1aWxkIHNldHRpbmcgdG8gdXNlLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBlc2J1aWxkVGFyZ2V0OiBzdHJpbmdcbiAgKSB7fVxufVxuXG4vKipcbiAqIENvbnZlcnRzIHRoZSBnaXZlbiBwYXRoIHN0cmluZyB0byBwb3NpeCBpZiBpdCB3YXNuJ3QgYWxyZWFkeS5cbiAqL1xuZnVuY3Rpb24gY29udmVydFRvUG9zaXhQYXRoKHA6IHN0cmluZykge1xuICByZXR1cm4gcC5zcGxpdChzZXApLmpvaW4ocG9zaXguc2VwKTtcbn1cbiJdfQ==