"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const PULL_REQUEST_REF = "${{ github.event.pull_request.head.ref }}";
const PULL_REQUEST_REPOSITORY = "${{ github.event.pull_request.head.repo.full_name }}";
const BUILD_JOBID = "build";
const SELF_MUTATION_STEP = "self_mutation";
const SELF_MUTATION_HAPPENED_OUTPUT = "self_mutation_happened";
const IS_FORK = "github.event.pull_request.head.repo.full_name != github.repository";
const NOT_FORK = `!(${IS_FORK})`;
const SELF_MUTATION_CONDITION = `needs.${BUILD_JOBID}.outputs.${SELF_MUTATION_HAPPENED_OUTPUT}`;
class BuildWorkflow extends component_1.Component {
    constructor(project, options) {
        var _b, _c, _d, _e, _f;
        super(project);
        this.defaultRunners = ["ubuntu-latest"];
        this._postBuildJobs = [];
        const github = github_1.GitHub.of(project);
        if (!github) {
            throw new Error("BuildWorkflow is currently only supported for GitHub projects");
        }
        this.github = github;
        this.preBuildSteps = (_b = options.preBuildSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.gitIdentity = (_d = options.gitIdentity) !== null && _d !== void 0 ? _d : constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory = options.artifactsDirectory;
        const mutableBuilds = (_e = options.mutableBuild) !== null && _e !== void 0 ? _e : true;
        this.workflow = new github_1.GithubWorkflow(github, "build");
        this.workflow.on((_f = options.workflowTriggers) !== null && _f !== void 0 ? _f : {
            pullRequest: {},
            workflowDispatch: {},
        });
        this.addBuildJob(options);
        if (mutableBuilds) {
            this.addSelfMutationJob(options);
        }
    }
    addBuildJob(options) {
        var _b;
        this.workflow.addJob(BUILD_JOBID, {
            runsOn: (_b = options.runsOn) !== null && _b !== void 0 ? _b : this.defaultRunners,
            container: options.containerImage
                ? { image: options.containerImage }
                : undefined,
            env: {
                CI: "true",
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            steps: (() => this.renderBuildSteps()),
            outputs: {
                [SELF_MUTATION_HAPPENED_OUTPUT]: {
                    stepId: SELF_MUTATION_STEP,
                    outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                },
            },
        });
    }
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds() {
        return [BUILD_JOBID, ...this._postBuildJobs];
    }
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id, job) {
        const steps = [];
        if (this.artifactsDirectory) {
            steps.push({
                name: "Download build artifacts",
                uses: "actions/download-artifact@v2",
                with: {
                    name: constants_1.BUILD_ARTIFACT_NAME,
                    path: this.artifactsDirectory,
                },
            });
        }
        steps.push(...job.steps);
        this.workflow.addJob(id, {
            needs: [BUILD_JOBID],
            // only run if build did not self-mutate
            if: `! ${SELF_MUTATION_CONDITION}`,
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._postBuildJobs.push(id);
    }
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task, options) {
        this.addPostBuildJobCommands(`post-build-${task.name}`, [`${this.project.projenCommand} ${task.name}`], {
            checkoutRepo: true,
            installDeps: true,
            tools: options.tools,
            runsOn: options.runsOn,
        });
    }
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id, commands, options) {
        var _b;
        const steps = [];
        if (options === null || options === void 0 ? void 0 : options.checkoutRepo) {
            steps.push({
                name: "Checkout",
                uses: "actions/checkout@v2",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                },
            });
        }
        if ((options === null || options === void 0 ? void 0 : options.checkoutRepo) && (options === null || options === void 0 ? void 0 : options.installDeps) &&
            this.project instanceof javascript_1.NodeProject) {
            steps.push({
                name: "Install dependencies",
                run: `${this.project.package.installCommand}`,
            });
        }
        steps.push({ run: commands.join("\n") });
        this.addPostBuildJob(id, {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: options === null || options === void 0 ? void 0 : options.tools,
            runsOn: (_b = options === null || options === void 0 ? void 0 : options.runsOn) !== null && _b !== void 0 ? _b : this.defaultRunners,
            steps,
        });
    }
    addSelfMutationJob(options) {
        var _b;
        this.workflow.addJob("self-mutation", {
            runsOn: (_b = options.runsOn) !== null && _b !== void 0 ? _b : this.defaultRunners,
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            needs: [BUILD_JOBID],
            if: `always() && ${SELF_MUTATION_CONDITION} && ${NOT_FORK}`,
            steps: [
                ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                    // we need to use a PAT so that our push will trigger the build workflow
                    token: `\${{ secrets.${this.workflow.projenTokenSecret} }}`,
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                }),
                ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
                {
                    name: "Push changes",
                    run: [
                        "  git add .",
                        '  git commit -s -m "chore: self mutation"',
                        `  git push origin HEAD:${PULL_REQUEST_REF}`,
                    ].join("\n"),
                },
            ],
        });
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderBuildSteps() {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v2",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                },
            },
            ...this.preBuildSteps,
            {
                name: this.buildTask.name,
                run: this.github.project.runTaskCommand(this.buildTask),
            },
            ...this.postBuildSteps,
            // check for mutations and upload a git patch file as an artifact
            ...workflow_actions_1.WorkflowActions.createUploadGitPatch({
                stepId: SELF_MUTATION_STEP,
                outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                mutationError: "Files were changed during build (see build log). If this was triggered from a fork, you will need to update your branch.",
            }),
            // upload the build artifact only if we have post-build jobs (otherwise, there's no point)
            ...(this._postBuildJobs.length == 0
                ? []
                : [
                    {
                        name: "Upload artifact",
                        uses: "actions/upload-artifact@v2.1.1",
                        with: {
                            name: constants_1.BUILD_ARTIFACT_NAME,
                            path: this.artifactsDirectory,
                        },
                    },
                ]),
        ];
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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