"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const semver = require("semver");
const yargs = require("yargs");
const inventory = require("../../inventory");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const projects_1 = require("../../projects");
const util_1 = require("../../util");
const macros_1 = require("../macros");
const util_2 = require("../util");
class Command {
    constructor() {
        this.command = "new [PROJECT-TYPE-NAME] [OPTIONS]";
        this.describe = "Creates a new projen project";
    }
    builder(args) {
        var _a;
        args.positional("PROJECT-TYPE-NAME", {
            describe: "optional only when --from is used and there is a single project type in the external module",
            type: "string",
        });
        args.option("synth", {
            type: "boolean",
            default: true,
            desc: "Synthesize after creating .projenrc.js",
        });
        args.option("comments", {
            type: "boolean",
            default: true,
            desc: "Include commented out options in .projenrc.js (use --no-comments to disable)",
        });
        args.option("from", {
            type: "string",
            alias: "f",
            desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")',
        });
        args.option("git", {
            type: "boolean",
            default: true,
            desc: "Run `git init` and create an initial commit (use --no-git to disable)",
        });
        args.example("projen new awscdk-app-ts", 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example("projen new --from projen-vue@^2", 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : "", {
                builder: (cargs) => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(false);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.simpleType !== "string" &&
                            option.simpleType !== "number" &&
                            option.simpleType !== "boolean" &&
                            option.kind !== "enum") {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, "")) !== null && _c !== void 0 ? _c : ""];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== "undefined") {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default
                                    .replace(/^\ *-/, "")
                                    .replace(/\.$/, "")
                                    .trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        const argType = option.kind === "enum" ? "string" : option.simpleType;
                        cargs.option(option.switch, {
                            group: required ? "Required:" : "Optional:",
                            type: argType,
                            description: desc.join(" "),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: (argv) => initProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return initProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map((x) => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(cwd, value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== "undefined" && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function initProjectFromModule(baseDir, spec, args) {
    var _a, _b;
    const projenVersion = (_a = args.projenVersion) !== null && _a !== void 0 ? _a : "latest";
    const installCommand = util_2.renderInstallCommand(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        util_1.exec(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`npm ls --prefix=${baseDir} --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = util_2.installPackage(baseDir, spec);
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [baseDir],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter((x) => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map((p) => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(",")}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested
        ? projects[0]
        : projects.find((p) => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(",")}`);
    }
    for (const option of (_b = type.options) !== null && _b !== void 0 ? _b : []) {
        if (option.simpleType !== "string" &&
            option.simpleType !== "number" &&
            option.simpleType !== "boolean") {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            if (option.simpleType === "number") {
                args[option.name] = parseInt(args[option.name]);
                args[option.switch] = args[option.name];
            }
            else if (option.simpleType === "boolean") {
                const raw = args[option.name];
                const safe = typeof raw === "string" ? util_1.isTruthy(raw) : raw;
                args[option.name] = safe;
                args[option.switch] = safe;
            }
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== "undefined") {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    args.devDeps = [spec];
    args["dev-deps"] = [spec];
    await initProject(baseDir, type, args);
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function initProject(baseDir, type, args) {
    var _a;
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    projects_1.Projects.createProject({
        dir: (_a = props.outdir) !== null && _a !== void 0 ? _a : baseDir,
        projectFqn: type.fqn,
        projectOptions: props,
        optionHints: args.comments
            ? option_hints_1.InitProjectOptionHints.FEATURED
            : option_hints_1.InitProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (fs.existsSync(path.join(baseDir, "package.json")) && args.post) {
        util_1.exec("npm run eslint --if-present", { cwd: baseDir });
    }
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        const gitversion = util_1.getGitVersion(util_1.execCapture("git --version", { cwd: baseDir }).toString());
        logging.debug("system using git version ", gitversion);
        if (gitversion && semver.gte(gitversion, "2.28.0")) {
            git("init -b main");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("default branch name set to main");
        }
        else {
            git("init");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("older version of git detected, changed default branch name to main");
            git("branch -M main");
        }
    }
}
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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