"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const yargs = require("yargs");
const common_1 = require("../common");
const task_runtime_1 = require("../task-runtime");
const synth_1 = require("./synth");
const tasks_1 = require("./tasks");
const DEFAULT_RC = path_1.resolve(common_1.PROJEN_RC);
async function main() {
    const ya = yargs;
    ya.commandDir("cmds");
    const runtime = new task_runtime_1.TaskRuntime(".");
    tasks_1.discoverTaskCommands(runtime, ya);
    ya.recommendCommands();
    ya.strictCommands();
    ya.showHelpOnFail(false);
    ya.wrap(yargs.terminalWidth());
    ya.option("post", {
        type: "boolean",
        default: true,
        desc: "Run post-synthesis steps such as installing dependencies. Use --no-post to skip",
    });
    ya.option("watch", {
        type: "boolean",
        default: false,
        desc: "Keep running and resynthesize when projenrc changes",
        alias: "w",
    });
    ya.options("debug", { type: "boolean", default: false, desc: "Debug logs" });
    ya.options("rc", {
        desc: "path to .projenrc.js file",
        default: DEFAULT_RC,
        type: "string",
    });
    ya.completion();
    ya.help();
    // do not use the default yargs '--version' implementation since it is
    // global by default (it appears on all subcommands)
    ya.version(false);
    ya.option("version", {
        type: "boolean",
        description: "Show version number",
        global: false,
    });
    const args = ya.argv;
    if (args.debug) {
        process.env.DEBUG = "true";
    }
    // no command means synthesize
    if (args._.length === 0) {
        if (args.version) {
            console.log(common_1.PROJEN_VERSION);
            process.exit(0);
        }
        await synth_1.synth(runtime, {
            post: args.post,
            watch: args.watch,
            rcfile: args.rc,
        });
    }
}
main().catch((e) => {
    console.error(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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