"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synth = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const common_1 = require("../common");
const logging = require("../logging");
const project_1 = require("../project");
const projenModule = path.dirname(require.resolve("../../package.json"));
async function synth(runtime, options) {
    var _a;
    const workdir = runtime.workdir;
    const rcfile = path.resolve(workdir, (_a = options.rcfile) !== null && _a !== void 0 ? _a : common_1.PROJEN_RC); // TODO: support non javascript projenrc (e.g. java projects)
    // if --rc points to .projenrc.js, then behave as if it wasn't specified.
    if (rcfile === path.resolve(workdir, common_1.PROJEN_RC)) {
        delete options.rcfile;
    }
    // if there are no tasks, we assume this is not a projen project (modern
    // projects must at least have the "default" task).
    if (runtime.tasks.length === 0 && !fs.existsSync(rcfile)) {
        logging.error('Unable to find projen project. Use "projen new" to create a new project.');
        process.exit(1);
    }
    // run synth once
    const success = await trySynth();
    if (options.watch) {
        // if we are in watch mode, start the watch loop
        watchLoop();
    }
    else if (!success) {
        // make sure exit code is non-zero if we are not in watch mode
        process.exit(1);
    }
    async function trySynth() {
        // determine if post synthesis tasks should be executed (e.g. "yarn install").
        process.env.PROJEN_DISABLE_POST = (!options.post).toString();
        try {
            const defaultTask = runtime.tasks.find((t) => t.name === project_1.Project.DEFAULT_TASK);
            // if "--rc" is specified, ignore the default task
            if (defaultTask) {
                if (!options.rcfile) {
                    runtime.runTask(defaultTask.name);
                    return true;
                }
                else {
                    logging.warn("Default task skipped. Trying legacy synthesis since --rc is specified");
                }
            }
            // for backwards compatibility, if there is a .projenrc.js file, default to "node .projenrc.js"
            if (tryLegacySynth()) {
                return true;
            }
            throw new Error('Unable to find a task named "default"');
        }
        catch (e) {
            logging.error(`Synthesis failed: ${e.message}`);
            return false;
        }
    }
    function watchLoop() {
        logging.info(`Watching for changes in ${workdir}...`);
        const watch = fs.watch(workdir, { recursive: true });
        watch.on("change", (event) => {
            // we only care about "change" events
            if (event !== "change") {
                return;
            }
            process.stdout.write("\x1Bc"); // clear screen
            watch.close();
            trySynth()
                .then(() => watchLoop())
                .catch(() => watchLoop());
        });
    }
    function tryLegacySynth() {
        const rcdir = path.dirname(rcfile);
        if (!fs.existsSync(rcfile)) {
            return false;
        }
        // if node_modules/projen is not a directory or does not exist, create a
        // temporary symlink to the projen that we are currently running in order to
        // allow .projenrc.js to `require()` it.
        const nodeModules = path.resolve(rcdir, "node_modules");
        const projenModulePath = path.resolve(nodeModules, "projen");
        if (!fs.existsSync(path.join(projenModulePath, "package.json")) ||
            !fs.statSync(projenModulePath).isDirectory()) {
            fs.removeSync(projenModulePath);
            fs.mkdirpSync(nodeModules);
            fs.symlinkSync(projenModule, projenModulePath, os.platform() === "win32" ? "junction" : null);
        }
        const ret = child_process_1.spawnSync(process.execPath, [rcfile], {
            stdio: ["inherit", "inherit", "pipe"],
        });
        if (ret.error) {
            throw new Error(`Synthesis failed: ${ret.error}`);
        }
        else if (ret.status !== 0) {
            logging.error(ret.stderr.toString());
            throw new Error(`Synthesis failed: calling "${process.execPath} ${rcfile}" exited with status=${ret.status}`);
        }
        return true;
    }
}
exports.synth = synth;
//# sourceMappingURL=data:application/json;base64,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