"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyType = exports.Dependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const common_1 = require("./common");
const component_1 = require("./component");
const json_1 = require("./json");
/**
 * The `Dependencies` component is responsible to track the list of dependencies
 * a project has, and then used by project types as the model for rendering
 * project-specific dependency manifests such as the dependencies section
 * `package.json` files.
 *
 * To add a dependency you can use a project-type specific API such as
 * `nodeProject.addDeps()` or use the generic API of `project.deps`:
 */
class Dependencies extends component_1.Component {
    /**
     * Adds a dependencies component to the project.
     * @param project The parent project
     */
    constructor(project) {
        super(project);
        this._deps = new Array();
        // this is not really required at the moment, but actually quite useful as a
        // checked-in source of truth for dependencies and will potentially be
        // valuable in the future for CLI tools.
        if (!project.ejected) {
            new json_1.JsonFile(project, Dependencies.MANIFEST_FILE, {
                omitEmpty: true,
                obj: () => this.toJson(),
            });
        }
    }
    /**
     * Returns the coordinates of a dependency spec.
     *
     * Given `foo@^3.4.0` returns `{ name: "foo", version: "^3.4.0" }`.
     * Given `bar@npm:@bar/legacy` returns `{ name: "bar", version: "npm:@bar/legacy" }`.
     */
    static parseDependency(spec) {
        const scope = spec.startsWith("@");
        if (scope) {
            spec = spec.substr(1);
        }
        const [module, ...version] = spec.split("@");
        const name = scope ? `@${module}` : module;
        if (version.length == 0) {
            return { name };
        }
        else {
            return { name, version: version === null || version === void 0 ? void 0 : version.join("@") };
        }
    }
    /**
     * A copy of all dependencies recorded for this project.
     *
     * The list is sorted by type->name->version
     */
    get all() {
        return [...this._deps].sort(compareDeps).map(normalizeDep);
    }
    /**
     * Returns a dependency by name.
     *
     * Fails if there is no dependency defined by that name or if `type` is not
     * provided and there is more then one dependency type for this dependency.
     *
     * @param name The name of the dependency
     * @param type The dependency type. If this dependency is defined only for a
     * single type, this argument can be omitted.
     *
     * @returns a copy (cannot be modified)
     */
    getDependency(name, type) {
        const dep = this.tryGetDependency(name, type);
        if (!dep) {
            const msg = type
                ? `there is no ${type} dependency defined on "${name}"`
                : `there is no dependency defined on "${name}"`;
            throw new Error(msg);
        }
        return dep;
    }
    /**
     * Returns a dependency by name.
     *
     * Returns `undefined` if there is no dependency defined by that name or if
     * `type` is not provided and there is more then one dependency type for this
     * dependency.
     *
     * @param name The name of the dependency
     * @param type The dependency type. If this dependency is defined only for a
     * single type, this argument can be omitted.
     *
     * @returns a copy (cannot be modified) or undefined if there is no match
     */
    tryGetDependency(name, type) {
        const idx = this.tryGetDependencyIndex(name, type);
        if (idx === -1) {
            return undefined;
        }
        return {
            ...normalizeDep(this._deps[idx]),
        };
    }
    /**
     * Adds a dependency to this project.
     * @param spec The dependency spec in the format `MODULE[@VERSION]` where
     * `MODULE` is the package-manager-specific module name and `VERSION` is an
     * optional semantic version requirement (e.g. `^3.4.0`).
     * @param type The type of the dependency.
     */
    addDependency(spec, type, metadata = {}) {
        this.project.logger.debug(`${type}-dep ${spec}`);
        const dep = {
            ...Dependencies.parseDependency(spec),
            type,
            metadata,
        };
        const existingDepIndex = this.tryGetDependencyIndex(dep.name, type);
        if (existingDepIndex !== -1) {
            this.project.logger.debug(`updating existing ${dep.type}-dep ${dep.name} with more specific version/metadata`);
            this._deps[existingDepIndex] = dep;
        }
        else {
            this._deps.push(dep);
        }
        return dep;
    }
    /**
     * Removes a dependency.
     * @param name The name of the module to remove (without the version)
     * @param type The dependency type. This is only required if there the
     * dependency is defined for multiple types.
     */
    removeDependency(name, type) {
        const removeIndex = this.tryGetDependencyIndex(name, type);
        if (removeIndex === -1) {
            return;
        }
        this._deps.splice(removeIndex, 1);
    }
    tryGetDependencyIndex(name, type) {
        const deps = this._deps.filter((d) => d.name === name);
        if (deps.length === 0) {
            return -1; // not found
        }
        if (!type) {
            if (deps.length > 1) {
                throw new Error(`"${name}" is defined for multiple dependency types: ${deps
                    .map((d) => d.type)
                    .join(",")}. Please specify dependency type`);
            }
            type = deps[0].type;
        }
        return this._deps.findIndex((dep) => dep.name === name && dep.type === type);
    }
    toJson() {
        if (this._deps.length === 0) {
            return undefined;
        }
        return {
            dependencies: this._deps.sort(compareDeps).map(normalizeDep),
        };
    }
}
exports.Dependencies = Dependencies;
_a = JSII_RTTI_SYMBOL_1;
Dependencies[_a] = { fqn: "projen.Dependencies", version: "0.52.53" };
/**
 * The project-relative path of the deps manifest file.
 */
Dependencies.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, "deps.json");
function normalizeDep(d) {
    const obj = {};
    for (const [k, v] of Object.entries(d)) {
        if (v == undefined) {
            continue;
        }
        if (typeof v === "object" && Object.keys(v).length === 0) {
            continue;
        }
        if (Array.isArray(v) && v.length === 0) {
            continue;
        }
        obj[k] = v;
    }
    return obj;
}
function compareDeps(d1, d2) {
    return specOf(d1).localeCompare(specOf(d2));
    function specOf(dep) {
        let spec = dep.type + ":" + dep.name;
        if (dep.version) {
            spec += "@" + dep.version;
        }
        return spec;
    }
}
/**
 * Type of dependency.
 */
var DependencyType;
(function (DependencyType) {
    /**
     * The dependency is required for the program/library during runtime.
     */
    DependencyType["RUNTIME"] = "runtime";
    /**
     * The dependency is required at runtime but expected to be installed by the
     * consumer.
     */
    DependencyType["PEER"] = "peer";
    /**
     * The dependency is bundled and shipped with the module, so consumers are not
     * required to install it.
     */
    DependencyType["BUNDLED"] = "bundled";
    /**
     * The dependency is required to run the `build` task.
     */
    DependencyType["BUILD"] = "build";
    /**
     * The dependency is required to run the `test` task.
     */
    DependencyType["TEST"] = "test";
    /**
     * The dependency is required for development (e.g. IDE plugins).
     */
    DependencyType["DEVENV"] = "devenv";
})(DependencyType = exports.DependencyType || (exports.DependencyType = {}));
//# sourceMappingURL=data:application/json;base64,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