"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerComposeProtocol = exports.DockerComposeService = exports.DockerCompose = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("./component");
const util_1 = require("./util");
const yaml_1 = require("./yaml");
/**
 * Create a docker-compose YAML file.
 */
class DockerCompose extends component_1.Component {
    constructor(project, props) {
        var _c;
        super(project);
        const nameSuffix = (props === null || props === void 0 ? void 0 : props.nameSuffix) ? `${props.nameSuffix}.yml` : "yml";
        new yaml_1.YamlFile(project, `docker-compose.${nameSuffix}`, {
            committed: true,
            readonly: true,
            obj: () => this._synthesizeDockerCompose(),
        });
        if ((props === null || props === void 0 ? void 0 : props.schemaVersion) && !parseFloat(props.schemaVersion)) {
            throw Error("Version tag needs to be a number");
        }
        this.version = (props === null || props === void 0 ? void 0 : props.schemaVersion) ? props.schemaVersion : "3.3";
        this.services = {};
        // Add the services provided via the constructor argument.
        const initialServices = (_c = props === null || props === void 0 ? void 0 : props.services) !== null && _c !== void 0 ? _c : {};
        for (const [name, serviceDescription] of Object.entries(initialServices)) {
            this.addService(name, serviceDescription);
        }
    }
    /**
     * Depends on a service name.
     */
    static serviceName(serviceName) {
        return {
            serviceName,
        };
    }
    /**
     * Create a port mapping.
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    static portMapping(publishedPort, targetPort, options) {
        var _c;
        const protocol = (_c = options === null || options === void 0 ? void 0 : options.protocol) !== null && _c !== void 0 ? _c : DockerComposeProtocol.TCP;
        return {
            target: targetPort,
            published: publishedPort,
            protocol: protocol,
            mode: "host",
        };
    }
    /**
     * Create a bind volume that binds a host path to the target path in the container.
     * @param sourcePath Host path name
     * @param targetPath Target path name
     */
    static bindVolume(sourcePath, targetPath) {
        return {
            bind(_volumeInfo) {
                return {
                    type: "bind",
                    source: sourcePath,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named volume and mount it to the target path. If you use this
     * named volume in several services, the volume will be shared. In this
     * case, the volume configuration of the first-provided options are used.
     *
     * @param volumeName Name of the volume
     * @param targetPath Target path
     * @param options volume configuration (default: docker compose defaults)
     */
    static namedVolume(volumeName, targetPath, options = {}) {
        return {
            bind(volumeInfo) {
                volumeInfo.addVolumeConfiguration(volumeName, options);
                return {
                    type: "volume",
                    source: volumeName,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Add a service to the docker-compose file.
     * @param serviceName name of the service
     * @param description a service description
     */
    addService(serviceName, description) {
        const service = new DockerComposeService(serviceName, description);
        this.services[serviceName] = service;
        return service;
    }
    /**
     * @internal
     */
    _synthesizeDockerCompose() {
        if (Object.keys(this.services).length === 0) {
            throw new Error("DockerCompose requires at least one service");
        }
        return renderDockerComposeFile(this.services, this.version);
    }
}
exports.DockerCompose = DockerCompose;
_a = JSII_RTTI_SYMBOL_1;
DockerCompose[_a] = { fqn: "projen.DockerCompose", version: "0.52.53" };
/**
 * A docker-compose service.
 */
class DockerComposeService {
    constructor(serviceName, serviceDescription) {
        var _c, _d, _e, _f;
        if ((!serviceDescription.imageBuild && !serviceDescription.image) ||
            (serviceDescription.imageBuild && serviceDescription.image)) {
            throw new Error(`A service ${serviceName} requires exactly one of a \`imageBuild\` or \`image\` key`);
        }
        this.serviceName = serviceName;
        this.command = serviceDescription.command;
        this.image = serviceDescription.image;
        this.imageBuild = serviceDescription.imageBuild;
        this.dependsOn = (_c = serviceDescription.dependsOn) !== null && _c !== void 0 ? _c : [];
        this.volumes = (_d = serviceDescription.volumes) !== null && _d !== void 0 ? _d : [];
        this.ports = (_e = serviceDescription.ports) !== null && _e !== void 0 ? _e : [];
        this.environment = (_f = serviceDescription.environment) !== null && _f !== void 0 ? _f : {};
    }
    /**
     * Add a port mapping
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    addPort(publishedPort, targetPort, options) {
        var _c;
        (_c = this.ports) === null || _c === void 0 ? void 0 : _c.push(DockerCompose.portMapping(publishedPort, targetPort, options));
    }
    /**
     * Add an environment variable
     * @param name environment variable name
     * @param value value of the environment variable
     */
    addEnvironment(name, value) {
        this.environment[name] = value;
    }
    /**
     * Make the service depend on another service.
     * @param serviceName
     */
    addDependsOn(serviceName) {
        this.dependsOn.push(serviceName);
    }
    /**
     * Add a volume to the service.
     * @param volume
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
}
exports.DockerComposeService = DockerComposeService;
_b = JSII_RTTI_SYMBOL_1;
DockerComposeService[_b] = { fqn: "projen.DockerComposeService", version: "0.52.53" };
/**
 * Network protocol for port mapping
 */
var DockerComposeProtocol;
(function (DockerComposeProtocol) {
    /**
     * TCP protocol
     */
    DockerComposeProtocol["TCP"] = "tcp";
    /**
     * UDP protocol
     */
    DockerComposeProtocol["UDP"] = "udp";
})(DockerComposeProtocol = exports.DockerComposeProtocol || (exports.DockerComposeProtocol = {}));
function renderDockerComposeFile(serviceDescriptions, version) {
    var _c, _d;
    // Record volume configuration
    const volumeConfig = {};
    const volumeInfo = {
        addVolumeConfiguration(volumeName, configuration) {
            if (!volumeConfig[volumeName]) {
                // First volume configuration takes precedence.
                volumeConfig[volumeName] = configuration;
            }
        },
    };
    // Render service configuration
    const services = {};
    for (const [serviceName, serviceDescription] of Object.entries(serviceDescriptions !== null && serviceDescriptions !== void 0 ? serviceDescriptions : {})) {
        // Resolve the names of each dependency and check that they exist.
        // Note: They may not exist if the user made a mistake when referencing a
        // service by name via `DockerCompose.serviceName()`.
        // @see DockerCompose.serviceName
        const dependsOn = Array();
        for (const dependsOnServiceName of (_c = serviceDescription.dependsOn) !== null && _c !== void 0 ? _c : []) {
            const resolvedServiceName = dependsOnServiceName.serviceName;
            if (resolvedServiceName === serviceName) {
                throw new Error(`Service ${serviceName} cannot depend on itself`);
            }
            if (!serviceDescriptions[resolvedServiceName]) {
                throw new Error(`Unable to resolve service named ${resolvedServiceName} for ${serviceName}`);
            }
            dependsOn.push(resolvedServiceName);
        }
        // Give each volume binding a chance to bind any necessary volume
        // configuration and provide volume mount information for the service.
        const volumes = [];
        for (const volumeBinding of (_d = serviceDescription.volumes) !== null && _d !== void 0 ? _d : []) {
            volumes.push(volumeBinding.bind(volumeInfo));
        }
        // Create and store the service configuration, taking care not to create
        // object members with undefined values.
        services[serviceName] = {
            ...getObjectWithKeyAndValueIfValueIsDefined("image", serviceDescription.image),
            ...getObjectWithKeyAndValueIfValueIsDefined("build", serviceDescription.imageBuild),
            ...getObjectWithKeyAndValueIfValueIsDefined("command", serviceDescription.command),
            ...(Object.keys(serviceDescription.environment).length > 0
                ? { environment: serviceDescription.environment }
                : {}),
            ...(serviceDescription.ports.length > 0
                ? { ports: serviceDescription.ports }
                : {}),
            ...(dependsOn.length > 0 ? { dependsOn } : {}),
            ...(volumes.length > 0 ? { volumes } : {}),
        };
    }
    // Explicit with the type here because the decamelize step after this wipes
    // out types.
    const input = {
        version,
        services,
        ...(Object.keys(volumeConfig).length > 0 ? { volumes: volumeConfig } : {}),
    };
    // Change most keys to snake case.
    return util_1.decamelizeKeysRecursively(input, {
        shouldDecamelize: shouldDecamelizeDockerComposeKey,
        separator: "_",
    });
}
/**
 * Returns `{ [key]: value }` if `value` is defined, otherwise returns `{}` so
 * that object spreading can be used to generate a peculiar interface.
 * @param key
 * @param value
 */
function getObjectWithKeyAndValueIfValueIsDefined(key, value) {
    return value !== undefined ? { [key]: value } : {};
}
/**
 * Determines whether the key at the given path should be decamelized.
 * Largely, all keys should be snake cased. But, there are some
 * exceptions for user-provided names for services, volumes, and
 * environment variables.
 *
 * @param path
 */
function shouldDecamelizeDockerComposeKey(path) {
    const poundPath = path.join("#");
    // Does not decamelize user's names.
    // services.namehere:
    // volumes.namehere:
    if (/^(services|volumes)#[^#]+$/.test(poundPath)) {
        return false;
    }
    // Does not decamelize environment variables
    // services.namehere.environment.*
    if (/^services#[^#]+#environment#/.test(poundPath)) {
        return false;
    }
    // Does not decamelize build arguments
    // services.namehere.build.args.*
    if (/^services#[^#]+#build#args#/.test(poundPath)) {
        return false;
    }
    // Otherwise, let it all decamelize.
    return true;
}
//# sourceMappingURL=data:application/json;base64,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