"use strict";
// @see https://docs.gitlab.com/ee/ci/yaml/
// Schema from: https://gitlab.com/gitlab-org/gitlab/-/raw/master/app/assets/javascripts/editor/schema/ci.json
// Initial TypeScript Interfaces from: https://app.quicktype.io/#
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowWhen = exports.Strategy = exports.LinkType = exports.DefaultElement = exports.KubernetesEnum = exports.DeploymentTier = exports.Action = exports.JobWhen = exports.CacheWhen = void 0;
/**
 * Configure when artifacts are uploaded depended on job status.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachewhen
 */
var CacheWhen;
(function (CacheWhen) {
    /** Upload artifacts regardless of job status. */
    CacheWhen["ALWAYS"] = "always";
    /** Upload artifacts only when the job fails. */
    CacheWhen["ON_FAILURE"] = "on_failure";
    /** Upload artifacts only when the job succeeds (this is the default). */
    CacheWhen["ON_SUCCESS"] = "on_success";
})(CacheWhen = exports.CacheWhen || (exports.CacheWhen = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * @see https://docs.gitlab.com/ee/ci/yaml/#when
 */
var JobWhen;
(function (JobWhen) {
    JobWhen["ALWAYS"] = "always";
    JobWhen["DELAYED"] = "delayed";
    JobWhen["MANUAL"] = "manual";
    JobWhen["NEVER"] = "never";
    JobWhen["ON_FAILURE"] = "on_failure";
    JobWhen["ON_SUCCESS"] = "on_success";
})(JobWhen = exports.JobWhen || (exports.JobWhen = {}));
/**
 * Specifies what this job will do. 'start' (default) indicates the job will start the
 * deployment. 'prepare' indicates this will not affect the deployment. 'stop' indicates
 * this will stop the deployment.
 */
var Action;
(function (Action) {
    Action["PREPARE"] = "prepare";
    Action["START"] = "start";
    Action["STOP"] = "stop";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Explicitly specifies the tier of the deployment environment if non-standard environment
 * name is used.
 */
var DeploymentTier;
(function (DeploymentTier) {
    DeploymentTier["DEVELOPMENT"] = "development";
    DeploymentTier["OTHER"] = "other";
    DeploymentTier["PRODUCTION"] = "production";
    DeploymentTier["STAGING"] = "staging";
    DeploymentTier["TESTING"] = "testing";
})(DeploymentTier = exports.DeploymentTier || (exports.DeploymentTier = {}));
/**
 * Filter job based on if Kubernetes integration is active.
 */
var KubernetesEnum;
(function (KubernetesEnum) {
    KubernetesEnum["ACTIVE"] = "active";
})(KubernetesEnum = exports.KubernetesEnum || (exports.KubernetesEnum = {}));
var DefaultElement;
(function (DefaultElement) {
    DefaultElement["AFTER_SCRIPT"] = "after_script";
    DefaultElement["ARTIFACTS"] = "artifacts";
    DefaultElement["BEFORE_SCRIPT"] = "before_script";
    DefaultElement["CACHE"] = "cache";
    DefaultElement["IMAGE"] = "image";
    DefaultElement["INTERRUPTIBLE"] = "interruptible";
    DefaultElement["RETRY"] = "retry";
    DefaultElement["SERVICES"] = "services";
    DefaultElement["TAGS"] = "tags";
    DefaultElement["TIMEOUT"] = "timeout";
})(DefaultElement = exports.DefaultElement || (exports.DefaultElement = {}));
/**
 * The content kind of what users can download via url.
 */
var LinkType;
(function (LinkType) {
    LinkType["IMAGE"] = "image";
    LinkType["OTHER"] = "other";
    LinkType["PACKAGE"] = "package";
    LinkType["RUNBOOK"] = "runbook";
})(LinkType = exports.LinkType || (exports.LinkType = {}));
/**
 * You can mirror the pipeline status from the triggered pipeline to the source bridge job
 * by using strategy: depend
 * @see https://docs.gitlab.com/ee/ci/yaml/#triggerstrategy
 */
var Strategy;
(function (Strategy) {
    Strategy["DEPEND"] = "depend";
})(Strategy = exports.Strategy || (exports.Strategy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * The value can only be 'always' or 'never' when used with workflow.
 * @see https://docs.gitlab.com/ee/ci/yaml/#workflowrules
 */
var WorkflowWhen;
(function (WorkflowWhen) {
    WorkflowWhen["ALWAYS"] = "always";
    WorkflowWhen["NEVER"] = "never";
})(WorkflowWhen = exports.WorkflowWhen || (exports.WorkflowWhen = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uZmlndXJhdGlvbi1tb2RlbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9naXRsYWIvY29uZmlndXJhdGlvbi1tb2RlbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsMkNBQTJDO0FBQzNDLDhHQUE4RztBQUM5RyxpRUFBaUU7OztBQVdqRTs7O0dBR0c7QUFDSCxJQUFZLFNBT1g7QUFQRCxXQUFZLFNBQVM7SUFDbkIsaURBQWlEO0lBQ2pELDhCQUFpQixDQUFBO0lBQ2pCLGdEQUFnRDtJQUNoRCxzQ0FBeUIsQ0FBQTtJQUN6Qix5RUFBeUU7SUFDekUsc0NBQXlCLENBQUE7QUFDM0IsQ0FBQyxFQVBXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBT3BCO0FBc0xEOzs7R0FHRztBQUNILElBQVksT0FPWDtBQVBELFdBQVksT0FBTztJQUNqQiw0QkFBaUIsQ0FBQTtJQUNqQiw4QkFBbUIsQ0FBQTtJQUNuQiw0QkFBaUIsQ0FBQTtJQUNqQiwwQkFBZSxDQUFBO0lBQ2Ysb0NBQXlCLENBQUE7SUFDekIsb0NBQXlCLENBQUE7QUFDM0IsQ0FBQyxFQVBXLE9BQU8sR0FBUCxlQUFPLEtBQVAsZUFBTyxRQU9sQjtBQXlGRDs7OztHQUlHO0FBQ0gsSUFBWSxNQUlYO0FBSkQsV0FBWSxNQUFNO0lBQ2hCLDZCQUFtQixDQUFBO0lBQ25CLHlCQUFlLENBQUE7SUFDZix1QkFBYSxDQUFBO0FBQ2YsQ0FBQyxFQUpXLE1BQU0sR0FBTixjQUFNLEtBQU4sY0FBTSxRQUlqQjtBQUVEOzs7R0FHRztBQUNILElBQVksY0FNWDtBQU5ELFdBQVksY0FBYztJQUN4Qiw2Q0FBMkIsQ0FBQTtJQUMzQixpQ0FBZSxDQUFBO0lBQ2YsMkNBQXlCLENBQUE7SUFDekIscUNBQW1CLENBQUE7SUFDbkIscUNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQU5XLGNBQWMsR0FBZCxzQkFBYyxLQUFkLHNCQUFjLFFBTXpCO0FBeUJEOztHQUVHO0FBQ0gsSUFBWSxjQUVYO0FBRkQsV0FBWSxjQUFjO0lBQ3hCLG1DQUFpQixDQUFBO0FBQ25CLENBQUMsRUFGVyxjQUFjLEdBQWQsc0JBQWMsS0FBZCxzQkFBYyxRQUV6QjtBQWVELElBQVksY0FXWDtBQVhELFdBQVksY0FBYztJQUN4QiwrQ0FBNkIsQ0FBQTtJQUM3Qix5Q0FBdUIsQ0FBQTtJQUN2QixpREFBK0IsQ0FBQTtJQUMvQixpQ0FBZSxDQUFBO0lBQ2YsaUNBQWUsQ0FBQTtJQUNmLGlEQUErQixDQUFBO0lBQy9CLGlDQUFlLENBQUE7SUFDZix1Q0FBcUIsQ0FBQTtJQUNyQiwrQkFBYSxDQUFBO0lBQ2IscUNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQVhXLGNBQWMsR0FBZCxzQkFBYyxLQUFkLHNCQUFjLFFBV3pCO0FBOEREOztHQUVHO0FBQ0gsSUFBWSxRQUtYO0FBTEQsV0FBWSxRQUFRO0lBQ2xCLDJCQUFlLENBQUE7SUFDZiwyQkFBZSxDQUFBO0lBQ2YsK0JBQW1CLENBQUE7SUFDbkIsK0JBQW1CLENBQUE7QUFDckIsQ0FBQyxFQUxXLFFBQVEsR0FBUixnQkFBUSxLQUFSLGdCQUFRLFFBS25CO0FBc0VEOzs7O0dBSUc7QUFDSCxJQUFZLFFBRVg7QUFGRCxXQUFZLFFBQVE7SUFDbEIsNkJBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQUZXLFFBQVEsR0FBUixnQkFBUSxLQUFSLGdCQUFRLFFBRW5CO0FBdUNEOzs7O0dBSUc7QUFDSCxJQUFZLFlBR1g7QUFIRCxXQUFZLFlBQVk7SUFDdEIsaUNBQWlCLENBQUE7SUFDakIsK0JBQWUsQ0FBQTtBQUNqQixDQUFDLEVBSFcsWUFBWSxHQUFaLG9CQUFZLEtBQVosb0JBQVksUUFHdkIiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvXG4vLyBTY2hlbWEgZnJvbTogaHR0cHM6Ly9naXRsYWIuY29tL2dpdGxhYi1vcmcvZ2l0bGFiLy0vcmF3L21hc3Rlci9hcHAvYXNzZXRzL2phdmFzY3JpcHRzL2VkaXRvci9zY2hlbWEvY2kuanNvblxuLy8gSW5pdGlhbCBUeXBlU2NyaXB0IEludGVyZmFjZXMgZnJvbTogaHR0cHM6Ly9hcHAucXVpY2t0eXBlLmlvLyNcblxuLyoqXG4gKiBDYWNoZSBEZWZpbml0aW9uLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyNjYWNoZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhY2hlIHtcbiAgLyoqIERlZmluZXMgd2hlbiB0byBzYXZlIHRoZSBjYWNoZSwgYmFzZWQgb24gdGhlIHN0YXR1cyBvZiB0aGUgam9iIChEZWZhdWx0OiBKb2IgU3VjY2VzcykuICovXG4gIHJlYWRvbmx5IHdoZW4/OiBDYWNoZVdoZW47XG59XG5cbi8qKlxuICogQ29uZmlndXJlIHdoZW4gYXJ0aWZhY3RzIGFyZSB1cGxvYWRlZCBkZXBlbmRlZCBvbiBqb2Igc3RhdHVzLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyNjYWNoZXdoZW5cbiAqL1xuZXhwb3J0IGVudW0gQ2FjaGVXaGVuIHtcbiAgLyoqIFVwbG9hZCBhcnRpZmFjdHMgcmVnYXJkbGVzcyBvZiBqb2Igc3RhdHVzLiAqL1xuICBBTFdBWVMgPSBcImFsd2F5c1wiLFxuICAvKiogVXBsb2FkIGFydGlmYWN0cyBvbmx5IHdoZW4gdGhlIGpvYiBmYWlscy4gKi9cbiAgT05fRkFJTFVSRSA9IFwib25fZmFpbHVyZVwiLFxuICAvKiogVXBsb2FkIGFydGlmYWN0cyBvbmx5IHdoZW4gdGhlIGpvYiBzdWNjZWVkcyAodGhpcyBpcyB0aGUgZGVmYXVsdCkuICovXG4gIE9OX1NVQ0NFU1MgPSBcIm9uX3N1Y2Nlc3NcIixcbn1cblxuLyoqXG4gKiBEZWZhdWx0IHNldHRpbmdzIGZvciB0aGUgQ0kgQ29uZmlndXJhdGlvbi4gSm9icyB0aGF0IGRvIG5vdCBkZWZpbmUgb25lIG9yIG1vcmUgb2YgdGhlIGxpc3RlZCBrZXl3b3JkcyB1c2UgdGhlIHZhbHVlIGRlZmluZWQgaW4gdGhlIGRlZmF1bHQgc2VjdGlvbi5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGxhYi5jb20vZWUvY2kveWFtbC8jZGVmYXVsdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIERlZmF1bHQge1xuICAvKiBEZWZpbmVzIHNjcmlwdHMgdGhhdCBzaG91bGQgcnVuICphZnRlciogYWxsIGpvYnMuIENhbiBiZSBvdmVycmlkZW4gYnkgdGhlIGpvYiBsZXZlbCBgYWZ0ZXJTY3JpcHQuICovXG4gIHJlYWRvbmx5IGFmdGVyU2NyaXB0Pzogc3RyaW5nW107XG4gIC8qIExpc3Qgb2YgZmlsZXMgYW5kIGRpcmVjdG9yaWVzIHRoYXQgc2hvdWxkIGJlIGF0dGFjaGVkIHRvIHRoZSBqb2IgaWYgaXQgc3VjY2VlZHMuIEFydGlmYWN0cyBhcmUgc2VudCB0byBHaXRsYWIgd2hlcmUgdGhleSBjYW4gYmUgZG93bmxvYWRlZC4gKi9cbiAgcmVhZG9ubHkgYXJ0aWZhY3RzPzogQXJ0aWZhY3RzO1xuICAvKiBEZWZpbmVzIHNjcmlwdHMgdGhhdCBzaG91bGQgcnVuICpiZWZvcmUqIGFsbCBqb2JzLiBDYW4gYmUgb3ZlcnJpZGVuIGJ5IHRoZSBqb2IgbGV2ZWwgYGFmdGVyU2NyaXB0YC4gKi9cbiAgcmVhZG9ubHkgYmVmb3JlU2NyaXB0Pzogc3RyaW5nW107XG4gIC8qIEEgbGlzdCBvZiBmaWxlcyBhbmQgZGlyZWN0b3JpZXMgdG8gY2FjaGUgYmV0d2VlbiBqb2JzLiBZb3UgY2FuIG9ubHkgdXNlIHBhdGhzIHRoYXQgYXJlIGluIHRoZSBsb2NhbCB3b3JraW5nIGNvcHkuICovXG4gIHJlYWRvbmx5IGNhY2hlPzogQ2FjaGU7XG4gIC8qIFNwZWNpZmllcyB0aGUgZGVmYXVsdCBkb2NrZXIgaW1hZ2UgdG8gdXNlIGdsb2JhbGx5IGZvciBhbGwgam9icy4gKi9cbiAgcmVhZG9ubHkgaW1hZ2U/OiBJbWFnZTtcbiAgLyogSWYgYSBqb2Igc2hvdWxkIGJlIGNhbmNlbGVkIHdoZW4gYSBuZXdlciBwaXBlbGluZSBzdGFydHMgYmVmb3JlIHRoZSBqb2IgY29tcGxldGVzIChEZWZhdWx0OiBmYWxzZSkuKi9cbiAgcmVhZG9ubHkgaW50ZXJydXB0aWJsZT86IGJvb2xlYW47XG4gIC8qIEhvdyBtYW55IHRpbWVzIGEgam9iIGlzIHJldHJpZWQgaWYgaXQgZmFpbHMuIElmIG5vdCBkZWZpbmVkLCBkZWZhdWx0cyB0byAwIGFuZCBqb2JzIGRvIG5vdCByZXRyeS4gKi9cbiAgcmVhZG9ubHkgcmV0cnk/OiBSZXRyeTtcbiAgLyogQWRkaXRpb25hbCBEb2NrZXIgaW1hZ2VzIHRvIHJ1biBzY3JpcHRzIGluLiBUaGUgc2VydmljZSBpbWFnZSBpcyBsaW5rZWQgdG8gdGhlIGltYWdlIHNwZWNpZmllZCBpbiB0aGUgIGltYWdlIHBhcmFtZXRlci4gKi9cbiAgcmVhZG9ubHkgc2VydmljZXM/OiBTZXJ2aWNlW107XG4gIC8qIFVzZWQgdG8gc2VsZWN0IGEgc3BlY2lmaWMgcnVubmVyIGZyb20gdGhlIGxpc3Qgb2YgYWxsIHJ1bm5lcnMgdGhhdCBhcmUgYXZhaWxhYmxlIGZvciB0aGUgcHJvamVjdC4gKi9cbiAgcmVhZG9ubHkgdGFncz86IHN0cmluZ1tdO1xuICAvKiBBIGRlZmF1bHQgdGltZW91dCBqb2Igd3JpdHRlbiBpbiBuYXR1cmFsIGxhbmd1YWdlIChFeC4gb25lIGhvdXIsIDM2MDAgc2Vjb25kcywgNjAgbWludXRlcykuICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogVXNlZCB0byBzcGVjaWZ5IGEgbGlzdCBvZiBmaWxlcyBhbmQgZGlyZWN0b3JpZXMgdGhhdCBzaG91bGQgYmUgYXR0YWNoZWQgdG8gdGhlIGpvYiBpZiBpdFxuICogc3VjY2VlZHMuIEFydGlmYWN0cyBhcmUgc2VudCB0byBHaXRsYWIgd2hlcmUgdGhleSBjYW4gYmUgZG93bmxvYWRlZC5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGxhYi5jb20vZWUvY2kveWFtbC8jYXJ0aWZhY3RzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXJ0aWZhY3RzIHtcbiAgLyoqIEEgbGlzdCBvZiBwYXRocyB0byBmaWxlcy9mb2xkZXJzIHRoYXQgc2hvdWxkIGJlIGV4Y2x1ZGVkIGluIHRoZSBhcnRpZmFjdC4gKi9cbiAgcmVhZG9ubHkgZXhjbHVkZT86IHN0cmluZ1tdO1xuICAvKiogSG93IGxvbmcgYXJ0aWZhY3RzIHNob3VsZCBiZSBrZXB0LiBUaGV5IGFyZSBzYXZlZCAzMCBkYXlzIGJ5IGRlZmF1bHQuIEFydGlmYWN0cyB0aGF0IGhhdmUgZXhwaXJlZCBhcmUgcmVtb3ZlZCBwZXJpb2RpY2FsbHkgdmlhIGNyb24gam9iLiBTdXBwb3J0cyBhIHdpZGUgdmFyaWV0eSBvZiBmb3JtYXRzLCBlLmcuICcxIHdlZWsnLCAnMyBtaW5zIDQgc2VjJywgJzIgaHJzIDIwIG1pbicsICcyaDIwbWluJywgJzYgbW9zIDEgZGF5JywgJzQ3IHlycyA2IG1vcyBhbmQgNGQnLCAnMyB3ZWVrcyBhbmQgMiBkYXlzJy4gKi9cbiAgcmVhZG9ubHkgZXhwaXJlSW4/OiBzdHJpbmc7XG4gIC8qKiBDYW4gYmUgdXNlZCB0byBleHBvc2Ugam9iIGFydGlmYWN0cyBpbiB0aGUgbWVyZ2UgcmVxdWVzdCBVSS4gR2l0TGFiIHdpbGwgYWRkIGEgbGluayA8ZXhwb3NlX2FzPiB0byB0aGUgcmVsZXZhbnQgbWVyZ2UgcmVxdWVzdCB0aGF0IHBvaW50cyB0byB0aGUgYXJ0aWZhY3QuICovXG4gIHJlYWRvbmx5IGV4cG9zZUFzPzogc3RyaW5nO1xuICAvKiogTmFtZSBmb3IgdGhlIGFyY2hpdmUgY3JlYXRlZCBvbiBqb2Igc3VjY2Vzcy4gQ2FuIHVzZSB2YXJpYWJsZXMgaW4gdGhlIG5hbWUsIGUuZy4gJyRDSV9KT0JfTkFNRScgKi9cbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgLyoqIEEgbGlzdCBvZiBwYXRocyB0byBmaWxlcy9mb2xkZXJzIHRoYXQgc2hvdWxkIGJlIGluY2x1ZGVkIGluIHRoZSBhcnRpZmFjdC4gKi9cbiAgcmVhZG9ubHkgcGF0aHM/OiBzdHJpbmdbXTtcbiAgLyoqIFJlcG9ydHMgd2lsbCBiZSB1cGxvYWRlZCBhcyBhcnRpZmFjdHMsIGFuZCBvZnRlbiBkaXNwbGF5ZWQgaW4gdGhlIEdpdGxhYiBVSSwgc3VjaCBhcyBpbiBNZXJnZSBSZXF1ZXN0cy4qL1xuICByZWFkb25seSByZXBvcnRzPzogUmVwb3J0cztcbiAgLyoqIFdoZXRoZXIgdG8gYWRkIGFsbCB1bnRyYWNrZWQgZmlsZXMgKGFsb25nIHdpdGggJ2FydGlmYWN0cy5wYXRocycpIHRvIHRoZSBhcnRpZmFjdC4qL1xuICByZWFkb25seSB1bnRyYWNrZWQ/OiBib29sZWFuO1xuICAvKiogQ29uZmlndXJlIHdoZW4gYXJ0aWZhY3RzIGFyZSB1cGxvYWRlZCBkZXBlbmRlZCBvbiBqb2Igc3RhdHVzLiovXG4gIHJlYWRvbmx5IHdoZW4/OiBDYWNoZVdoZW47XG59XG5cbi8qKlxuICogUmVwb3J0cyB3aWxsIGJlIHVwbG9hZGVkIGFzIGFydGlmYWN0cywgYW5kIG9mdGVuIGRpc3BsYXllZCBpbiB0aGUgR2l0bGFiIFVJLCBzdWNoIGFzIGluXG4gKiBNZXJnZSBSZXF1ZXN0cy5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGxhYi5jb20vZWUvY2kveWFtbC8jYXJ0aWZhY3RzcmVwb3J0c1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlcG9ydHMge1xuICAvKiogUGF0aCBmb3IgZmlsZShzKSB0aGF0IHNob3VsZCBiZSBwYXJzZWQgYXMgQ29iZXJ0dXJhIFhNTCBjb3ZlcmFnZSByZXBvcnQqL1xuICByZWFkb25seSBjb2JlcnR1cmE/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIHdpdGggY29kZSBxdWFsaXR5IHJlcG9ydChzKSAoc3VjaCBhcyBDb2RlIENsaW1hdGUpLiovXG4gIHJlYWRvbmx5IGNvZGVxdWFsaXR5Pzogc3RyaW5nW107XG4gIC8qKiBQYXRoIHRvIGZpbGUgb3IgbGlzdCBvZiBmaWxlcyB3aXRoIENvbnRhaW5lciBzY2FubmluZyB2dWxuZXJhYmlsaXRpZXMgcmVwb3J0KHMpLiovXG4gIHJlYWRvbmx5IGNvbnRhaW5lclNjYW5uaW5nPzogc3RyaW5nW107XG4gIC8qKiBQYXRoIHRvIGZpbGUgb3IgbGlzdCBvZiBmaWxlcyB3aXRoIERBU1QgdnVsbmVyYWJpbGl0aWVzIHJlcG9ydChzKS4qL1xuICByZWFkb25seSBkYXN0Pzogc3RyaW5nW107XG4gIC8qKiBQYXRoIHRvIGZpbGUgb3IgbGlzdCBvZiBmaWxlcyB3aXRoIERlcGVuZGVuY3kgc2Nhbm5pbmcgdnVsbmVyYWJpbGl0aWVzIHJlcG9ydChzKS4qL1xuICByZWFkb25seSBkZXBlbmRlbmN5U2Nhbm5pbmc/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIGNvbnRhaW5pbmcgcnVudGltZS1jcmVhdGVkIHZhcmlhYmxlcyBmb3IgdGhpcyBqb2IuKi9cbiAgcmVhZG9ubHkgZG90ZW52Pzogc3RyaW5nW107XG4gIC8qKiBQYXRoIGZvciBmaWxlKHMpIHRoYXQgc2hvdWxkIGJlIHBhcnNlZCBhcyBKVW5pdCBYTUwgcmVzdWx0Ki9cbiAgcmVhZG9ubHkganVuaXQ/OiBzdHJpbmdbXTtcbiAgLyoqIERlcHJlY2F0ZWQgaW4gMTIuODogUGF0aCB0byBmaWxlIG9yIGxpc3Qgb2YgZmlsZXMgd2l0aCBsaWNlbnNlIHJlcG9ydChzKS4qL1xuICByZWFkb25seSBsaWNlbnNlTWFuYWdlbWVudD86IHN0cmluZ1tdO1xuICAvKiogUGF0aCB0byBmaWxlIG9yIGxpc3Qgb2YgZmlsZXMgd2l0aCBsaWNlbnNlIHJlcG9ydChzKS4qL1xuICByZWFkb25seSBsaWNlbnNlU2Nhbm5pbmc/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIGNvbnRhaW5pbmcgY29kZSBpbnRlbGxpZ2VuY2UgKExhbmd1YWdlIFNlcnZlciBJbmRleCBGb3JtYXQpLiovXG4gIHJlYWRvbmx5IGxzaWY/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIHdpdGggY3VzdG9tIG1ldHJpY3MgcmVwb3J0KHMpLiovXG4gIHJlYWRvbmx5IG1ldHJpY3M/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIHdpdGggcGVyZm9ybWFuY2UgbWV0cmljcyByZXBvcnQocykuKi9cbiAgcmVhZG9ubHkgcGVyZm9ybWFuY2U/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIHdpdGggcmVxdWlyZW1lbnRzIHJlcG9ydChzKS4qL1xuICByZWFkb25seSByZXF1aXJlbWVudHM/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIHdpdGggU0FTVCB2dWxuZXJhYmlsaXRpZXMgcmVwb3J0KHMpLiovXG4gIHJlYWRvbmx5IHNhc3Q/OiBzdHJpbmdbXTtcbiAgLyoqIFBhdGggdG8gZmlsZSBvciBsaXN0IG9mIGZpbGVzIHdpdGggc2VjcmV0IGRldGVjdGlvbiByZXBvcnQocykuKi9cbiAgcmVhZG9ubHkgc2VjcmV0RGV0ZWN0aW9uPzogc3RyaW5nW107XG4gIC8qKiBQYXRoIHRvIGZpbGUgb3IgbGlzdCBvZiBmaWxlcyB3aXRoIHRlcnJhZm9ybSBwbGFuKHMpLiovXG4gIHJlYWRvbmx5IHRlcnJhZm9ybT86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIFNwZWNpZmllcyB0aGUgZG9ja2VyIGltYWdlIHRvIHVzZSBmb3IgdGhlIGpvYiBvciBnbG9iYWxseSBmb3IgYWxsIGpvYnMuIEpvYiBjb25maWd1cmF0aW9uXG4gKiB0YWtlcyBwcmVjZWRlbmNlIG92ZXIgZ2xvYmFsIHNldHRpbmcuIFJlcXVpcmVzIGEgY2VydGFpbiBraW5kIG9mIEdpdGxhYiBydW5uZXIgZXhlY3V0b3IuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvI2ltYWdlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW1hZ2Uge1xuICAvKiogQ29tbWFuZCBvciBzY3JpcHQgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgYXMgdGhlIGNvbnRhaW5lcidzIGVudHJ5cG9pbnQuIEl0IHdpbGwgYmUgdHJhbnNsYXRlZCB0byBEb2NrZXIncyAtLWVudHJ5cG9pbnQgb3B0aW9uIHdoaWxlIGNyZWF0aW5nIHRoZSBjb250YWluZXIuIFRoZSBzeW50YXggaXMgc2ltaWxhciB0byBEb2NrZXJmaWxlJ3MgRU5UUllQT0lOVCBkaXJlY3RpdmUsIHdoZXJlIGVhY2ggc2hlbGwgdG9rZW4gaXMgYSBzZXBhcmF0ZSBzdHJpbmcgaW4gdGhlIGFycmF5LiovXG4gIHJlYWRvbmx5IGVudHJ5cG9pbnQ/OiBhbnlbXTtcbiAgLyoqIEZ1bGwgbmFtZSBvZiB0aGUgaW1hZ2UgdGhhdCBzaG91bGQgYmUgdXNlZC4gSXQgc2hvdWxkIGNvbnRhaW4gdGhlIFJlZ2lzdHJ5IHBhcnQgaWYgbmVlZGVkLiovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbn1cbi8qKlxuICogSG93IG1hbnkgdGltZXMgYSBqb2IgaXMgcmV0cmllZCBpZiBpdCBmYWlscy4gSWYgbm90IGRlZmluZWQsIGRlZmF1bHRzIHRvIDAgYW5kIGpvYnMgZG8gbm90IHJldHJ5LlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyNyZXRyeVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJldHJ5IHtcbiAgLyoqIDAgKGRlZmF1bHQpLCAxLCBvciAyLiovXG4gIHJlYWRvbmx5IG1heD86IG51bWJlcjtcbiAgLyoqIEVpdGhlciBhIHNpbmdsZSBvciBhcnJheSBvZiBlcnJvciB0eXBlcyB0byB0cmlnZ2VyIGpvYiByZXRyeS4qL1xuICByZWFkb25seSB3aGVuPzogYW55O1xufVxuLyoqXG4gKiBVc2VkIHRvIHNwZWNpZnkgYW4gYWRkaXRpb25hbCBEb2NrZXIgaW1hZ2UgdG8gcnVuIHNjcmlwdHMgaW4uIFRoZSBzZXJ2aWNlIGltYWdlIGlzIGxpbmtlZCB0byB0aGUgaW1hZ2Ugc3BlY2lmaWVkIGluIHRoZSBARGVmYXVsdCBpbWFnZSBrZXl3b3JkLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyNzZXJ2aWNlc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlcnZpY2Uge1xuICAvKiogQWRkaXRpb25hbCBhbGlhcyB0aGF0IGNhbiBiZSB1c2VkIHRvIGFjY2VzcyB0aGUgc2VydmljZSBmcm9tIHRoZSBqb2IncyBjb250YWluZXIuIFJlYWQgQWNjZXNzaW5nIHRoZSBzZXJ2aWNlcyBmb3IgbW9yZSBpbmZvcm1hdGlvbi4qL1xuICByZWFkb25seSBhbGlhcz86IHN0cmluZztcbiAgLyoqIENvbW1hbmQgb3Igc2NyaXB0IHRoYXQgc2hvdWxkIGJlIHVzZWQgYXMgdGhlIGNvbnRhaW5lcidzIGNvbW1hbmQuIEl0IHdpbGwgYmUgdHJhbnNsYXRlZCB0byBhcmd1bWVudHMgcGFzc2VkIHRvIERvY2tlciBhZnRlciB0aGUgaW1hZ2UncyBuYW1lLiBUaGUgc3ludGF4IGlzIHNpbWlsYXIgdG8gRG9ja2VyZmlsZSdzIENNRCBkaXJlY3RpdmUsIHdoZXJlIGVhY2ggc2hlbGwgdG9rZW4gaXMgYSBzZXBhcmF0ZSBzdHJpbmcgaW4gdGhlIGFycmF5LiovXG4gIHJlYWRvbmx5IGNvbW1hbmQ/OiBzdHJpbmdbXTtcbiAgLyoqIENvbW1hbmQgb3Igc2NyaXB0IHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGFzIHRoZSBjb250YWluZXIncyBlbnRyeXBvaW50LiBJdCB3aWxsIGJlIHRyYW5zbGF0ZWQgdG8gRG9ja2VyJ3MgLS1lbnRyeXBvaW50IG9wdGlvbiB3aGlsZSBjcmVhdGluZyB0aGUgY29udGFpbmVyLiBUaGUgc3ludGF4IGlzIHNpbWlsYXIgdG8gRG9ja2VyZmlsZSdzIEVOVFJZUE9JTlQgZGlyZWN0aXZlLCB3aGVyZSBlYWNoIHNoZWxsIHRva2VuIGlzIGEgc2VwYXJhdGUgc3RyaW5nIGluIHRoZSBhcnJheS4qL1xuICByZWFkb25seSBlbnRyeXBvaW50Pzogc3RyaW5nW107XG4gIC8qKiBGdWxsIG5hbWUgb2YgdGhlIGltYWdlIHRoYXQgc2hvdWxkIGJlIHVzZWQuIEl0IHNob3VsZCBjb250YWluIHRoZSBSZWdpc3RyeSBwYXJ0IGlmIG5lZWRlZC4qL1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQW4gaW5jbHVkZWQgWUFNTCBmaWxlLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyNpbmNsdWRlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW5jbHVkZSB7XG4gIC8qKiBSZWxhdGl2ZSBwYXRoIGZyb20gbG9jYWwgcmVwb3NpdG9yeSByb290IChgL2ApIHRvIHRoZSBgeWFtbGAvYHltbGAgZmlsZSB0ZW1wbGF0ZS4gVGhlIGZpbGUgbXVzdCBiZSBvbiB0aGUgc2FtZSBicmFuY2gsIGFuZCBkb2VzIG5vdCB3b3JrIGFjcm9zcyBnaXQgc3VibW9kdWxlcy4qL1xuICByZWFkb25seSBsb2NhbD86IHN0cmluZztcbiAgLyoqUnVsZXMgYWxsb3dzIGZvciBhbiBhcnJheSBvZiBpbmRpdmlkdWFsIHJ1bGUgb2JqZWN0cyB0byBiZSBldmFsdWF0ZWQgaW4gb3JkZXIsIHVudGlsIG9uZSBtYXRjaGVzIGFuZCBkeW5hbWljYWxseSBwcm92aWRlcyBhdHRyaWJ1dGVzIHRvIHRoZSBqb2IuICovXG4gIHJlYWRvbmx5IHJ1bGVzPzogSW5jbHVkZVJ1bGVbXTtcbiAgLyoqIEZpbGVzIGZyb20gYW5vdGhlciBwcml2YXRlIHByb2plY3Qgb24gdGhlIHNhbWUgR2l0TGFiIGluc3RhbmNlLiBZb3UgY2FuIHVzZSBgZmlsZWAgaW4gY29tYmluYXRpb24gd2l0aCBgcHJvamVjdGAgb25seS4gKi9cbiAgcmVhZG9ubHkgZmlsZT86IHN0cmluZ1tdO1xuICAvKiogUGF0aCB0byB0aGUgcHJvamVjdCwgZS5nLiBgZ3JvdXAvcHJvamVjdGAsIG9yIGBncm91cC9zdWItZ3JvdXAvcHJvamVjdGAuKi9cbiAgcmVhZG9ubHkgcHJvamVjdD86IHN0cmluZztcbiAgLyoqIEJyYW5jaC9UYWcvQ29tbWl0LWhhc2ggZm9yIHRoZSB0YXJnZXQgcHJvamVjdC4qL1xuICByZWFkb25seSByZWY/OiBzdHJpbmc7XG4gIC8qKiBVc2UgYSBgLmdpdGxhYi1jaS55bWxgIHRlbXBsYXRlIGFzIGEgYmFzZSwgZS5nLiBgTm9kZWpzLmdpdGxhYi1jaS55bWxgLiovXG4gIHJlYWRvbmx5IHRlbXBsYXRlPzogc3RyaW5nO1xuICAvKiogVVJMIHRvIGEgYHlhbWxgL2B5bWxgIHRlbXBsYXRlIGZpbGUgdXNpbmcgSFRUUC9IVFRQUy4qL1xuICByZWFkb25seSByZW1vdGU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUnVsZXMgYWxsb3dzIGZvciBhbiBhcnJheSBvZiBpbmRpdmlkdWFsIHJ1bGUgb2JqZWN0cyB0byBiZSBldmFsdWF0ZWQgaW4gb3JkZXIsIHVudGlsIG9uZVxuICogbWF0Y2hlcyBhbmQgZHluYW1pY2FsbHkgcHJvdmlkZXMgYXR0cmlidXRlcyB0byB0aGUgam9iLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sL2luY2x1ZGVzLmh0bWwjdXNlLXJ1bGVzLXdpdGgtaW5jbHVkZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEluY2x1ZGVSdWxlIHtcbiAgLyogV2hldGhlciBhIHBpcGVsaW5lIHNob3VsZCBjb250aW51ZSBydW5uaW5nIHdoZW4gYSBqb2IgZmFpbHMuICovXG4gIHJlYWRvbmx5IGFsbG93RmFpbHVyZT86IGJvb2xlYW4gfCBBbGxvd0ZhaWx1cmU7XG4gIC8qIFNwZWNpZnkgd2hlbiB0byBhZGQgYSBqb2IgdG8gYSBwaXBlbGluZSBieSBjaGVja2luZyBmb3IgY2hhbmdlcyB0byBzcGVjaWZpYyBmaWxlcy4gKi9cbiAgcmVhZG9ubHkgY2hhbmdlcz86IHN0cmluZ1tdO1xuICAvKiBSdW4gYSBqb2Igd2hlbiBjZXJ0YWluIGZpbGVzIGV4aXN0IGluIHRoZSByZXBvc2l0b3J5LiAqL1xuICByZWFkb25seSBleGlzdHM/OiBzdHJpbmdbXTtcbiAgLyogQ2xhdXNlcyB0byBzcGVjaWZ5IHdoZW4gdG8gYWRkIGEgam9iIHRvIGEgcGlwZWxpbmUuKi9cbiAgcmVhZG9ubHkgaWY/OiBzdHJpbmc7XG4gIC8qIEV4ZWN1dGUgc2NyaXB0cyBhZnRlciBhIHdhaXRpbmcgcGVyaW9kIHdyaXR0ZW4gaW4gbmF0dXJhbCBsYW5ndWFnZSAoRXguIG9uZSBob3VyLCAzNjAwIHNlY29uZHMsIDYwIG1pbnV0ZXMpLiAqL1xuICByZWFkb25seSBzdGFydEluPzogc3RyaW5nO1xuICAvKiBVc2UgdmFyaWFibGVzIGluIHJ1bGVzIHRvIGRlZmluZSB2YXJpYWJsZXMgZm9yIHNwZWNpZmljIGNvbmRpdGlvbnMuICovXG4gIHJlYWRvbmx5IHZhcmlhYmxlcz86IFJlY29yZDxzdHJpbmcsIG51bWJlciB8IHN0cmluZz47XG4gIC8qIENvbmRpdGlvbnMgZm9yIHdoZW4gdG8gcnVuIHRoZSBqb2IuIERlZmF1bHRzIHRvICdvbl9zdWNjZXNzJyAqL1xuICByZWFkb25seSB3aGVuPzogSm9iV2hlbjtcbn1cblxuLyoqXG4gKiBFeGl0IGNvZGUgdGhhdCBhcmUgbm90IGNvbnNpZGVyZWQgZmFpbHVyZS4gVGhlIGpvYiBmYWlscyBmb3IgYW55IG90aGVyIGV4aXQgY29kZS5cbiAqIFlvdSBjYW4gbGlzdCB3aGljaCBleGl0IGNvZGVzIGFyZSBub3QgY29uc2lkZXJlZCBmYWlsdXJlcy4gVGhlIGpvYiBmYWlscyBmb3IgYW55IG90aGVyXG4gKiBleGl0IGNvZGUuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvI2FsbG93X2ZhaWx1cmVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBbGxvd0ZhaWx1cmUge1xuICByZWFkb25seSBleGl0Q29kZXM6IG51bWJlcltdIHwgbnVtYmVyO1xufVxuXG4vKipcbiAqIERlc2NyaWJlcyB0aGUgY29uZGl0aW9ucyBmb3Igd2hlbiB0byBydW4gdGhlIGpvYi4gRGVmYXVsdHMgdG8gJ29uX3N1Y2Nlc3MnLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyN3aGVuXG4gKi9cbmV4cG9ydCBlbnVtIEpvYldoZW4ge1xuICBBTFdBWVMgPSBcImFsd2F5c1wiLFxuICBERUxBWUVEID0gXCJkZWxheWVkXCIsXG4gIE1BTlVBTCA9IFwibWFudWFsXCIsXG4gIE5FVkVSID0gXCJuZXZlclwiLFxuICBPTl9GQUlMVVJFID0gXCJvbl9mYWlsdXJlXCIsXG4gIE9OX1NVQ0NFU1MgPSBcIm9uX3N1Y2Nlc3NcIixcbn1cblxuLyoqXG4gKiBKb2JzIGFyZSB0aGUgbW9zdCBmdW5kYW1lbnRhbCBlbGVtZW50IG9mIGEgLmdpdGxhYi1jaS55bWwgZmlsZS5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGxhYi5jb20vZWUvY2kvam9icy9cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2Ige1xuICAvKiBEZWZpbmVzIHNjcmlwdHMgdGhhdCBzaG91bGQgcnVuICphZnRlciogdGhlIGpvYi4gKi9cbiAgcmVhZG9ubHkgYWZ0ZXJTY3JpcHQ/OiBzdHJpbmdbXTtcbiAgLyoqIFdoZXRoZXIgdG8gYWxsb3cgdGhlIHBpcGVsaW5lIHRvIGNvbnRpbnVlIHJ1bm5pbmcgb24gam9iIGZhaWx1cmUgKERlZmF1bHQ6IGZhbHNlKS4gKi9cbiAgcmVhZG9ubHkgYWxsb3dGYWlsdXJlPzogYm9vbGVhbiB8IEFsbG93RmFpbHVyZTtcbiAgLyogQSBsaXN0IG9mIGZpbGVzIGFuZCBkaXJlY3RvcmllcyB0aGF0IHNob3VsZCBiZSBhdHRhY2hlZCB0byB0aGUgam9iIGlmIGl0IHN1Y2NlZWRzLiBBcnRpZmFjdHMgYXJlIHNlbnQgdG8gR2l0bGFiIHdoZXJlIHRoZXkgY2FuIGJlIGRvd25sb2FkZWQuICovXG4gIHJlYWRvbmx5IGFydGlmYWN0cz86IEFydGlmYWN0cztcbiAgLyogRGVmaW5lcyBzY3JpcHRzIHRoYXQgc2hvdWxkIHJ1biAqYmVmb3JlKiB0aGUgam9iLiAqL1xuICByZWFkb25seSBiZWZvcmVTY3JpcHQ/OiBzdHJpbmdbXTtcbiAgLyogQSBsaXN0IG9mIGZpbGVzIGFuZCBkaXJlY3RvcmllcyB0byBjYWNoZSBiZXR3ZWVuIGpvYnMuIFlvdSBjYW4gb25seSB1c2UgcGF0aHMgdGhhdCBhcmUgaW4gdGhlIGxvY2FsIHdvcmtpbmcgY29weS4gKi9cbiAgcmVhZG9ubHkgY2FjaGU/OiBDYWNoZTtcbiAgLyoqIE11c3QgYmUgYSByZWd1bGFyIGV4cHJlc3Npb24sIG9wdGlvbmFsbHkgYnV0IHJlY29tbWVuZGVkIHRvIGJlIHF1b3RlZCwgYW5kIG11c3QgYmUgc3Vycm91bmRlZCB3aXRoICcvJy4gRXhhbXBsZTogJy9Db2RlIGNvdmVyYWdlOiBcXGQrXFwuXFxkKy8nKi9cbiAgcmVhZG9ubHkgY292ZXJhZ2U/OiBzdHJpbmc7XG4gIC8qKiBTcGVjaWZ5IGEgbGlzdCBvZiBqb2IgbmFtZXMgZnJvbSBlYXJsaWVyIHN0YWdlcyBmcm9tIHdoaWNoIGFydGlmYWN0cyBzaG91bGQgYmUgbG9hZGVkLiBCeSBkZWZhdWx0LCBhbGwgcHJldmlvdXMgYXJ0aWZhY3RzIGFyZSBwYXNzZWQuIFVzZSBhbiBlbXB0eSBhcnJheSB0byBza2lwIGRvd25sb2FkaW5nIGFydGlmYWN0cy4qL1xuICByZWFkb25seSBkZXBlbmRlbmNpZXM/OiBzdHJpbmdbXTtcbiAgLyoqIFVzZWQgdG8gYXNzb2NpYXRlIGVudmlyb25tZW50IG1ldGFkYXRhIHdpdGggYSBkZXBsb3kuIEVudmlyb25tZW50IGNhbiBoYXZlIGEgbmFtZSBhbmQgVVJMIGF0dGFjaGVkIHRvIGl0LCBhbmQgd2lsbCBiZSBkaXNwbGF5ZWQgdW5kZXIgL2Vudmlyb25tZW50cyB1bmRlciB0aGUgcHJvamVjdC4qL1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IEVudmlyb25tZW50IHwgc3RyaW5nO1xuICAvKiogSm9iIHdpbGwgcnVuICpleGNlcHQqIGZvciB3aGVuIHRoZXNlIGZpbHRlcmluZyBvcHRpb25zIG1hdGNoLiovXG4gIHJlYWRvbmx5IGV4Y2VwdD86IHN0cmluZ1tdIHwgRmlsdGVyO1xuICAvKiogVGhlIG5hbWUgb2Ygb25lIG9yIG1vcmUgam9icyB0byBpbmhlcml0IGNvbmZpZ3VyYXRpb24gZnJvbS4qL1xuICByZWFkb25seSBleHRlbmRzPzogc3RyaW5nW107XG4gIC8qIFNwZWNpZmllcyB0aGUgZGVmYXVsdCBkb2NrZXIgaW1hZ2UgdG8gdXNlZCBmb3IgdGhlIGpvYi4gKi9cbiAgcmVhZG9ubHkgaW1hZ2U/OiBJbWFnZTtcbiAgLyoqIENvbnRyb2xzIGluaGVyaXRhbmNlIG9mIGdsb2JhbGx5LWRlZmluZWQgZGVmYXVsdHMgYW5kIHZhcmlhYmxlcy4gQm9vbGVhbiB2YWx1ZXMgY29udHJvbCBpbmhlcml0YW5jZSBvZiBhbGwgZGVmYXVsdDogb3IgdmFyaWFibGVzOiBrZXl3b3Jkcy4gVG8gaW5oZXJpdCBvbmx5IGEgc3Vic2V0IG9mIGRlZmF1bHQ6IG9yIHZhcmlhYmxlczoga2V5d29yZHMsIHNwZWNpZnkgd2hhdCB5b3Ugd2lzaCB0byBpbmhlcml0LiBBbnl0aGluZyBub3QgbGlzdGVkIGlzIG5vdCBpbmhlcml0ZWQuKi9cbiAgcmVhZG9ubHkgaW5oZXJpdD86IEluaGVyaXQ7XG4gIC8qIElmIGEgam9iIHNob3VsZCBiZSBjYW5jZWxlZCB3aGVuIGEgbmV3ZXIgcGlwZWxpbmUgc3RhcnRzIGJlZm9yZSB0aGUgam9iIGNvbXBsZXRlcyAoRGVmYXVsdDogZmFsc2UpLiovXG4gIHJlYWRvbmx5IGludGVycnVwdGlibGU/OiBib29sZWFuO1xuICAvKiogVGhlIGxpc3Qgb2Ygam9icyBpbiBwcmV2aW91cyBzdGFnZXMgd2hvc2Ugc29sZSBjb21wbGV0aW9uIGlzIG5lZWRlZCB0byBzdGFydCB0aGUgY3VycmVudCBqb2IuKi9cbiAgcmVhZG9ubHkgbmVlZHM/OiBBcnJheTxOZWVkIHwgc3RyaW5nPjtcbiAgLyoqIEpvYiB3aWxsIHJ1biAqb25seSogd2hlbiB0aGVzZSBmaWx0ZXJpbmcgb3B0aW9ucyBtYXRjaC4qL1xuICByZWFkb25seSBvbmx5Pzogc3RyaW5nW10gfCBGaWx0ZXI7XG4gIC8qKiBQYXJhbGxlbCB3aWxsIHNwbGl0IHVwIGEgc2luZ2xlIGpvYiBpbnRvIHNldmVyYWwsIGFuZCBwcm92aWRlIGBDSV9OT0RFX0lOREVYYCBhbmQgYENJX05PREVfVE9UQUxgIGVudmlyb25tZW50IHZhcmlhYmxlcyBmb3IgdGhlIHJ1bm5pbmcgam9icy4qL1xuICByZWFkb25seSBwYXJhbGxlbD86IFBhcmFsbGVsIHwgbnVtYmVyO1xuICAvKiogSW5kaWNhdGVzIHRoYXQgdGhlIGpvYiBjcmVhdGVzIGEgUmVsZWFzZS4qL1xuICByZWFkb25seSByZWxlYXNlPzogUmVsZWFzZTtcbiAgLyoqIExpbWl0IGpvYiBjb25jdXJyZW5jeS4gQ2FuIGJlIHVzZWQgdG8gZW5zdXJlIHRoYXQgdGhlIFJ1bm5lciB3aWxsIG5vdCBydW4gY2VydGFpbiBqb2JzIHNpbXVsdGFuZW91c2x5LiovXG4gIHJlYWRvbmx5IHJlc291cmNlR3JvdXA/OiBzdHJpbmc7XG4gIC8qIEhvdyBtYW55IHRpbWVzIGEgam9iIGlzIHJldHJpZWQgaWYgaXQgZmFpbHMuIElmIG5vdCBkZWZpbmVkLCBkZWZhdWx0cyB0byAwIGFuZCBqb2JzIGRvIG5vdCByZXRyeS4gKi9cbiAgcmVhZG9ubHkgcmV0cnk/OiBSZXRyeTtcbiAgLyoqUnVsZXMgYWxsb3dzIGZvciBhbiBhcnJheSBvZiBpbmRpdmlkdWFsIHJ1bGUgb2JqZWN0cyB0byBiZSBldmFsdWF0ZWQgaW4gb3JkZXIsIHVudGlsIG9uZSBtYXRjaGVzIGFuZCBkeW5hbWljYWxseSBwcm92aWRlcyBhdHRyaWJ1dGVzIHRvIHRoZSBqb2IuICovXG4gIHJlYWRvbmx5IHJ1bGVzPzogSW5jbHVkZVJ1bGVbXTtcbiAgLyoqIFNoZWxsIHNjcmlwdHMgZXhlY3V0ZWQgYnkgdGhlIFJ1bm5lci4gVGhlIG9ubHkgcmVxdWlyZWQgcHJvcGVydHkgb2Ygam9icy4gQmUgY2FyZWZ1bCB3aXRoIHNwZWNpYWwgY2hhcmFjdGVycyAoZS5nLiBgOmAsIGB7YCwgYH1gLCBgJmApIGFuZCB1c2Ugc2luZ2xlIG9yIGRvdWJsZSBxdW90ZXMgdG8gYXZvaWQgaXNzdWVzLiovXG4gIHJlYWRvbmx5IHNjcmlwdD86IHN0cmluZ1tdO1xuICAvKiogQ0kvQ0Qgc2VjcmV0cyAqL1xuICByZWFkb25seSBzZWNyZXRzPzogUmVjb3JkPHN0cmluZywgUmVjb3JkPHN0cmluZywgU2VjcmV0Pj47XG4gIC8qIEFkZGl0aW9uYWwgRG9ja2VyIGltYWdlcyB0byBydW4gc2NyaXB0cyBpbi4gVGhlIHNlcnZpY2UgaW1hZ2UgaXMgbGlua2VkIHRvIHRoZSBpbWFnZSBzcGVjaWZpZWQgaW4gdGhlICBpbWFnZSBwYXJhbWV0ZXIuICovXG4gIHJlYWRvbmx5IHNlcnZpY2VzPzogU2VydmljZVtdO1xuICAvKiogRGVmaW5lIHdoYXQgc3RhZ2UgdGhlIGpvYiB3aWxsIHJ1biBpbi4qL1xuICByZWFkb25seSBzdGFnZT86IHN0cmluZztcbiAgLyogRXhlY3V0ZSBzY3JpcHRzIGFmdGVyIGEgd2FpdGluZyBwZXJpb2Qgd3JpdHRlbiBpbiBuYXR1cmFsIGxhbmd1YWdlIChFeC4gb25lIGhvdXIsIDM2MDAgc2Vjb25kcywgNjAgbWludXRlcykuICovXG4gIHJlYWRvbmx5IHN0YXJ0SW4/OiBzdHJpbmc7XG4gIC8qIFVzZWQgdG8gc2VsZWN0IGEgc3BlY2lmaWMgcnVubmVyIGZyb20gdGhlIGxpc3Qgb2YgYWxsIHJ1bm5lcnMgdGhhdCBhcmUgYXZhaWxhYmxlIGZvciB0aGUgcHJvamVjdC4gKi9cbiAgcmVhZG9ubHkgdGFncz86IHN0cmluZ1tdO1xuICAvKiBBIGRlZmF1bHQgdGltZW91dCBqb2Igd3JpdHRlbiBpbiBuYXR1cmFsIGxhbmd1YWdlIChFeC4gb25lIGhvdXIsIDM2MDAgc2Vjb25kcywgNjAgbWludXRlcykuICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBzdHJpbmc7XG4gIC8qKiBUcmlnZ2VyIGFsbG93cyB5b3UgdG8gZGVmaW5lIGRvd25zdHJlYW0gcGlwZWxpbmUgdHJpZ2dlci4gV2hlbiBhIGpvYiBjcmVhdGVkIGZyb20gdHJpZ2dlciBkZWZpbml0aW9uIGlzIHN0YXJ0ZWQgYnkgR2l0TGFiLCBhIGRvd25zdHJlYW0gcGlwZWxpbmUgZ2V0cyBjcmVhdGVkLiBSZWFkIG1vcmU6IGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvUkVBRE1FLmh0bWwjdHJpZ2dlciovXG4gIHJlYWRvbmx5IHRyaWdnZXI/OiBUcmlnZ2VyIHwgc3RyaW5nO1xuICAvKiogQ29uZmlndXJhYmxlIHZhbHVlcyB0aGF0IGFyZSBwYXNzZWQgdG8gdGhlIEpvYi4gKi9cbiAgcmVhZG9ubHkgdmFyaWFibGVzPzogUmVjb3JkPHN0cmluZywgbnVtYmVyIHwgc3RyaW5nPjtcbiAgLyoqIERlc2NyaWJlcyB0aGUgY29uZGl0aW9ucyBmb3Igd2hlbiB0byBydW4gdGhlIGpvYi4gRGVmYXVsdHMgdG8gJ29uX3N1Y2Nlc3MnLiAqL1xuICByZWFkb25seSB3aGVuPzogSm9iV2hlbjtcbn1cblxuLyoqXG4gKiBUaGUgZW52aXJvbm1lbnQgdGhhdCBhIGpvYiBkZXBsb3lzIHRvLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVudmlyb25tZW50IHtcbiAgLyoqIFNwZWNpZmllcyB3aGF0IHRoaXMgam9iIHdpbGwgZG8uICdzdGFydCcgKGRlZmF1bHQpIGluZGljYXRlcyB0aGUgam9iIHdpbGwgc3RhcnQgdGhlIGRlcGxveW1lbnQuICdwcmVwYXJlJyBpbmRpY2F0ZXMgdGhpcyB3aWxsIG5vdCBhZmZlY3QgdGhlIGRlcGxveW1lbnQuICdzdG9wJyBpbmRpY2F0ZXMgdGhpcyB3aWxsIHN0b3AgdGhlIGRlcGxveW1lbnQuKi9cbiAgcmVhZG9ubHkgYWN0aW9uPzogQWN0aW9uO1xuICAvKiogVGhlIGFtb3VudCBvZiB0aW1lIGl0IHNob3VsZCB0YWtlIGJlZm9yZSBHaXRsYWIgd2lsbCBhdXRvbWF0aWNhbGx5IHN0b3AgdGhlIGVudmlyb25tZW50LiBTdXBwb3J0cyBhIHdpZGUgdmFyaWV0eSBvZiBmb3JtYXRzLCBlLmcuICcxIHdlZWsnLCAnMyBtaW5zIDQgc2VjJywgJzIgaHJzIDIwIG1pbicsICcyaDIwbWluJywgJzYgbW9zIDEgZGF5JywgJzQ3IHlycyA2IG1vcyBhbmQgNGQnLCAnMyB3ZWVrcyBhbmQgMiBkYXlzJy4qL1xuICByZWFkb25seSBhdXRvU3RvcEluPzogc3RyaW5nO1xuICAvKiogRXhwbGljaXRseSBzcGVjaWZpZXMgdGhlIHRpZXIgb2YgdGhlIGRlcGxveW1lbnQgZW52aXJvbm1lbnQgaWYgbm9uLXN0YW5kYXJkIGVudmlyb25tZW50IG5hbWUgaXMgdXNlZC4qL1xuICByZWFkb25seSBkZXBsb3ltZW50VGllcj86IERlcGxveW1lbnRUaWVyO1xuICAvKiogVXNlZCB0byBjb25maWd1cmUgdGhlIGt1YmVybmV0ZXMgZGVwbG95bWVudCBmb3IgdGhpcyBlbnZpcm9ubWVudC4gVGhpcyBpcyBjdXJyZW50bHkgbm90IHN1cHBvcnRlZCBmb3Iga3ViZXJuZXRlcyBjbHVzdGVycyB0aGF0IGFyZSBtYW5hZ2VkIGJ5IEdpdGxhYi4qL1xuICByZWFkb25seSBrdWJlcm5ldGVzPzogS3ViZXJuZXRlc0NvbmZpZztcbiAgLyoqIFRoZSBuYW1lIG9mIHRoZSBlbnZpcm9ubWVudCwgZS5nLiAncWEnLCAnc3RhZ2luZycsICdwcm9kdWN0aW9uJy4qL1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIC8qKiBUaGUgbmFtZSBvZiBhIGpvYiB0byBleGVjdXRlIHdoZW4gdGhlIGVudmlyb25tZW50IGlzIGFib3V0IHRvIGJlIHN0b3BwZWQuKi9cbiAgcmVhZG9ubHkgb25TdG9wPzogc3RyaW5nO1xuICAvKiogV2hlbiBzZXQsIHRoaXMgd2lsbCBleHBvc2UgYnV0dG9ucyBpbiB2YXJpb3VzIHBsYWNlcyBmb3IgdGhlIGN1cnJlbnQgZW52aXJvbm1lbnQgaW4gR2l0bGFiLCB0aGF0IHdpbGwgdGFrZSB5b3UgdG8gdGhlIGRlZmluZWQgVVJMLiovXG4gIHJlYWRvbmx5IHVybD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBTcGVjaWZpZXMgd2hhdCB0aGlzIGpvYiB3aWxsIGRvLiAnc3RhcnQnIChkZWZhdWx0KSBpbmRpY2F0ZXMgdGhlIGpvYiB3aWxsIHN0YXJ0IHRoZVxuICogZGVwbG95bWVudC4gJ3ByZXBhcmUnIGluZGljYXRlcyB0aGlzIHdpbGwgbm90IGFmZmVjdCB0aGUgZGVwbG95bWVudC4gJ3N0b3AnIGluZGljYXRlc1xuICogdGhpcyB3aWxsIHN0b3AgdGhlIGRlcGxveW1lbnQuXG4gKi9cbmV4cG9ydCBlbnVtIEFjdGlvbiB7XG4gIFBSRVBBUkUgPSBcInByZXBhcmVcIixcbiAgU1RBUlQgPSBcInN0YXJ0XCIsXG4gIFNUT1AgPSBcInN0b3BcIixcbn1cblxuLyoqXG4gKiBFeHBsaWNpdGx5IHNwZWNpZmllcyB0aGUgdGllciBvZiB0aGUgZGVwbG95bWVudCBlbnZpcm9ubWVudCBpZiBub24tc3RhbmRhcmQgZW52aXJvbm1lbnRcbiAqIG5hbWUgaXMgdXNlZC5cbiAqL1xuZXhwb3J0IGVudW0gRGVwbG95bWVudFRpZXIge1xuICBERVZFTE9QTUVOVCA9IFwiZGV2ZWxvcG1lbnRcIixcbiAgT1RIRVIgPSBcIm90aGVyXCIsXG4gIFBST0RVQ1RJT04gPSBcInByb2R1Y3Rpb25cIixcbiAgU1RBR0lORyA9IFwic3RhZ2luZ1wiLFxuICBURVNUSU5HID0gXCJ0ZXN0aW5nXCIsXG59XG5cbi8qKlxuICogVXNlZCB0byBjb25maWd1cmUgdGhlIGt1YmVybmV0ZXMgZGVwbG95bWVudCBmb3IgdGhpcyBlbnZpcm9ubWVudC4gVGhpcyBpcyBjdXJyZW50bHkgbm90XG4gKiBzdXBwb3J0ZWQgZm9yIGt1YmVybmV0ZXMgY2x1c3RlcnMgdGhhdCBhcmUgbWFuYWdlZCBieSBHaXRsYWIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgS3ViZXJuZXRlc0NvbmZpZyB7XG4gIC8qKiBUaGUga3ViZXJuZXRlcyBuYW1lc3BhY2Ugd2hlcmUgdGhpcyBlbnZpcm9ubWVudCBzaG91bGQgYmUgZGVwbG95ZWQgdG8uKi9cbiAgcmVhZG9ubHkgbmFtZXNwYWNlPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEZpbHRlcmluZyBvcHRpb25zIGZvciB3aGVuIGEgam9iIHdpbGwgcnVuLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZpbHRlciB7XG4gIC8qKiBGaWx0ZXIgam9iIGNyZWF0aW9uIGJhc2VkIG9uIGZpbGVzIHRoYXQgd2VyZSBtb2RpZmllZCBpbiBhIGdpdCBwdXNoLiovXG4gIHJlYWRvbmx5IGNoYW5nZXM/OiBzdHJpbmdbXTtcbiAgLyoqIEZpbHRlciBqb2IgYmFzZWQgb24gaWYgS3ViZXJuZXRlcyBpbnRlZ3JhdGlvbiBpcyBhY3RpdmUuKi9cbiAgcmVhZG9ubHkga3ViZXJuZXRlcz86IEt1YmVybmV0ZXNFbnVtO1xuICAvKiogQ29udHJvbCB3aGVuIHRvIGFkZCBqb2JzIHRvIGEgcGlwZWxpbmUgYmFzZWQgb24gYnJhbmNoIG5hbWVzIG9yIHBpcGVsaW5lIHR5cGVzLiAqL1xuICByZWFkb25seSByZWZzPzogc3RyaW5nW107XG4gIC8qKiBGaWx0ZXIgam9iIGJ5IGNoZWNraW5nIGNvbXBhcmluZyB2YWx1ZXMgb2YgZW52aXJvbm1lbnQgdmFyaWFibGVzLiBSZWFkIG1vcmUgYWJvdXQgdmFyaWFibGUgZXhwcmVzc2lvbnM6IGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3ZhcmlhYmxlcy9SRUFETUUuaHRtbCN2YXJpYWJsZXMtZXhwcmVzc2lvbnMqL1xuICByZWFkb25seSB2YXJpYWJsZXM/OiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBGaWx0ZXIgam9iIGJhc2VkIG9uIGlmIEt1YmVybmV0ZXMgaW50ZWdyYXRpb24gaXMgYWN0aXZlLlxuICovXG5leHBvcnQgZW51bSBLdWJlcm5ldGVzRW51bSB7XG4gIEFDVElWRSA9IFwiYWN0aXZlXCIsXG59XG5cbi8qKlxuICogQ29udHJvbHMgaW5oZXJpdGFuY2Ugb2YgZ2xvYmFsbHktZGVmaW5lZCBkZWZhdWx0cyBhbmQgdmFyaWFibGVzLiBCb29sZWFuIHZhbHVlcyBjb250cm9sXG4gKiBpbmhlcml0YW5jZSBvZiBhbGwgZGVmYXVsdDogb3IgdmFyaWFibGVzOiBrZXl3b3Jkcy4gVG8gaW5oZXJpdCBvbmx5IGEgc3Vic2V0IG9mIGRlZmF1bHQ6XG4gKiBvciB2YXJpYWJsZXM6IGtleXdvcmRzLCBzcGVjaWZ5IHdoYXQgeW91IHdpc2ggdG8gaW5oZXJpdC4gQW55dGhpbmcgbm90IGxpc3RlZCBpcyBub3RcbiAqIGluaGVyaXRlZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbmhlcml0IHtcbiAgLyoqIFdoZXRoZXIgdG8gaW5oZXJpdCBhbGwgZ2xvYmFsbHktZGVmaW5lZCBkZWZhdWx0cyBvciBub3QuIE9yIHN1YnNldCBvZiBpbmhlcml0ZWQgZGVmYXVsdHMqL1xuICByZWFkb25seSBkZWZhdWx0PzogRGVmYXVsdEVsZW1lbnRbXSB8IGJvb2xlYW47XG4gIC8qKiBXaGV0aGVyIHRvIGluaGVyaXQgYWxsIGdsb2JhbGx5LWRlZmluZWQgdmFyaWFibGVzIG9yIG5vdC4gT3Igc3Vic2V0IG9mIGluaGVyaXRlZCB2YXJpYWJsZXMqL1xuICByZWFkb25seSB2YXJpYWJsZXM/OiBzdHJpbmdbXSB8IGJvb2xlYW47XG59XG5cbmV4cG9ydCBlbnVtIERlZmF1bHRFbGVtZW50IHtcbiAgQUZURVJfU0NSSVBUID0gXCJhZnRlcl9zY3JpcHRcIixcbiAgQVJUSUZBQ1RTID0gXCJhcnRpZmFjdHNcIixcbiAgQkVGT1JFX1NDUklQVCA9IFwiYmVmb3JlX3NjcmlwdFwiLFxuICBDQUNIRSA9IFwiY2FjaGVcIixcbiAgSU1BR0UgPSBcImltYWdlXCIsXG4gIElOVEVSUlVQVElCTEUgPSBcImludGVycnVwdGlibGVcIixcbiAgUkVUUlkgPSBcInJldHJ5XCIsXG4gIFNFUlZJQ0VTID0gXCJzZXJ2aWNlc1wiLFxuICBUQUdTID0gXCJ0YWdzXCIsXG4gIFRJTUVPVVQgPSBcInRpbWVvdXRcIixcbn1cblxuLyoqXG4gKiBBIGpvYnMgaW4gYSBwcmV2aW91cyBzdGFnZSB3aG9zZSBzb2xlIGNvbXBsZXRpb24gaXMgbmVlZGVkIHRvIHN0YXJ0IHRoZSBjdXJyZW50IGpvYi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZWVkIHtcbiAgcmVhZG9ubHkgYXJ0aWZhY3RzPzogYm9vbGVhbjtcbiAgcmVhZG9ubHkgam9iOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG9wdGlvbmFsPzogYm9vbGVhbjtcbiAgcmVhZG9ubHkgcGlwZWxpbmU/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHByb2plY3Q/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHJlZj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBVc2VkIHRvIHJ1biBhIGpvYiBtdWx0aXBsZSB0aW1lcyBpbiBwYXJhbGxlbCBpbiBhIHNpbmdsZSBwaXBlbGluZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQYXJhbGxlbCB7XG4gIC8qKiBEZWZpbmVzIGRpZmZlcmVudCB2YXJpYWJsZXMgZm9yIGpvYnMgdGhhdCBhcmUgcnVubmluZyBpbiBwYXJhbGxlbC4qL1xuICByZWFkb25seSBtYXRyaXg6IFJlY29yZDxzdHJpbmcsIGFueVtdPltdO1xufVxuXG4vKipcbiAqIEluZGljYXRlcyB0aGF0IHRoZSBqb2IgY3JlYXRlcyBhIFJlbGVhc2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVsZWFzZSB7XG4gIHJlYWRvbmx5IGFzc2V0cz86IEFzc2V0cztcbiAgLyoqIFNwZWNpZmllcyB0aGUgbG9uZ2VyIGRlc2NyaXB0aW9uIG9mIHRoZSBSZWxlYXNlLiovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBzdHJpbmc7XG4gIC8qKiBUaGUgdGl0bGUgb2YgZWFjaCBtaWxlc3RvbmUgdGhlIHJlbGVhc2UgaXMgYXNzb2NpYXRlZCB3aXRoLiovXG4gIHJlYWRvbmx5IG1pbGVzdG9uZXM/OiBzdHJpbmdbXTtcbiAgLyoqIFRoZSBSZWxlYXNlIG5hbWUuIElmIG9taXR0ZWQsIGl0IGlzIHBvcHVsYXRlZCB3aXRoIHRoZSB2YWx1ZSBvZiByZWxlYXNlOiB0YWdfbmFtZS4qL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuICAvKiogSWYgdGhlIHJlbGVhc2U6IHRhZ19uYW1lIGRvZXNu4oCZdCBleGlzdCB5ZXQsIHRoZSByZWxlYXNlIGlzIGNyZWF0ZWQgZnJvbSByZWYuIHJlZiBjYW4gYmUgYSBjb21taXQgU0hBLCBhbm90aGVyIHRhZyBuYW1lLCBvciBhIGJyYW5jaCBuYW1lLiovXG4gIHJlYWRvbmx5IHJlZj86IHN0cmluZztcbiAgLyoqIFRoZSBkYXRlIGFuZCB0aW1lIHdoZW4gdGhlIHJlbGVhc2UgaXMgcmVhZHkuIERlZmF1bHRzIHRvIHRoZSBjdXJyZW50IGRhdGUgYW5kIHRpbWUgaWYgbm90IGRlZmluZWQuIFNob3VsZCBiZSBlbmNsb3NlZCBpbiBxdW90ZXMgYW5kIGV4cHJlc3NlZCBpbiBJU08gODYwMSBmb3JtYXQuKi9cbiAgcmVhZG9ubHkgcmVsZWFzZWRBdD86IHN0cmluZztcbiAgLyoqIFRoZSB0YWdfbmFtZSBtdXN0IGJlIHNwZWNpZmllZC4gSXQgY2FuIHJlZmVyIHRvIGFuIGV4aXN0aW5nIEdpdCB0YWcgb3IgY2FuIGJlIHNwZWNpZmllZCBieSB0aGUgdXNlci4qL1xuICByZWFkb25seSB0YWdOYW1lOiBzdHJpbmc7XG59XG4vKipcbiAqIEFzc2V0IGNvbmZpZ3VyYXRpb24gZm9yIGEgcmVsZWFzZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBc3NldHMge1xuICAvKiogSW5jbHVkZSBhc3NldCBsaW5rcyBpbiB0aGUgcmVsZWFzZS4qL1xuICByZWFkb25seSBsaW5rczogTGlua1tdO1xufVxuXG4vKipcbiAqIExpbmsgY29uZmlndXJhdGlvbiBmb3IgYW4gYXNzZXQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGluayB7XG4gIC8qKiBUaGUgcmVkaXJlY3QgbGluayB0byB0aGUgdXJsLiovXG4gIHJlYWRvbmx5IGZpbGVwYXRoPzogc3RyaW5nO1xuICAvKiogVGhlIGNvbnRlbnQga2luZCBvZiB3aGF0IHVzZXJzIGNhbiBkb3dubG9hZCB2aWEgdXJsLiovXG4gIHJlYWRvbmx5IGxpbmtUeXBlPzogTGlua1R5cGU7XG4gIC8qKiBUaGUgbmFtZSBvZiB0aGUgbGluay4qL1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIC8qKiBUaGUgVVJMIHRvIGRvd25sb2FkIGEgZmlsZS4qL1xuICByZWFkb25seSB1cmw6IHN0cmluZztcbn1cblxuLyoqXG4gKiBUaGUgY29udGVudCBraW5kIG9mIHdoYXQgdXNlcnMgY2FuIGRvd25sb2FkIHZpYSB1cmwuXG4gKi9cbmV4cG9ydCBlbnVtIExpbmtUeXBlIHtcbiAgSU1BR0UgPSBcImltYWdlXCIsXG4gIE9USEVSID0gXCJvdGhlclwiLFxuICBQQUNLQUdFID0gXCJwYWNrYWdlXCIsXG4gIFJVTkJPT0sgPSBcInJ1bmJvb2tcIixcbn1cblxuLyoqXG4gKiBBIENJL0NEIHNlY3JldFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlY3JldCB7XG4gIC8qIFNwZWNpZmljYXRpb24gZm9yIGEgc2VjcmV0IHByb3ZpZGVkIGJ5IGEgSGFzaGlDb3JwIFZhdWx0LiAqL1xuICByZWFkb25seSB2YXVsdDogVmF1bHRDb25maWc7XG59XG5cbi8qKlxuICogU3BlY2lmaWNhdGlvbiBmb3IgYSBzZWNyZXQgcHJvdmlkZWQgYnkgYSBIYXNoaUNvcnAgVmF1bHQuXG4gKiBAc2VlIGh0dHBzOi8vd3d3LnZhdWx0cHJvamVjdC5pby9cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWYXVsdENvbmZpZyB7XG4gIC8qIFRoZSBlbmdpbmUgY29uZmlndXJhdGlvbiBmb3IgYSBzZWNyZXQuICovXG4gIHJlYWRvbmx5IGVuZ2luZTogRW5naW5lO1xuICAvKiBUaGUgbmFtZSBvZiB0aGUgZmllbGQgd2hlcmUgdGhlIHBhc3N3b3JkIGlzIHN0b3JlZC4gKi9cbiAgcmVhZG9ubHkgZmllbGQ6IHN0cmluZztcbiAgLyoqIFBhdGggdG8gdGhlIHNlY3JldC4gKi9cbiAgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xufVxuLyoqXG4gKiBUaGUgZW5naW5lIGNvbmZpZ3VyYXRpb24gZm9yIGEgc2VjcmV0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVuZ2luZSB7XG4gIC8qKiBOYW1lIG9mIHRoZSBzZWNyZXRzIGVuZ2luZS4gKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAvKiogUGF0aCB0byB0aGUgc2VjcmV0cyBlbmdpbmUuICovXG4gIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcbn1cblxuLyoqXG4gKiBUcmlnZ2VyIGEgbXVsdGktcHJvamVjdCBvciBhIGNoaWxkIHBpcGVsaW5lLiBSZWFkIG1vcmU6XG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvUkVBRE1FLmh0bWwjc2ltcGxlLXRyaWdnZXItc3ludGF4LWZvci1tdWx0aS1wcm9qZWN0LXBpcGVsaW5lc1xuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sL1JFQURNRS5odG1sI3RyaWdnZXItc3ludGF4LWZvci1jaGlsZC1waXBlbGluZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRyaWdnZXIge1xuICAvKiogVGhlIGJyYW5jaCBuYW1lIHRoYXQgYSBkb3duc3RyZWFtIHBpcGVsaW5lIHdpbGwgdXNlKi9cbiAgcmVhZG9ubHkgYnJhbmNoPzogc3RyaW5nO1xuICAvKiogUGF0aCB0byB0aGUgcHJvamVjdCwgZS5nLiBgZ3JvdXAvcHJvamVjdGAsIG9yIGBncm91cC9zdWItZ3JvdXAvcHJvamVjdGAuKi9cbiAgcmVhZG9ubHkgcHJvamVjdD86IHN0cmluZztcbiAgLyoqIFlvdSBjYW4gbWlycm9yIHRoZSBwaXBlbGluZSBzdGF0dXMgZnJvbSB0aGUgdHJpZ2dlcmVkIHBpcGVsaW5lIHRvIHRoZSBzb3VyY2UgYnJpZGdlIGpvYiBieSB1c2luZyBzdHJhdGVneTogZGVwZW5kKi9cbiAgcmVhZG9ubHkgc3RyYXRlZ3k/OiBTdHJhdGVneTtcbiAgLyoqIEEgbGlzdCBvZiBsb2NhbCBmaWxlcyBvciBhcnRpZmFjdHMgZnJvbSBvdGhlciBqb2JzIHRvIGRlZmluZSB0aGUgcGlwZWxpbmUgKi9cbiAgcmVhZG9ubHkgaW5jbHVkZT86IFRyaWdnZXJJbmNsdWRlW107XG59XG5cbi8qKlxuICogUmVmZXJlbmNlcyBhIGxvY2FsIGZpbGUgb3IgYW4gYXJ0aWZhY3QgZnJvbSBhbm90aGVyIGpvYiB0byBkZWZpbmUgdGhlIHBpcGVsaW5lXG4gKiBjb25maWd1cmF0aW9uLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyN0cmlnZ2VyaW5jbHVkZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRyaWdnZXJJbmNsdWRlIHtcbiAgLyoqIFJlbGF0aXZlIHBhdGggZnJvbSBsb2NhbCByZXBvc2l0b3J5IHJvb3QgKGAvYCkgdG8gdGhlIGxvY2FsIFlBTUwgZmlsZSB0byBkZWZpbmUgdGhlIHBpcGVsaW5lIGNvbmZpZ3VyYXRpb24uKi9cbiAgcmVhZG9ubHkgbG9jYWw/OiBzdHJpbmc7XG4gIC8qKiBOYW1lIG9mIHRoZSB0ZW1wbGF0ZSBZQU1MIGZpbGUgdG8gdXNlIGluIHRoZSBwaXBlbGluZSBjb25maWd1cmF0aW9uLiovXG4gIHJlYWRvbmx5IHRlbXBsYXRlPzogc3RyaW5nO1xuICAvKiogUmVsYXRpdmUgcGF0aCB0byB0aGUgZ2VuZXJhdGVkIFlBTUwgZmlsZSB3aGljaCBpcyBleHRyYWN0ZWQgZnJvbSB0aGUgYXJ0aWZhY3RzIGFuZCB1c2VkIGFzIHRoZSBjb25maWd1cmF0aW9uIGZvciB0cmlnZ2VyaW5nIHRoZSBjaGlsZCBwaXBlbGluZS4qL1xuICByZWFkb25seSBhcnRpZmFjdD86IHN0cmluZztcbiAgLyoqIEpvYiBuYW1lIHdoaWNoIGdlbmVyYXRlcyB0aGUgYXJ0aWZhY3QqL1xuICByZWFkb25seSBqb2I/OiBzdHJpbmc7XG4gIC8qKiBSZWxhdGl2ZSBwYXRoIGZyb20gcmVwb3NpdG9yeSByb290IChgL2ApIHRvIHRoZSBwaXBlbGluZSBjb25maWd1cmF0aW9uIFlBTUwgZmlsZS4qL1xuICByZWFkb25seSBmaWxlPzogc3RyaW5nO1xuICAvKiogUGF0aCB0byBhbm90aGVyIHByaXZhdGUgcHJvamVjdCB1bmRlciB0aGUgc2FtZSBHaXRMYWIgaW5zdGFuY2UsIGxpa2UgYGdyb3VwL3Byb2plY3RgIG9yIGBncm91cC9zdWItZ3JvdXAvcHJvamVjdGAuKi9cbiAgcmVhZG9ubHkgcHJvamVjdD86IHN0cmluZztcbiAgLyoqIEJyYW5jaC9UYWcvQ29tbWl0IGhhc2ggZm9yIHRoZSB0YXJnZXQgcHJvamVjdC4qL1xuICByZWFkb25seSByZWY/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogWW91IGNhbiBtaXJyb3IgdGhlIHBpcGVsaW5lIHN0YXR1cyBmcm9tIHRoZSB0cmlnZ2VyZWQgcGlwZWxpbmUgdG8gdGhlIHNvdXJjZSBicmlkZ2Ugam9iXG4gKiBieSB1c2luZyBzdHJhdGVneTogZGVwZW5kXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvI3RyaWdnZXJzdHJhdGVneVxuICovXG5leHBvcnQgZW51bSBTdHJhdGVneSB7XG4gIERFUEVORCA9IFwiZGVwZW5kXCIsXG59XG5cbi8qKlxuICogRXhwbGFpbnMgd2hhdCB0aGUgZ2xvYmFsIHZhcmlhYmxlIGlzIHVzZWQgZm9yLCB3aGF0IHRoZSBhY2NlcHRhYmxlIHZhbHVlcyBhcmUuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvI3ZhcmlhYmxlc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFZhcmlhYmxlQ29uZmlnIHtcbiAgLyoqIERlZmluZSBhIGdsb2JhbCB2YXJpYWJsZSB0aGF0IGlzIHByZWZpbGxlZCB3aGVuIHJ1bm5pbmcgYSBwaXBlbGluZSBtYW51YWxseS4gTXVzdCBiZSB1c2VkIHdpdGggdmFsdWUuICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbiAgLyoqIFRoZSB2YXJpYWJsZSB2YWx1ZS4gKi9cbiAgcmVhZG9ubHkgdmFsdWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogVXNlZCB0byBjb250cm9sIHBpcGVsaW5lIGJlaGF2aW9yLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sLyN3b3JrZmxvd1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFdvcmtmbG93IHtcbiAgLyoqIFVzZWQgdG8gY29udHJvbCB3aGV0aGVyIG9yIG5vdCBhIHdob2xlIHBpcGVsaW5lIGlzIGNyZWF0ZWQuICovXG4gIHJlYWRvbmx5IHJ1bGVzPzogV29ya2Zsb3dSdWxlW107XG59XG5cbi8qKlxuICogVXNlZCB0byBjb250cm9sIHdoZXRoZXIgb3Igbm90IGEgd2hvbGUgcGlwZWxpbmUgaXMgY3JlYXRlZC5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGxhYi5jb20vZWUvY2kveWFtbC8jd29ya2Zsb3dydWxlc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFdvcmtmbG93UnVsZSB7XG4gIC8qIFNwZWNpZnkgd2hlbiB0byBhZGQgYSBqb2IgdG8gYSBwaXBlbGluZSBieSBjaGVja2luZyBmb3IgY2hhbmdlcyB0byBzcGVjaWZpYyBmaWxlcy4gKi9cbiAgcmVhZG9ubHkgY2hhbmdlcz86IHN0cmluZ1tdO1xuICAvKiBSdW4gYSBqb2Igd2hlbiBjZXJ0YWluIGZpbGVzIGV4aXN0IGluIHRoZSByZXBvc2l0b3J5LiAqL1xuICByZWFkb25seSBleGlzdHM/OiBzdHJpbmdbXTtcbiAgLyogQ2xhdXNlcyB0byBzcGVjaWZ5IHdoZW4gdG8gYWRkIGEgam9iIHRvIGEgcGlwZWxpbmUuKi9cbiAgcmVhZG9ubHkgaWY/OiBzdHJpbmc7XG4gIC8qIFVzZSB2YXJpYWJsZXMgaW4gcnVsZXMgdG8gZGVmaW5lIHZhcmlhYmxlcyBmb3Igc3BlY2lmaWMgY29uZGl0aW9ucy4gKi9cbiAgcmVhZG9ubHkgdmFyaWFibGVzPzogUmVjb3JkPHN0cmluZywgbnVtYmVyIHwgc3RyaW5nPjtcbiAgLyogQ29uZGl0aW9ucyBmb3Igd2hlbiB0byBydW4gdGhlIGpvYi4gRGVmYXVsdHMgdG8gJ29uX3N1Y2Nlc3MnICovXG4gIHJlYWRvbmx5IHdoZW4/OiBKb2JXaGVuO1xufVxuXG4vKipcbiAqIERlc2NyaWJlcyB0aGUgY29uZGl0aW9ucyBmb3Igd2hlbiB0byBydW4gdGhlIGpvYi4gRGVmYXVsdHMgdG8gJ29uX3N1Y2Nlc3MnLlxuICogVGhlIHZhbHVlIGNhbiBvbmx5IGJlICdhbHdheXMnIG9yICduZXZlcicgd2hlbiB1c2VkIHdpdGggd29ya2Zsb3cuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL3lhbWwvI3dvcmtmbG93cnVsZXNcbiAqL1xuZXhwb3J0IGVudW0gV29ya2Zsb3dXaGVuIHtcbiAgQUxXQVlTID0gXCJhbHdheXNcIixcbiAgTkVWRVIgPSBcIm5ldmVyXCIsXG59XG4iXX0=