"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const case_1 = require("case");
const fs = require("fs-extra");
const PROJEN_MODULE_ROOT = path.join(__dirname, "..");
const PROJECT_BASE_FQN = "projen.Project";
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        if (isProjectType(jsii, fqn)) {
            const p = toProjectType(jsii, fqn);
            result.push(p);
        }
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, ".jsii");
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes("@") && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map((file) => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [process.cwd()],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        throw new Error(`Fully qualified name "${fqn}" is not a valid project type.`);
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf(".") + 1);
    const docsurl = `https://projen.io/api/API.html#projen-${typename
        .toLocaleLowerCase()
        .replace(/\./g, "-")}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : case_1.snake(typename).replace(/_project$/, "");
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 ||
        (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== "options")) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let jsiiKind;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.fqn) {
                jsiiKind = jsii[(_c = prop.type) === null || _c === void 0 ? void 0 : _c.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_d = prop.docs) === null || _d === void 0 ? void 0 : _d.default;
            if (defaultValue === "undefined") {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_e = prop.type) === null || _e === void 0 ? void 0 : _e.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${JSON.stringify(prop.type)}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_f = prop.type) === null || _f === void 0 ? void 0 : _f.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: (_g = prop.type) === null || _g === void 0 ? void 0 : _g.fqn,
                docs: prop.docs.summary,
                simpleType: prop.type ? getSimpleTypeName(prop.type) : "unknown",
                fullType: prop.type,
                kind: jsiiKind,
                jsonLike: prop.type ? isJsonLike(jsii, prop.type) : undefined,
                switch: propPath.map((p) => case_1.snake(p).replace(/_/g, "-")).join("-"),
                default: defaultValue,
                optional: isOptional,
                featured: ((_j = (_h = prop.docs) === null || _h === void 0 ? void 0 : _h.custom) === null || _j === void 0 ? void 0 : _j.featured) === "true",
                deprecated: prop.docs.stability === "deprecated" ? true : undefined,
            });
        }
        for (const ifc of (_k = struct.interfaces) !== null && _k !== void 0 ? _k : []) {
            addOptions(ifc);
        }
    }
}
function getSimpleTypeName(type) {
    if (type === null || type === void 0 ? void 0 : type.primitive) {
        return type.primitive; // e.g. 'string', 'boolean', 'number'
    }
    else if (type === null || type === void 0 ? void 0 : type.fqn) {
        return type.fqn.split(".").pop(); // projen.NodeProjectOptions -> NodeProjectOptions
    }
    else {
        // any other types such as collection types
        return "unknown";
    }
}
/**
 * Whether a value of this type is serializable into JSON.
 */
function isJsonLike(jsii, type) {
    if (type.primitive) {
        // string, boolean, number, any
        return true;
    }
    else if (type.fqn) {
        const kind = jsii[type.fqn].kind;
        if (["interface", "enum"].includes(kind)) {
            // not 'class'
            return true;
        }
    }
    else if (type.collection) {
        return isJsonLike(jsii, type.collection.elementtype);
    }
    return false;
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (!type) {
        throw new Error(`Could not find project type with fqn "${fqn}" in  .jsii file.`);
    }
    if (type.kind !== "class") {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith("$")) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof parsed !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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