"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    "@xsi:schemaLocation": "http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd",
    "@xmlns": "http://maven.apache.org/POM/4.0.0",
    "@xmlns:xsi": "http://www.w3.org/2001/XMLSchema-instance",
};
/**
 * A Project Object Model or POM is the fundamental unit of work in Maven. It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 */
class Pom extends component_1.Component {
    constructor(project, options) {
        var _b;
        super(project);
        this.properties = {};
        this.fileName = "pom.xml";
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = (_b = options.packaging) !== null && _b !== void 0 ? _b : "jar";
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom() });
    }
    /**
     * Adds a key/value property to the pom.
     * @param key the key
     * @param value the value
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.TEST);
    }
    /**
     * Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`)
     * @param options plugin options
     */
    addPlugin(spec, options = {}) {
        var _b;
        for (const dep of (_b = options.dependencies) !== null && _b !== void 0 ? _b : []) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, dependencies_1.DependencyType.BUILD, options);
    }
    synthPom() {
        return _resolve_1.resolve({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: "4.0.0",
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                ...this.synthDependencies(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case dependencies_1.DependencyType.PEER:
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case dependencies_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: "test",
                    });
                    break;
                // build maps to plugins
                case dependencies_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
}
exports.Pom = Pom;
_a = JSII_RTTI_SYMBOL_1;
Pom[_a] = { fqn: "projen.java.Pom", version: "0.52.53" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split("/");
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? semver_1.toMavenVersionRange(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    var _b, _c;
    return {
        configuration: options.configuration,
        dependencies: options.dependencies && options.dependencies.length > 0
            ? {
                dependency: (_b = options.dependencies) === null || _b === void 0 ? void 0 : _b.map((d) => mavenCoords(dependencies_1.Dependencies.parseDependency(d))),
            }
            : undefined,
        executions: (_c = options.executions) === null || _c === void 0 ? void 0 : _c.map((e) => ({
            execution: {
                id: e.id,
                goals: e.goals.map((goal) => ({ goal })),
            },
        })),
    };
}
//# sourceMappingURL=data:application/json;base64,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