"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 */
class Projenrc extends component_1.Component {
    constructor(project, pom, options = {}) {
        var _b, _c, _d, _e, _f;
        super(project);
        const projenVersion = (_b = options.projenVersion) !== null && _b !== void 0 ? _b : common_1.PROJEN_VERSION;
        this.className = (_c = options.className) !== null && _c !== void 0 ? _c : "projenrc";
        this.testScope = (_d = options.testScope) !== null && _d !== void 0 ? _d : true;
        const depType = this.testScope
            ? dependencies_1.DependencyType.TEST
            : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : "";
        const compileGoal = this.testScope
            ? "compiler:testCompile"
            : "compiler:compile";
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin("org.codehaus.mojo/exec-maven-plugin@3.0.0");
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        (_e = project.defaultTask) === null || _e === void 0 ? void 0 : _e.exec(`mvn ${compileGoal} --quiet`);
        (_f = project.defaultTask) === null || _f === void 0 ? void 0 : _f.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    generateProjenrc() {
        var _b, _c, _d;
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = (_d = (_c = (_b = jsiiType.initializer) === null || _b === void 0 ? void 0 : _b.parameters) === null || _c === void 0 ? void 0 : _c[0].type) === null || _d === void 0 ? void 0 : _d.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        const dir = this.testScope ? "src/test/java" : "src/main/java";
        const split = this.className.split(".");
        let javaClass, javaPackage;
        if (split.length === 1) {
            javaClass = split[0];
            javaPackage = [];
        }
        else {
            javaPackage = split.slice(0, split.length - 2);
            javaClass = split[split.length - 1];
        }
        const javaFile = path_1.join(this.project.outdir, dir, ...javaPackage, javaClass + ".java");
        // skip if file exists
        if (fs_extra_1.existsSync(javaFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        const openBlock = (line = "") => {
            emit(line + " {");
            indent++;
        };
        const closeBlock = () => {
            indent--;
            emit("}");
        };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
            }
        }
        if (javaPackage.length > 0) {
            emit(`package ${javaPackage.join(".")};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiType)};`);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiOptionsType)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${javaClass}`);
        openBlock("public static void main(String[] args)");
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit("project.synth();");
        closeBlock();
        closeBlock();
        fs_extra_1.mkdirpSync(path_1.dirname(javaFile));
        fs_extra_1.writeFileSync(javaFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${javaFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.52.53" };
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: "", imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push(".build()");
    const renderedOptions = lines.join(`\n${" ".repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === "string" && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split(".");
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, "_");
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter((x) => x).join(".");
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvamVucmMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvamF2YS9wcm9qZW5yYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLCtCQUFxQztBQUNyQyx1Q0FBaUU7QUFDakUsc0NBQTJDO0FBQzNDLDRDQUF5QztBQUN6QyxrREFBaUQ7QUFDakQsNENBQWdEO0FBbUNoRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLFFBQVMsU0FBUSxxQkFBUztJQVdyQyxZQUFZLE9BQWdCLEVBQUUsR0FBUSxFQUFFLFVBQTJCLEVBQUU7O1FBQ25FLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVmLE1BQU0sYUFBYSxTQUFHLE9BQU8sQ0FBQyxhQUFhLG1DQUFJLHVCQUFjLENBQUM7UUFDOUQsSUFBSSxDQUFDLFNBQVMsU0FBRyxPQUFPLENBQUMsU0FBUyxtQ0FBSSxVQUFVLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsU0FBRyxPQUFPLENBQUMsU0FBUyxtQ0FBSSxJQUFJLENBQUM7UUFFM0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLFNBQVM7WUFDNUIsQ0FBQyxDQUFDLDZCQUFjLENBQUMsSUFBSTtZQUNyQixDQUFDLENBQUMsNkJBQWMsQ0FBQyxPQUFPLENBQUM7UUFDM0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQStCLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUN2RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsU0FBUztZQUNoQyxDQUFDLENBQUMsc0JBQXNCO1lBQ3hCLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQztRQUV2QixPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FDeEIsNEJBQTRCLGFBQWEsRUFBRSxFQUMzQyxPQUFPLENBQ1IsQ0FBQztRQUNGLEdBQUcsQ0FBQyxTQUFTLENBQUMsMkNBQTJDLENBQUMsQ0FBQztRQUUzRCxtR0FBbUc7UUFDbkcsTUFBQSxPQUFPLENBQUMsV0FBVywwQ0FBRSxJQUFJLENBQUMsT0FBTyxXQUFXLFVBQVUsRUFBRTtRQUN4RCxNQUFBLE9BQU8sQ0FBQyxXQUFXLDBDQUFFLElBQUksQ0FDdkIsMENBQTBDLElBQUksQ0FBQyxTQUFTLEdBQUcsUUFBUSxFQUFFLEVBQ3JFO1FBRUYsa0VBQWtFO1FBQ2xFLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO0lBQzFCLENBQUM7SUFFTyxnQkFBZ0I7O1FBQ3RCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQzNDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDZCxPQUFPO1NBQ1I7UUFDRCxNQUFNLE9BQU8sR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDO1FBQzlCLE1BQU0sWUFBWSxHQUFHLDRCQUFnQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQy9DLE1BQU0sUUFBUSxHQUFHLFlBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDN0MsTUFBTSxVQUFVLEdBQUcsWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUM7UUFDN0MsTUFBTSxjQUFjLHFCQUFHLFFBQVEsQ0FBQyxXQUFXLDBDQUFFLFVBQVUsMENBQUcsQ0FBQyxFQUFFLElBQUksMENBQUUsR0FBRyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDbkIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUN0QixnREFBZ0QsQ0FDakQsQ0FBQztZQUNGLE9BQU87U0FDUjtRQUNELE1BQU0sZUFBZSxHQUFHLFlBQVksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDM0QsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUNwQixJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3RCLDhDQUE4QyxjQUFjLEVBQUUsQ0FDL0QsQ0FBQztZQUNGLE9BQU87U0FDUjtRQUVELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO1FBQy9ELE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3hDLElBQUksU0FBaUIsRUFBRSxXQUFxQixDQUFDO1FBQzdDLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDdEIsU0FBUyxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyQixXQUFXLEdBQUcsRUFBRSxDQUFDO1NBQ2xCO2FBQU07WUFDTCxXQUFXLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztZQUMvQyxTQUFTLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDckM7UUFFRCxNQUFNLFFBQVEsR0FBRyxXQUFJLENBQ25CLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUNuQixHQUFHLEVBQ0gsR0FBRyxXQUFXLEVBQ2QsU0FBUyxHQUFHLE9BQU8sQ0FDcEIsQ0FBQztRQUVGLHNCQUFzQjtRQUN0QixJQUFJLHFCQUFVLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDeEIsT0FBTztTQUNSO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUVsQyxJQUFJLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFDZixNQUFNLElBQUksR0FBRyxDQUFDLE9BQWUsRUFBRSxFQUFFLEVBQUUsQ0FDakMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztRQUM1QyxNQUFNLFNBQVMsR0FBRyxDQUFDLE9BQWUsRUFBRSxFQUFFLEVBQUU7WUFDdEMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsQ0FBQztZQUNsQixNQUFNLEVBQUUsQ0FBQztRQUNYLENBQUMsQ0FBQztRQUNGLE1BQU0sVUFBVSxHQUFHLEdBQUcsRUFBRTtZQUN0QixNQUFNLEVBQUUsQ0FBQztZQUNULElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNaLENBQUMsQ0FBQztRQUVGLE1BQU0sVUFBVSxHQUEyQixFQUFFLENBQUM7UUFDOUMsS0FBSyxNQUFNLE1BQU0sSUFBSSxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMzQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2QsVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxrQkFBa0IsQ0FDMUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQy9CLENBQUM7YUFDSDtTQUNGO1FBRUQsSUFBSSxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUMxQixJQUFJLENBQUMsV0FBVyxXQUFXLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUMxQyxJQUFJLEVBQUUsQ0FBQztTQUNSO1FBRUQsTUFBTSxFQUFFLGVBQWUsRUFBRSxPQUFPLEVBQUUsR0FBRyxpQkFBaUIsQ0FDcEQsQ0FBQyxFQUNELGVBQWUsQ0FBQyxJQUFJLEVBQ3BCLFVBQVUsRUFDVixTQUFTLENBQUMsSUFBSSxDQUNmLENBQUM7UUFFRixJQUFJLENBQUMsVUFBVSxVQUFVLENBQUMsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQ0YsVUFBVSxVQUFVLENBQUMsT0FBTyxJQUFJLGtCQUFrQixDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQ3ZFLENBQUM7UUFDRixLQUFLLE1BQU0sY0FBYyxJQUFJLE9BQU8sRUFBRTtZQUNwQyxJQUFJLENBQUMsVUFBVSxVQUFVLENBQUMsT0FBTyxJQUFJLGNBQWMsR0FBRyxDQUFDLENBQUM7U0FDekQ7UUFDRCxJQUFJLEVBQUUsQ0FBQztRQUNQLFNBQVMsQ0FBQyxnQkFBZ0IsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUN2QyxTQUFTLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNwRCxJQUFJLENBQ0YsR0FBRyxRQUFRLENBQUMsSUFBSSxrQkFBa0IsUUFBUSxDQUFDLElBQUksSUFBSSxlQUFlLElBQUksQ0FDdkUsQ0FBQztRQUNGLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQ3pCLFVBQVUsRUFBRSxDQUFDO1FBQ2IsVUFBVSxFQUFFLENBQUM7UUFFYixxQkFBVSxDQUFDLGNBQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBQzlCLHdCQUFhLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUUxQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3RCLDBDQUEwQyxRQUFRLEVBQUUsQ0FDckQsQ0FBQztJQUNKLENBQUM7O0FBbkpILDRCQW9KQzs7O0FBRUQsU0FBUyxpQkFBaUIsQ0FDeEIsTUFBYyxFQUNkLGVBQXVCLEVBQ3ZCLFVBQWtDLEVBQ2xDLFdBQWlDO0lBRWpDLE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxFQUFVLENBQUM7SUFDbEMsSUFBSSxDQUFDLFdBQVcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDekQsT0FBTyxFQUFFLGVBQWUsRUFBRSxFQUFFLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQyxhQUFhO0tBQ3ZEO0lBRUQsTUFBTSxLQUFLLEdBQUcsQ0FBQyxHQUFHLGVBQWUsWUFBWSxDQUFDLENBQUM7SUFFL0MsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUU7UUFDdkQsTUFBTSxFQUFFLFNBQVMsRUFBRSxVQUFVLEVBQUUsR0FBRyxXQUFXLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUN2RSxJQUFJLFVBQVU7WUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ3hDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksU0FBUyxHQUFHLENBQUMsQ0FBQztLQUN0RDtJQUVELEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7SUFFdkIsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3hFLE9BQU8sRUFBRSxlQUFlLEVBQUUsT0FBTyxFQUFFLENBQUM7QUFDdEMsQ0FBQztBQUVELFNBQVMsY0FBYyxDQUFDLElBQVk7SUFDbEMsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBRUQsU0FBUyxXQUFXLENBQ2xCLEtBQVUsRUFDVixJQUFZLEVBQ1osVUFBa0M7SUFFbEMsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLFNBQVMsRUFBRTtRQUMvRCxNQUFNLEtBQUssR0FBRyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3JDLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQzlELE9BQU8sRUFBRSxTQUFTLEVBQUUsR0FBRyxJQUFJLElBQUksTUFBTSxFQUFFLEVBQUUsVUFBVSxFQUFFLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO0tBQ3pFO1NBQU07UUFDTCxPQUFPLEVBQUUsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztLQUM3QztBQUNILENBQUM7QUFFRCxTQUFTLGtCQUFrQixDQUFDLFFBQWE7SUFDdkMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBQ3hFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBkaXJuYW1lLCBqb2luIH0gZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IGV4aXN0c1N5bmMsIG1rZGlycFN5bmMsIHdyaXRlRmlsZVN5bmMgfSBmcm9tIFwiZnMtZXh0cmFcIjtcbmltcG9ydCB7IFBST0pFTl9WRVJTSU9OIH0gZnJvbSBcIi4uL2NvbW1vblwiO1xuaW1wb3J0IHsgQ29tcG9uZW50IH0gZnJvbSBcIi4uL2NvbXBvbmVudFwiO1xuaW1wb3J0IHsgRGVwZW5kZW5jeVR5cGUgfSBmcm9tIFwiLi4vZGVwZW5kZW5jaWVzXCI7XG5pbXBvcnQgeyByZWFkSnNpaU1hbmlmZXN0IH0gZnJvbSBcIi4uL2ludmVudG9yeVwiO1xuaW1wb3J0IHsgUHJvamVjdCB9IGZyb20gXCIuLi9wcm9qZWN0XCI7XG5pbXBvcnQgeyBQb20gfSBmcm9tIFwiLi9wb21cIjtcblxuLyoqXG4gKiBPcHRpb25zIGZvciBgUHJvamVucmNgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFByb2plbnJjT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgSmF2YSBjbGFzcyB3aGljaCBjb250YWlucyB0aGUgYG1haW4oKWAgbWV0aG9kIGZvciBwcm9qZW4uXG4gICAqIEBkZWZhdWx0IFwicHJvamVucmNcIlxuICAgKi9cbiAgcmVhZG9ubHkgY2xhc3NOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJvamVuIHZlcnNpb24gdG8gdXNlXG4gICAqIEBkZWZhdWx0IC0gY3VycmVudCB2ZXJzaW9uXG4gICAqL1xuICByZWFkb25seSBwcm9qZW5WZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHByb2plbnJjIHVuZGVyIHRoZSB0ZXN0IHNjb3BlIGluc3RlYWQgb2YgdGhlIG1haW4gc2NvcGUsIHdoaWNoIGlzXG4gICAqIHJlc2VydmVkIHRvIHRoZSBhcHAuIFRoaXMgbWVhbnMgdGhhdCBwcm9qZW5yYyB3aWxsIGJlIHVuZGVyXG4gICAqIGBzcmMvdGVzdC9qYXZhL3Byb2plbnJjLmphdmFgIGFuZCBwcm9qZW4gd2lsbCBiZSBkZWZpbmVkIGFzIGEgdGVzdFxuICAgKiBkZXBlbmRlbmN5LiBUaGlzIGVuZm9yY2VzIHRoYXQgYXBwbGljYXRpb24gY29kZSBkb2VzIG5vdCB0YWtlIGEgZGVwZW5kZW5jeVxuICAgKiBvbiBwcm9qZW4gY29kZS5cbiAgICpcbiAgICogSWYgdGhpcyBpcyBkaXNhYmxlZCwgcHJvamVucmMgc2hvdWxkIGJlIHVuZGVyXG4gICAqIGBzcmMvbWFpbi9qYXZhL3Byb2plbnJjLmphdmFgLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSB0ZXN0U2NvcGU/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEFsbG93cyB3cml0aW5nIHByb2plbnJjIGZpbGVzIGluIGphdmEuXG4gKlxuICogVGhpcyB3aWxsIGluc3RhbGwgYG9yZy5wcm9qZW4vcHJvamVuYCBhcyBhIE1hdmVuIGRlcGVuZGVuY3kgYW5kIHdpbGwgYWRkIGFcbiAqIGBzeW50aGAgdGFzayB3aGljaCB3aWxsIGNvbXBpbGUgJiBleGVjdXRlIGBtYWluKClgIGZyb21cbiAqIGBzcmMvbWFpbi9qYXZhL3Byb2plbnJjLmphdmFgLlxuICovXG5leHBvcnQgY2xhc3MgUHJvamVucmMgZXh0ZW5kcyBDb21wb25lbnQge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGphdmEgY2xhc3MgdGhhdCBpbmNsdWRlcyB0aGUgcHJvamVuIGVudHJ5cG9pbnQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2xhc3NOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIGNsYXNzIGlzIGluIHRoZSBcInRlc3RcIiBvciBcIm1haW5cIiBzY29wZT9cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgdGVzdFNjb3BlOiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKHByb2plY3Q6IFByb2plY3QsIHBvbTogUG9tLCBvcHRpb25zOiBQcm9qZW5yY09wdGlvbnMgPSB7fSkge1xuICAgIHN1cGVyKHByb2plY3QpO1xuXG4gICAgY29uc3QgcHJvamVuVmVyc2lvbiA9IG9wdGlvbnMucHJvamVuVmVyc2lvbiA/PyBQUk9KRU5fVkVSU0lPTjtcbiAgICB0aGlzLmNsYXNzTmFtZSA9IG9wdGlvbnMuY2xhc3NOYW1lID8/IFwicHJvamVucmNcIjtcbiAgICB0aGlzLnRlc3RTY29wZSA9IG9wdGlvbnMudGVzdFNjb3BlID8/IHRydWU7XG5cbiAgICBjb25zdCBkZXBUeXBlID0gdGhpcy50ZXN0U2NvcGVcbiAgICAgID8gRGVwZW5kZW5jeVR5cGUuVEVTVFxuICAgICAgOiBEZXBlbmRlbmN5VHlwZS5SVU5USU1FO1xuICAgIGNvbnN0IGV4ZWNPcHRzID0gdGhpcy50ZXN0U2NvcGUgPyAnIC1EZXhlYy5jbGFzc3BhdGhTY29wZT1cInRlc3RcIicgOiBcIlwiO1xuICAgIGNvbnN0IGNvbXBpbGVHb2FsID0gdGhpcy50ZXN0U2NvcGVcbiAgICAgID8gXCJjb21waWxlcjp0ZXN0Q29tcGlsZVwiXG4gICAgICA6IFwiY29tcGlsZXI6Y29tcGlsZVwiO1xuXG4gICAgcHJvamVjdC5kZXBzLmFkZERlcGVuZGVuY3koXG4gICAgICBgaW8uZ2l0aHViLmNka2xhYnMvcHJvamVuQCR7cHJvamVuVmVyc2lvbn1gLFxuICAgICAgZGVwVHlwZVxuICAgICk7XG4gICAgcG9tLmFkZFBsdWdpbihcIm9yZy5jb2RlaGF1cy5tb2pvL2V4ZWMtbWF2ZW4tcGx1Z2luQDMuMC4wXCIpO1xuXG4gICAgLy8gc2V0IHVwIHRoZSBcImRlZmF1bHRcIiB0YXNrIHdoaWNoIGlzIHRoZSB0YXNrIGV4ZWN1dGVkIHdoZW4gYHByb2plbmAgaXMgZXhlY3V0ZWQgZm9yIHRoaXMgcHJvamVjdC5cbiAgICBwcm9qZWN0LmRlZmF1bHRUYXNrPy5leGVjKGBtdm4gJHtjb21waWxlR29hbH0gLS1xdWlldGApO1xuICAgIHByb2plY3QuZGVmYXVsdFRhc2s/LmV4ZWMoXG4gICAgICBgbXZuIGV4ZWM6amF2YSAtLXF1aWV0IC1EZXhlYy5tYWluQ2xhc3M9JHt0aGlzLmNsYXNzTmFtZX0ke2V4ZWNPcHRzfWBcbiAgICApO1xuXG4gICAgLy8gaWYgdGhpcyBpcyBhIG5ldyBwcm9qZWN0LCBnZW5lcmF0ZSBhIHNrZWxldG9uIGZvciBwcm9qZW5yYy5qYXZhXG4gICAgdGhpcy5nZW5lcmF0ZVByb2plbnJjKCk7XG4gIH1cblxuICBwcml2YXRlIGdlbmVyYXRlUHJvamVucmMoKSB7XG4gICAgY29uc3QgYm9vdHN0cmFwID0gdGhpcy5wcm9qZWN0LmluaXRQcm9qZWN0O1xuICAgIGlmICghYm9vdHN0cmFwKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IGpzaWlGcW4gPSBib290c3RyYXAuZnFuO1xuICAgIGNvbnN0IGpzaWlNYW5pZmVzdCA9IHJlYWRKc2lpTWFuaWZlc3QoanNpaUZxbik7XG4gICAgY29uc3QganNpaVR5cGUgPSBqc2lpTWFuaWZlc3QudHlwZXNbanNpaUZxbl07XG4gICAgY29uc3QgamF2YVRhcmdldCA9IGpzaWlNYW5pZmVzdC50YXJnZXRzLmphdmE7XG4gICAgY29uc3Qgb3B0aW9uc1R5cGVGcW4gPSBqc2lpVHlwZS5pbml0aWFsaXplcj8ucGFyYW1ldGVycz8uWzBdLnR5cGU/LmZxbjtcbiAgICBpZiAoIW9wdGlvbnNUeXBlRnFuKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLndhcm4oXG4gICAgICAgIFwiY2Fubm90IGRldGVybWluZSBqc2lpIHR5cGUgZm9yIHByb2plY3Qgb3B0aW9uc1wiXG4gICAgICApO1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBqc2lpT3B0aW9uc1R5cGUgPSBqc2lpTWFuaWZlc3QudHlwZXNbb3B0aW9uc1R5cGVGcW5dO1xuICAgIGlmICghanNpaU9wdGlvbnNUeXBlKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLndhcm4oXG4gICAgICAgIGBjYW5ub3QgZmluZCBqc2lpIHR5cGUgZm9yIHByb2plY3Qgb3B0aW9uczogJHtvcHRpb25zVHlwZUZxbn1gXG4gICAgICApO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGRpciA9IHRoaXMudGVzdFNjb3BlID8gXCJzcmMvdGVzdC9qYXZhXCIgOiBcInNyYy9tYWluL2phdmFcIjtcbiAgICBjb25zdCBzcGxpdCA9IHRoaXMuY2xhc3NOYW1lLnNwbGl0KFwiLlwiKTtcbiAgICBsZXQgamF2YUNsYXNzOiBzdHJpbmcsIGphdmFQYWNrYWdlOiBzdHJpbmdbXTtcbiAgICBpZiAoc3BsaXQubGVuZ3RoID09PSAxKSB7XG4gICAgICBqYXZhQ2xhc3MgPSBzcGxpdFswXTtcbiAgICAgIGphdmFQYWNrYWdlID0gW107XG4gICAgfSBlbHNlIHtcbiAgICAgIGphdmFQYWNrYWdlID0gc3BsaXQuc2xpY2UoMCwgc3BsaXQubGVuZ3RoIC0gMik7XG4gICAgICBqYXZhQ2xhc3MgPSBzcGxpdFtzcGxpdC5sZW5ndGggLSAxXTtcbiAgICB9XG5cbiAgICBjb25zdCBqYXZhRmlsZSA9IGpvaW4oXG4gICAgICB0aGlzLnByb2plY3Qub3V0ZGlyLFxuICAgICAgZGlyLFxuICAgICAgLi4uamF2YVBhY2thZ2UsXG4gICAgICBqYXZhQ2xhc3MgKyBcIi5qYXZhXCJcbiAgICApO1xuXG4gICAgLy8gc2tpcCBpZiBmaWxlIGV4aXN0c1xuICAgIGlmIChleGlzdHNTeW5jKGphdmFGaWxlKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGxpbmVzID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICAgIGxldCBpbmRlbnQgPSAwO1xuICAgIGNvbnN0IGVtaXQgPSAobGluZTogc3RyaW5nID0gXCJcIikgPT5cbiAgICAgIGxpbmVzLnB1c2goXCIgXCIucmVwZWF0KGluZGVudCAqIDQpICsgbGluZSk7XG4gICAgY29uc3Qgb3BlbkJsb2NrID0gKGxpbmU6IHN0cmluZyA9IFwiXCIpID0+IHtcbiAgICAgIGVtaXQobGluZSArIFwiIHtcIik7XG4gICAgICBpbmRlbnQrKztcbiAgICB9O1xuICAgIGNvbnN0IGNsb3NlQmxvY2sgPSAoKSA9PiB7XG4gICAgICBpbmRlbnQtLTtcbiAgICAgIGVtaXQoXCJ9XCIpO1xuICAgIH07XG5cbiAgICBjb25zdCBvcHRpb25GcW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0ge307XG4gICAgZm9yIChjb25zdCBvcHRpb24gb2YgYm9vdHN0cmFwLnR5cGUub3B0aW9ucykge1xuICAgICAgaWYgKG9wdGlvbi5mcW4pIHtcbiAgICAgICAgb3B0aW9uRnFuc1tvcHRpb24ubmFtZV0gPSB0b0phdmFGdWxsVHlwZU5hbWUoXG4gICAgICAgICAganNpaU1hbmlmZXN0LnR5cGVzW29wdGlvbi5mcW5dXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKGphdmFQYWNrYWdlLmxlbmd0aCA+IDApIHtcbiAgICAgIGVtaXQoYHBhY2thZ2UgJHtqYXZhUGFja2FnZS5qb2luKFwiLlwiKX07YCk7XG4gICAgICBlbWl0KCk7XG4gICAgfVxuXG4gICAgY29uc3QgeyByZW5kZXJlZE9wdGlvbnMsIGltcG9ydHMgfSA9IHJlbmRlckphdmFPcHRpb25zKFxuICAgICAgMixcbiAgICAgIGpzaWlPcHRpb25zVHlwZS5uYW1lLFxuICAgICAgb3B0aW9uRnFucyxcbiAgICAgIGJvb3RzdHJhcC5hcmdzXG4gICAgKTtcblxuICAgIGVtaXQoYGltcG9ydCAke2phdmFUYXJnZXQucGFja2FnZX0uJHt0b0phdmFGdWxsVHlwZU5hbWUoanNpaVR5cGUpfTtgKTtcbiAgICBlbWl0KFxuICAgICAgYGltcG9ydCAke2phdmFUYXJnZXQucGFja2FnZX0uJHt0b0phdmFGdWxsVHlwZU5hbWUoanNpaU9wdGlvbnNUeXBlKX07YFxuICAgICk7XG4gICAgZm9yIChjb25zdCBvcHRpb25UeXBlTmFtZSBvZiBpbXBvcnRzKSB7XG4gICAgICBlbWl0KGBpbXBvcnQgJHtqYXZhVGFyZ2V0LnBhY2thZ2V9LiR7b3B0aW9uVHlwZU5hbWV9O2ApO1xuICAgIH1cbiAgICBlbWl0KCk7XG4gICAgb3BlbkJsb2NrKGBwdWJsaWMgY2xhc3MgJHtqYXZhQ2xhc3N9YCk7XG4gICAgb3BlbkJsb2NrKFwicHVibGljIHN0YXRpYyB2b2lkIG1haW4oU3RyaW5nW10gYXJncylcIik7XG4gICAgZW1pdChcbiAgICAgIGAke2pzaWlUeXBlLm5hbWV9IHByb2plY3QgPSBuZXcgJHtqc2lpVHlwZS5uYW1lfSgke3JlbmRlcmVkT3B0aW9uc30pO2BcbiAgICApO1xuICAgIGVtaXQoXCJwcm9qZWN0LnN5bnRoKCk7XCIpO1xuICAgIGNsb3NlQmxvY2soKTtcbiAgICBjbG9zZUJsb2NrKCk7XG5cbiAgICBta2RpcnBTeW5jKGRpcm5hbWUoamF2YUZpbGUpKTtcbiAgICB3cml0ZUZpbGVTeW5jKGphdmFGaWxlLCBsaW5lcy5qb2luKFwiXFxuXCIpKTtcblxuICAgIHRoaXMucHJvamVjdC5sb2dnZXIuaW5mbyhcbiAgICAgIGBQcm9qZWN0IGRlZmluaXRpb24gZmlsZSB3YXMgY3JlYXRlZCBhdCAke2phdmFGaWxlfWBcbiAgICApO1xuICB9XG59XG5cbmZ1bmN0aW9uIHJlbmRlckphdmFPcHRpb25zKFxuICBpbmRlbnQ6IG51bWJlcixcbiAgb3B0aW9uc1R5cGVOYW1lOiBzdHJpbmcsXG4gIG9wdGlvbkZxbnM6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4sXG4gIGluaXRPcHRpb25zPzogUmVjb3JkPHN0cmluZywgYW55PlxuKSB7XG4gIGNvbnN0IGltcG9ydHMgPSBuZXcgU2V0PHN0cmluZz4oKTtcbiAgaWYgKCFpbml0T3B0aW9ucyB8fCBPYmplY3Qua2V5cyhpbml0T3B0aW9ucykubGVuZ3RoID09PSAwKSB7XG4gICAgcmV0dXJuIHsgcmVuZGVyZWRPcHRpb25zOiBcIlwiLCBpbXBvcnRzIH07IC8vIG5vIG9wdGlvbnNcbiAgfVxuXG4gIGNvbnN0IGxpbmVzID0gW2Ake29wdGlvbnNUeXBlTmFtZX0uYnVpbGRlcigpYF07XG5cbiAgZm9yIChjb25zdCBbbmFtZSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKGluaXRPcHRpb25zKSkge1xuICAgIGNvbnN0IHsgamF2YVZhbHVlLCBpbXBvcnROYW1lIH0gPSB0b0phdmFWYWx1ZSh2YWx1ZSwgbmFtZSwgb3B0aW9uRnFucyk7XG4gICAgaWYgKGltcG9ydE5hbWUpIGltcG9ydHMuYWRkKGltcG9ydE5hbWUpO1xuICAgIGxpbmVzLnB1c2goYC4ke3RvSmF2YVByb3BlcnR5KG5hbWUpfSgke2phdmFWYWx1ZX0pYCk7XG4gIH1cblxuICBsaW5lcy5wdXNoKFwiLmJ1aWxkKClcIik7XG5cbiAgY29uc3QgcmVuZGVyZWRPcHRpb25zID0gbGluZXMuam9pbihgXFxuJHtcIiBcIi5yZXBlYXQoKGluZGVudCArIDEpICogNCl9YCk7XG4gIHJldHVybiB7IHJlbmRlcmVkT3B0aW9ucywgaW1wb3J0cyB9O1xufVxuXG5mdW5jdGlvbiB0b0phdmFQcm9wZXJ0eShwcm9wOiBzdHJpbmcpIHtcbiAgcmV0dXJuIHByb3A7XG59XG5cbmZ1bmN0aW9uIHRvSmF2YVZhbHVlKFxuICB2YWx1ZTogYW55LFxuICBuYW1lOiBzdHJpbmcsXG4gIG9wdGlvbkZxbnM6IFJlY29yZDxzdHJpbmcsIHN0cmluZz5cbikge1xuICBpZiAodHlwZW9mIHZhbHVlID09PSBcInN0cmluZ1wiICYmIG9wdGlvbkZxbnNbbmFtZV0gIT09IHVuZGVmaW5lZCkge1xuICAgIGNvbnN0IHBhcnRzID0gb3B0aW9uRnFuc1tuYW1lXS5zcGxpdChcIi5cIik7XG4gICAgY29uc3QgYmFzZSA9IHBhcnRzW3BhcnRzLmxlbmd0aCAtIDFdO1xuICAgIGNvbnN0IGNob2ljZSA9IFN0cmluZyh2YWx1ZSkudG9VcHBlckNhc2UoKS5yZXBsYWNlKC8tL2csIFwiX1wiKTtcbiAgICByZXR1cm4geyBqYXZhVmFsdWU6IGAke2Jhc2V9LiR7Y2hvaWNlfWAsIGltcG9ydE5hbWU6IG9wdGlvbkZxbnNbbmFtZV0gfTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4geyBqYXZhVmFsdWU6IEpTT04uc3RyaW5naWZ5KHZhbHVlKSB9O1xuICB9XG59XG5cbmZ1bmN0aW9uIHRvSmF2YUZ1bGxUeXBlTmFtZShqc2lpVHlwZTogYW55KSB7XG4gIHJldHVybiBbanNpaVR5cGUubmFtZXNwYWNlLCBqc2lpVHlwZS5uYW1lXS5maWx0ZXIoKHgpID0+IHgpLmpvaW4oXCIuXCIpO1xufVxuIl19