"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const util_1 = require("./util");
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
class Bundler extends component_1.Component {
    /**
     * Creates a `Bundler`.
     */
    constructor(project, options = {}) {
        var _b, _c;
        super(project);
        this.esbuildVersion = options.esbuildVersion;
        this.bundledir = (_b = options.assetsDir) !== null && _b !== void 0 ? _b : "assets";
        this.addToPreCompile = (_c = options.addToPreCompile) !== null && _c !== void 0 ? _c : true;
    }
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project) {
        const isBundler = (o) => o instanceof Bundler;
        return project.components.find(isBundler);
    }
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask() {
        if (!this._task) {
            this.addBundlingSupport();
            this._task = this.project.tasks.addTask("bundle", {
                description: "Prepare assets",
            });
            // install the bundle task into the pre-compile phase.
            if (this.addToPreCompile) {
                this.project.preCompileTask.spawn(this._task);
            }
        }
        return this._task;
    }
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint, options) {
        var _b, _c, _d;
        const name = util_1.renderBundleName(entrypoint);
        const outfile = path_1.join(this.bundledir, name, "index.js");
        const args = [
            "esbuild",
            "--bundle",
            entrypoint,
            `--target="${options.target}"`,
            `--platform="${options.platform}"`,
            `--outfile="${outfile}"`,
        ];
        for (const x of (_b = options.externals) !== null && _b !== void 0 ? _b : []) {
            args.push(`--external:${x}`);
        }
        const sourcemap = (_c = options.sourcemap) !== null && _c !== void 0 ? _c : false;
        if (sourcemap) {
            args.push("--sourcemap");
        }
        const bundleTask = this.project.addTask(`bundle:${name}`, {
            description: `Create a JavaScript bundle from ${entrypoint}`,
            exec: args.join(" "),
        });
        this.bundleTask.spawn(bundleTask);
        let watchTask;
        const watch = (_d = options.watchTask) !== null && _d !== void 0 ? _d : true;
        if (watch) {
            watchTask = this.project.addTask(`bundle:${name}:watch`, {
                description: `Continuously update the JavaScript bundle from ${entrypoint}`,
                exec: `${args.join(" ")} --watch`,
            });
        }
        return {
            bundleTask: bundleTask,
            watchTask: watchTask,
            outfile: outfile,
        };
    }
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    addBundlingSupport() {
        const ignoreEntry = `/${this.bundledir}/`;
        this.project.addGitIgnore(ignoreEntry);
        this.project.addPackageIgnore(`!${ignoreEntry}`); // include in tarball
        const dep = this.esbuildVersion
            ? `esbuild@${this.esbuildVersion}`
            : "esbuild";
        this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
    }
}
exports.Bundler = Bundler;
_a = JSII_RTTI_SYMBOL_1;
Bundler[_a] = { fqn: "projen.javascript.Bundler", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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