"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
const prettier_1 = require("./prettier");
/**
 * Represents eslint configuration.
 */
class Eslint extends component_1.Component {
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g;
        super(project);
        this._plugins = new Array();
        this._extends = new Array();
        this.nodeProject = project;
        project.addDevDeps("eslint@^8", "@typescript-eslint/eslint-plugin@^5", "@typescript-eslint/parser@^5", "eslint-import-resolver-node", "eslint-import-resolver-typescript", "eslint-plugin-import", "json-schema");
        if (options.aliasMap) {
            project.addDevDeps("eslint-import-resolver-alias");
        }
        const devdirs = (_b = options.devdirs) !== null && _b !== void 0 ? _b : [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = (_c = options.fileExtensions) !== null && _c !== void 0 ? _c : [".ts"];
        this._allowDevDeps = new Set((devdirs !== null && devdirs !== void 0 ? devdirs : []).map((dir) => `**/${dir}/**`));
        const lintProjenRc = (_d = options.lintProjenRc) !== null && _d !== void 0 ? _d : true;
        const eslint = project.addTask("eslint", {
            description: "Runs eslint against the codebase",
            exec: [
                "eslint",
                `--ext ${fileExtensions.join(",")}`,
                "--fix",
                "--no-error-on-unmatched-pattern",
                ...dirs,
                ...(lintProjenRc ? [common_1.PROJEN_RC] : []),
            ].join(" "),
        });
        project.testTask.spawn(eslint);
        // exclude some files
        (_e = project.npmignore) === null || _e === void 0 ? void 0 : _e.exclude("/.eslintrc.json");
        this._formattingRules = {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            indent: ["off"],
            "@typescript-eslint/indent": ["error", 2],
            // Style
            quotes: ["error", "single", { avoidEscape: true }],
            "comma-dangle": ["error", "always-multiline"],
            "comma-spacing": ["error", { before: false, after: true }],
            "no-multi-spaces": ["error", { ignoreEOLComments: false }],
            "array-bracket-spacing": ["error", "never"],
            "array-bracket-newline": ["error", "consistent"],
            "object-curly-spacing": ["error", "always"],
            "object-curly-newline": ["error", { multiline: true, consistent: true }],
            "object-property-newline": [
                "error",
                { allowAllPropertiesOnSameLine: true },
            ],
            "keyword-spacing": ["error"],
            "brace-style": ["error", "1tbs", { allowSingleLine: true }],
            "space-before-blocks": ["error"],
            curly: ["error", "multi-line", "consistent"],
            "@typescript-eslint/member-delimiter-style": ["error"],
            // Require semicolons
            semi: ["error", "always"],
            // Max line lengths
            "max-len": [
                "error",
                {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                },
            ],
            // Don't unnecessarily quote properties
            "quote-props": ["error", "consistent-as-needed"],
        };
        this.rules = {
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            "@typescript-eslint/no-require-imports": ["error"],
            // Require all imported dependencies are actually declared in package.json
            "import/no-extraneous-dependencies": [
                "error",
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            "import/no-unresolved": ["error"],
            // Require an ordering on all imports
            "import/order": [
                "warn",
                {
                    groups: ["builtin", "external"],
                    alphabetize: { order: "asc", caseInsensitive: true },
                },
            ],
            // Cannot import from the same module twice
            "no-duplicate-imports": ["error"],
            // Cannot shadow names
            "no-shadow": ["off"],
            "@typescript-eslint/no-shadow": ["error"],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            "key-spacing": ["error"],
            // No multiple empty lines
            "no-multiple-empty-lines": ["error"],
            // One of the easiest mistakes to make
            "@typescript-eslint/no-floating-promises": ["error"],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            "no-return-await": ["off"],
            "@typescript-eslint/return-await": ["error"],
            // Useless diff results
            "no-trailing-spaces": ["error"],
            // Must use foo.bar instead of foo['bar'] if possible
            "dot-notation": ["error"],
            // Are you sure | is not a typo for || ?
            "no-bitwise": ["error"],
            // Member ordering
            "@typescript-eslint/member-ordering": [
                "error",
                {
                    default: [
                        "public-static-field",
                        "public-static-method",
                        "protected-static-field",
                        "protected-static-method",
                        "private-static-field",
                        "private-static-method",
                        "field",
                        // Constructors
                        "constructor",
                        // Methods
                        "method",
                    ],
                },
            ],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [common_1.PROJEN_RC],
                rules: {
                    "@typescript-eslint/no-require-imports": "off",
                    "import/no-extraneous-dependencies": "off",
                },
            },
        ];
        this.ignorePatterns = (_f = options.ignorePatterns) !== null && _f !== void 0 ? _f : [
            "*.js",
            `!${common_1.PROJEN_RC}`,
            "*.d.ts",
            "node_modules/",
            "*.generated.ts",
            "coverage",
        ];
        const tsconfig = (_g = options.tsconfigPath) !== null && _g !== void 0 ? _g : "./tsconfig.json";
        this.addPlugins("@typescript-eslint");
        this.addPlugins("import");
        this.addExtends("plugin:import/typescript");
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins: () => this._plugins,
            parser: "@typescript-eslint/parser",
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: "module",
                project: tsconfig,
            },
            extends: () => this._extends,
            settings: {
                "import/parsers": {
                    "@typescript-eslint/parser": [".ts", ".tsx"],
                },
                "import/resolver": {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: () => ({ ...this._formattingRules, ...this.rules }),
            overrides: this.overrides,
        };
        new json_1.JsonFile(project, ".eslintrc.json", {
            obj: this.config,
            marker: false,
        });
        // if the user enabled prettier explicitly _or_ if the project has a
        // `Prettier` component, we shall tweak our configuration accordingly.
        if (options.prettier || prettier_1.Prettier.of(project)) {
            this.enablePrettier();
        }
    }
    /**
     * Returns the singletone Eslint component of a project or undefined if there is none.
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    /**
     * Add an eslint rule.
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * Adds an eslint plugin
     * @param plugins The names of plugins to add
     */
    addPlugins(...plugins) {
        this._plugins.push(...plugins);
    }
    /**
     * Add an eslint override.
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * Do not lint these files.
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * Adds an `extends` item to the eslint configuration.
     * @param extendList The list of "extends" to add.
     */
    addExtends(...extendList) {
        this._extends.push(...extendList);
    }
    /**
     * Add a glob file pattern which allows importing dev dependencies.
     * @param pattern glob pattern.
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    /**
     * Enables prettier for code formatting.
     */
    enablePrettier() {
        this.nodeProject.addDevDeps("prettier", "eslint-plugin-prettier", "eslint-config-prettier");
        this.addPlugins("prettier");
        this._formattingRules = {
            "prettier/prettier": ["error"],
        };
        this.addExtends("prettier", "plugin:prettier/recommended");
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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