"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultNpmToken = exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const url_1 = require("url");
const fs_extra_1 = require("fs-extra");
const semver = require("semver");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const util_1 = require("../util");
const UNLICENSED = "UNLICENSED";
const DEFAULT_NPM_REGISTRY_URL = "https://registry.npmjs.org/";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const DEFAULT_NPM_TOKEN_SECRET = "NPM_TOKEN";
const DEFAULT_GITHUB_TOKEN_SECRET = "GITHUB_TOKEN";
/**
 * Represents the npm `package.json` file.
 */
class NodePackage extends component_1.Component {
    constructor(project, options = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = (_b = options.packageName) !== null && _b !== void 0 ? _b : project.name;
        this.peerDependencyOptions = {
            pinnedDevDependency: true,
            ...options.peerDependencyOptions,
        };
        this.allowLibraryDependencies = (_c = options.allowLibraryDependencies) !== null && _c !== void 0 ? _c : true;
        this.packageManager = (_d = options.packageManager) !== null && _d !== void 0 ? _d : NodePackageManager.YARN;
        this.entrypoint = (_e = options.entrypoint) !== null && _e !== void 0 ? _e : "lib/index.js";
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.processDeps(options);
        const prev = (_f = this.readPackageJson()) !== null && _f !== void 0 ? _f : {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository
                ? undefined
                : {
                    type: "git",
                    url: options.repository,
                    directory: options.repositoryDirectory,
                },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== "" ? this.entrypoint : undefined,
            license: () => { var _b; return (_b = this.license) !== null && _b !== void 0 ? _b : UNLICENSED; },
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev === null || prev === void 0 ? void 0 : prev.version),
            bugs: options.bugsEmail || options.bugsUrl
                ? {
                    email: options.bugsEmail,
                    url: options.bugsUrl,
                }
                : undefined,
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries((_g = options.scripts) !== null && _g !== void 0 ? _g : {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, "package.json", {
            obj: this.manifest,
            readonly: false,
            newline: false,
        });
        this.addKeywords(...((_h = options.keywords) !== null && _h !== void 0 ? _h : []));
        this.addBin((_j = options.bin) !== null && _j !== void 0 ? _j : {});
        // automatically add all executable files under "bin"
        if ((_k = options.autoDetectBin) !== null && _k !== void 0 ? _k : true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.addNodeEngine();
        // license
        if ((_l = options.licensed) !== null && _l !== void 0 ? _l : true) {
            this.license = (_m = options.license) !== null && _m !== void 0 ? _m : "Apache-2.0";
        }
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * Override the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .map((d) => d.name)
                    .filter((d) => (include ? include.includes(d) : true))
                    .filter((d) => !exclude.includes(d))
                    .join(" ")}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                lazy = upgradePackages("yarn upgrade");
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    postSynthesize() {
        super.postSynthesize();
        // only run "install" if package.json has changed or if we don't have a
        // `node_modules` directory.
        if (this.file.changed ||
            !fs_extra_1.existsSync(path_1.join(this.project.outdir, "node_modules"))) {
            this.installDependencies();
        }
        // resolve "*" deps in package.json and update it. if it was changed,
        // install deps again so that lockfile is updated.
        if (this.resolveDepsAndWritePackageJson()) {
            this.installDependencies();
        }
    }
    /**
     * The command which executes "projen".
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_1.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return "0.0.0";
        }
        return currVersion !== null && currVersion !== void 0 ? currVersion : "0.0.0";
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return util_1.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl !== null && npmRegistryUrl !== void 0 ? npmRegistryUrl : DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = (_b = options.npmAccess) !== null && _b !== void 0 ? _b : defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const isAwsCodeArtifact = release_1.isAwsCodeArtifactRegistry(npmRegistryUrl);
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
        }
        else {
            if (((_c = options.codeArtifactOptions) === null || _c === void 0 ? void 0 : _c.accessKeyIdSecret) || ((_d = options.codeArtifactOptions) === null || _d === void 0 ? void 0 : _d.secretAccessKeySecret) || ((_e = options.codeArtifactOptions) === null || _e === void 0 ? void 0 : _e.roleToAssume)) {
                throw new Error("codeArtifactOptions must only be specified when publishing AWS CodeArtifact.");
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact) {
            codeArtifactOptions = {
                accessKeyIdSecret: (_g = (_f = options.codeArtifactOptions) === null || _f === void 0 ? void 0 : _f.accessKeyIdSecret) !== null && _g !== void 0 ? _g : "AWS_ACCESS_KEY_ID",
                secretAccessKeySecret: (_j = (_h = options.codeArtifactOptions) === null || _h === void 0 ? void 0 : _h.secretAccessKeySecret) !== null && _j !== void 0 ? _j : "AWS_SECRET_ACCESS_KEY",
                roleToAssume: (_k = options.codeArtifactOptions) === null || _k === void 0 ? void 0 : _k.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
        };
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = "";
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine("node", nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == "/") {
            return "";
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    "yarn install",
                    "--check-files",
                    ...(frozen ? ["--frozen-lockfile"] : []),
                ].join(" ");
            case NodePackageManager.NPM:
                return frozen ? "npm ci" : "npm install";
            case NodePackageManager.PNPM:
                return ["pnpm i", ...(frozen ? ["--frozen-lockfile"] : [])].join(" ");
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        var _b, _c, _d, _e;
        this.addDeps(...((_b = options.deps) !== null && _b !== void 0 ? _b : []));
        this.addDevDeps(...((_c = options.devDeps) !== null && _c !== void 0 ? _c : []));
        this.addPeerDeps(...((_d = options.peerDeps) !== null && _d !== void 0 ? _d : []));
        this.addBundledDeps(...((_e = options.bundledDeps) !== null && _e !== void 0 ? _e : []));
    }
    renderDependencies() {
        var _b, _c;
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const dep of this.project.deps.all.filter((d) => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // skip if we already have a runtime dependency on this peer
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME)) {
                    continue;
                }
                if (dep.version) {
                    const ver = (_b = semver.minVersion(dep.version)) === null || _b === void 0 ? void 0 : _b.version;
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += "@" + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            const version = (_c = dep.version) !== null && _c !== void 0 ? _c : "*";
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(dep.name);
                    if (this.project.deps.all.find((d) => d.name === dep.name && d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${dep.name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[dep.name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[dep.name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[dep.name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[dep.name] = version;
                    break;
            }
        }
        // returns a lazy value to normalize during synthesis
        const normalize = (obj) => () => util_1.sorted(obj);
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = normalize(devDependencies);
        this.manifest.peerDependencies = normalize(peerDependencies);
        this.manifest.dependencies = normalize(dependencies);
        this.manifest.bundledDependencies = util_1.sorted(bundledDependencies);
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== "*" || !currentVersion || currentVersion === "*") {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current !== null && current !== void 0 ? current : {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    resolveDepsAndWritePackageJson() {
        var _b;
        const outdir = this.project.outdir;
        const rootPackageJson = path_1.join(outdir, "package.json");
        const original = fs_1.readFileSync(rootPackageJson, "utf8");
        const pkg = JSON.parse(original);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current !== null && current !== void 0 ? current : {};
            user = user !== null && user !== void 0 ? user : {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === "*") {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, {
                            paths: [outdir],
                        });
                        const module = fs_extra_1.readJsonSync(modulePath);
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return result;
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error("assertion failed");
        }
        const deps = resolveDeps(pkg.dependencies, rendered.dependencies);
        const devDeps = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        const peerDeps = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const [name, version] of Object.entries(peerDeps)) {
                // Skip if we already have a runtime dependency on this peer
                // or if devDependency version is already set.
                // Relies on the "*" devDependency added in the presynth step
                if (deps[name] || rendered.devDependencies[name] !== "*") {
                    continue;
                }
                // Take version and pin as dev dependency
                const ver = (_b = semver.minVersion(version)) === null || _b === void 0 ? void 0 : _b.version;
                if (!ver) {
                    throw new Error(`unable to determine minimum semver for peer dependency ${name}@${version}`);
                }
                devDeps[name] = ver;
            }
        }
        pkg.dependencies = util_1.sorted(deps);
        pkg.devDependencies = util_1.sorted(devDeps);
        pkg.peerDependencies = util_1.sorted(peerDeps);
        const updated = JSON.stringify(pkg, undefined, 2);
        if (original === updated) {
            return false;
        }
        util_1.writeFile(rootPackageJson, updated);
        return true;
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL
                ? this.npmRegistryUrl
                : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName)
                ? this.npmAccess
                : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_1.sorted(kwds.sort());
    }
    renderEngines() {
        return util_1.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = "bin";
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_extra_1.existsSync(bindir)) {
            for (const file of fs_extra_1.readdirSync(bindir)) {
                try {
                    fs_extra_1.accessSync(path_1.join(bindir, file), fs_extra_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, "/");
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        var _b;
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: (_b = options.authorOrganization) !== null && _b !== void 0 ? _b : false,
            };
        }
        else {
            if (options.authorEmail ||
                options.authorUrl ||
                options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_1.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        for (const task of this.project.tasks.all.sort((x, y) => x.name.localeCompare(y.name))) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, "package.json");
        if (!fs_extra_1.existsSync(file)) {
            return undefined;
        }
        return fs_extra_1.readJsonSync(file);
    }
    installDependencies() {
        util_1.exec(this.renderInstallCommand(this.isAutomatedBuild), {
            cwd: this.project.outdir,
        });
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.52.53" };
/**
 * The node package manager to use.
 */
var NodePackageManager;
(function (NodePackageManager) {
    /**
     * Use `yarn` as the package manager.
     */
    NodePackageManager["YARN"] = "yarn";
    /**
     * Use `npm` as the package manager.
     */
    NodePackageManager["NPM"] = "npm";
    /**
     * Use `pnpm` as the package manager.
     */
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * Npm package access level
 */
var NpmAccess;
(function (NpmAccess) {
    /**
     * Package is public.
     */
    NpmAccess["PUBLIC"] = "public";
    /**
     * Package can only be accessed with credentials.
     */
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes("@");
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if (release_1.isAwsCodeArtifactRegistry(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return (npmToken !== null && npmToken !== void 0 ? npmToken : (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET));
}
exports.defaultNpmToken = defaultNpmToken;
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN) {
        return "yarn.lock";
    }
    else if (packageManager === NodePackageManager.NPM) {
        return "package-lock.json";
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return "pnpm-lock.yaml";
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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