"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJavaScriptOptions = exports.resolveInitProject = exports.renderProjenInitOptions = void 0;
const inventory = require("../inventory");
const option_hints_1 = require("../option-hints");
const PROJEN_NEW = "__new__";
const TAB = makePadding(2);
/**
 * Renders options as if the project was created via `projen new` (embeds the __new__ field).
 */
function renderProjenInitOptions(fqn, args, comments = option_hints_1.InitProjectOptionHints.NONE) {
    return {
        ...args,
        [PROJEN_NEW]: { fqn, args, comments },
    };
}
exports.renderProjenInitOptions = renderProjenInitOptions;
function resolveInitProject(opts) {
    const f = opts[PROJEN_NEW];
    if (!f) {
        return undefined;
    }
    const type = inventory.resolveProjectType(f.fqn);
    if (!type) {
        throw new Error(`unable to resolve project type for ${f.fqn}`);
    }
    return {
        args: f.args,
        fqn: f.fqn,
        type: type,
        comments: f.comments,
    };
}
exports.resolveInitProject = resolveInitProject;
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 *
 * Returns the printed output and a set of required imports as an object
 * in the form { options, imports }.
 */
function renderJavaScriptOptions(opts) {
    var _a, _b, _c, _d;
    const renders = {};
    const optionsWithDefaults = [];
    const allImports = new Set();
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        if (opts.args[optionName] !== undefined) {
            const arg = opts.args[optionName];
            const { js, imports } = renderArgAsJavaScript(arg, option);
            for (const importStr of imports) {
                allImports.add(importStr);
            }
            renders[optionName] = `${optionName}: ${js},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith("-")) ? undefined
                : (_b = option.default) !== null && _b !== void 0 ? _b : undefined;
            renders[optionName] = `// ${optionName}: ${defaultValue},`;
        }
    }
    const bootstrap = (_c = opts.bootstrap) !== null && _c !== void 0 ? _c : false;
    if (bootstrap) {
        for (const arg of (_d = opts.omitFromBootstrap) !== null && _d !== void 0 ? _d : []) {
            delete opts.args[arg];
        }
        renders[PROJEN_NEW] = `${PROJEN_NEW}: ${JSON.stringify({
            args: opts.args,
            fqn: opts.type.fqn,
            comments: opts.comments,
        })},`;
        optionsWithDefaults.push(PROJEN_NEW);
    }
    // generate rendering
    const result = [];
    result.push("{");
    // render options with defaults
    optionsWithDefaults.sort();
    for (const optionName of optionsWithDefaults) {
        result.push(`${TAB}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push("");
    }
    // render options without defaults as comments
    if (opts.comments === option_hints_1.InitProjectOptionHints.ALL) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined);
        result.push(...renderCommentedOptionsByModule(renders, options));
    }
    else if (opts.comments === option_hints_1.InitProjectOptionHints.FEATURED) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined && opt.featured);
        result.push(...renderCommentedOptionsInOrder(renders, options));
    }
    else if (opts.comments === option_hints_1.InitProjectOptionHints.NONE) {
        // don't render any extra options
    }
    if (result[result.length - 1] === "") {
        result.pop();
    }
    result.push("}");
    return { renderedOptions: result.join("\n"), imports: allImports };
}
exports.renderJavaScriptOptions = renderJavaScriptOptions;
function renderCommentedOptionsByModule(renders, options) {
    var _a;
    const optionsByModule = {};
    for (const option of options) {
        const parentModule = option.parent;
        optionsByModule[parentModule] = (_a = optionsByModule[parentModule]) !== null && _a !== void 0 ? _a : [];
        optionsByModule[parentModule].push(option);
    }
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const [moduleName, optionGroup] of Object.entries(optionsByModule).sort()) {
        result.push(`${TAB}/* ${moduleName} */`);
        for (const option of optionGroup) {
            const paramRender = renders[option.name];
            const docstring = option.docs || "No documentation found.";
            result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
        }
        result.push("");
    }
    return result;
}
function renderCommentedOptionsInOrder(renders, options) {
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const option of options) {
        const paramRender = renders[option.name];
        const docstring = option.docs || "No documentation found.";
        result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
    }
    return result;
}
/**
 * Renders a value as a JavaScript value, converting strings to enums where
 * appropriate. The type must be JSON-like (string, number, boolean, array,
 * enum, or JSON object).
 *
 * Returns a JavaScript expression as a string, and the names of any
 * necessary imports.
 */
function renderArgAsJavaScript(arg, option) {
    if (option.kind === "enum") {
        if (!option.fqn) {
            throw new Error(`fqn field is missing from enum option ${option.name}`);
        }
        const parts = option.fqn.split("."); // -> ['projen', 'web', 'MyEnum']
        const enumChoice = String(arg).toUpperCase().replace(/-/g, "_"); // custom-value -> CUSTOM_VALUE
        const js = `${parts.slice(1).join(".")}.${enumChoice}`; // -> web.MyEnum.CUSTOM_VALUE
        const importName = parts[1]; // -> web
        return { js, imports: [importName] };
    }
    else if (option.jsonLike) {
        return { js: JSON.stringify(arg), imports: [] };
    }
    else {
        throw new Error(`Unexpected option ${option.name} - cannot render a value for this option because it does not have a JSON-like type.`);
    }
}
function makePadding(paddingLength) {
    return " ".repeat(paddingLength);
}
//# sourceMappingURL=data:application/json;base64,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