import { Component } from "../component";
import { GithubWorkflow, GitIdentity, workflows } from "../github";
import { ContainerOptions, JobStep } from "../github/workflows-model";
import { NodeProject } from "../javascript";
import { Task } from "../task";
/**
 * Options for `UpgradeDependencies`.
 */
export interface UpgradeDependenciesOptions {
    /**
     * List of package names to exclude during the upgrade.
     *
     * @default - Nothing is excluded.
     */
    readonly exclude?: string[];
    /**
     * List of package names to include during the upgrade.
     *
     * @default - Everything is included.
     */
    readonly include?: string[];
    /**
     * Include a github workflow for creating PR's that upgrades the
     * required dependencies, either by manual dispatch, or by a schedule.
     *
     * If this is `false`, only a local projen task is created, which can be executed manually to
     * upgrade the dependencies.
     *
     * @default - true for root projects, false for sub-projects.
     */
    readonly workflow?: boolean;
    /**
     * Options for the github workflow. Only applies if `workflow` is true.
     *
     * @default - default options.
     */
    readonly workflowOptions?: UpgradeDependenciesWorkflowOptions;
    /**
     * The name of the task that will be created.
     * This will also be the workflow name.
     *
     * @default "upgrade".
     */
    readonly taskName?: string;
    /**
     * Title of the pull request to use (should be all lower-case).
     *
     * @default "upgrade dependencies"
     */
    readonly pullRequestTitle?: string;
    /**
     * Whether or not to ignore projen upgrades.
     *
     * @default true
     */
    readonly ignoreProjen?: boolean;
    /**
     * Add Signed-off-by line by the committer at the end of the commit log message.
     *
     * @default true
     */
    readonly signoff?: boolean;
}
/**
 * Upgrade node project dependencies.
 */
export declare class UpgradeDependencies extends Component {
    /**
     * The workflows that execute the upgrades. One workflow per branch.
     */
    readonly workflows: GithubWorkflow[];
    private readonly options;
    private readonly _project;
    private readonly _task;
    private readonly pullRequestTitle;
    /**
     * Whether or not projen is also upgraded in this workflow,
     */
    readonly ignoresProjen: boolean;
    /**
     * Container definitions for the upgrade workflow.
     */
    containerOptions?: ContainerOptions;
    /**
     * A task run after the upgrade task.
     */
    readonly postUpgradeTask: Task;
    private readonly gitIdentity;
    private readonly postBuildSteps;
    constructor(project: NodeProject, options?: UpgradeDependenciesOptions);
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps: JobStep[]): void;
    preSynthesize(): void;
    private createTask;
    private createWorkflow;
    private createUpgrade;
    private createPr;
}
/**
 * Options for `UpgradeDependencies.workflowOptions`.
 */
export interface UpgradeDependenciesWorkflowOptions {
    /**
     * Schedule to run on.
     *
     * @default UpgradeDependenciesSchedule.DAILY
     */
    readonly schedule?: UpgradeDependenciesSchedule;
    /**
     * Which secret to use when creating the PR.
     *
     * When using the default github token, PR's created by this workflow
     * will not trigger any subsequent workflows (i.e the build workflow).
     * This is why this workflow also runs 'build' by default, and manually updates
     * the status check of the PR.
     *
     * If you pass a token that has the `workflow` permissions, you can skip running
     * build in this workflow by specifying `rebuild: false`.
     *
     * @see https://github.com/peter-evans/create-pull-request/issues/48
     * @default - default github token.
     */
    readonly secret?: string;
    /**
     * Labels to apply on the PR.
     *
     * @default - no labels.
     */
    readonly labels?: string[];
    /**
     * Job container options.
     *
     * @default - defaults
     */
    readonly container?: workflows.ContainerOptions;
    /**
     * List of branches to create PR's for.
     *
     * @default - All release branches configured for the project.
     */
    readonly branches?: string[];
    /**
     * The git identity to use for commits.
     * @default "github-actions@github.com"
     */
    readonly gitIdentity?: GitIdentity;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     */
    readonly runsOn?: string[];
}
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
export declare class UpgradeDependenciesSchedule {
    readonly cron: string[];
    /**
     * Disables automatic upgrades.
     */
    static readonly NEVER: UpgradeDependenciesSchedule;
    /**
     * At 00:00.
     */
    static readonly DAILY: UpgradeDependenciesSchedule;
    /**
     * At 00:00 on every day-of-week from Monday through Friday.
     */
    static readonly WEEKDAY: UpgradeDependenciesSchedule;
    /**
     * At 00:00 on Monday.
     */
    static readonly WEEKLY: UpgradeDependenciesSchedule;
    /**
     * At 00:00 on day-of-month 1.
     */
    static readonly MONTHLY: UpgradeDependenciesSchedule;
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron: string[]): UpgradeDependenciesSchedule;
    private constructor();
}
