"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const logging_1 = require("../logging");
function context(value) {
    return `\${{ ${value} }}`;
}
const REPO = context("github.repository");
const RUN_ID = context("github.run_id");
const RUN_URL = `https://github.com/${REPO}/actions/runs/${RUN_ID}`;
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        var _c, _d, _e, _f, _g, _h;
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.pullRequestTitle = (_c = options.pullRequestTitle) !== null && _c !== void 0 ? _c : "upgrade dependencies";
        this.ignoresProjen = (_d = this.options.ignoreProjen) !== null && _d !== void 0 ? _d : true;
        this.gitIdentity = (_f = (_e = options.workflowOptions) === null || _e === void 0 ? void 0 : _e.gitIdentity) !== null && _f !== void 0 ? _f : constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.postBuildSteps = [];
        this.containerOptions = (_g = options.workflowOptions) === null || _g === void 0 ? void 0 : _g.container;
        project.addDevDeps("npm-check-updates@^12");
        this.postUpgradeTask = (_h = project.tasks.tryFind("post-upgrade")) !== null && _h !== void 0 ? _h : project.tasks.addTask("post-upgrade", {
            description: "Runs after upgrading dependencies",
        });
        // create the upgrade task
        this._task = this.createTask();
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    // create a corresponding github workflow for each requested branch.
    preSynthesize() {
        var _c, _d, _e, _f, _g;
        if (this._project.github && ((_c = this.options.workflow) !== null && _c !== void 0 ? _c : true)) {
            // represents the default repository branch.
            // just like not specifying anything.
            const defaultBranch = undefined;
            const branches = (_g = (_e = (_d = this.options.workflowOptions) === null || _d === void 0 ? void 0 : _d.branches) !== null && _e !== void 0 ? _e : (_f = this._project.release) === null || _f === void 0 ? void 0 : _f.branches) !== null && _g !== void 0 ? _g : [defaultBranch];
            for (const branch of branches) {
                this.workflows.push(this.createWorkflow(this._task, this._project.github, branch));
            }
        }
    }
    createTask() {
        var _c, _d;
        const taskName = (_c = this.options.taskName) !== null && _c !== void 0 ? _c : "upgrade";
        const task = this._project.addTask(taskName, {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
        });
        const exclude = (_d = this.options.exclude) !== null && _d !== void 0 ? _d : [];
        if (this.ignoresProjen) {
            exclude.push("projen");
        }
        for (const dep of ["dev", "optional", "peer", "prod", "bundle"]) {
            const ncuCommand = [
                "npm-check-updates",
                "--dep",
                dep,
                "--upgrade",
                "--target=minor",
            ];
            if (exclude.length > 0) {
                ncuCommand.push(`--reject='${exclude.join(",")}'`);
            }
            if (this.options.include) {
                ncuCommand.push(`--filter='${this.options.include.join(",")}'`);
            }
            task.exec(ncuCommand.join(" "));
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        task.exec(this._project.package.installAndUpdateLockfileCommand);
        // run upgrade command to upgrade transitive deps as well
        task.exec(this._project.package.renderUpgradePackagesCommand(exclude, this.options.include));
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        task.exec(this._project.projenCommand);
        task.spawn(this.postUpgradeTask);
        return task;
    }
    createWorkflow(task, github, branch) {
        var _c, _d;
        const schedule = (_d = (_c = this.options.workflowOptions) === null || _c === void 0 ? void 0 : _c.schedule) !== null && _d !== void 0 ? _d : UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, branch) {
        var _c, _d;
        const runsOn = (_d = (_c = this.options.workflowOptions) === null || _c === void 0 ? void 0 : _c.runsOn) !== null && _d !== void 0 ? _d : ["ubuntu-latest"];
        // thats all we should need at this stage since all we do is clone.
        // note that this also prevents new code that is introduced in the upgrade
        // to have write access to anything, in case its somehow executed. (for example during build)
        const permissions = {
            contents: github_1.workflows.JobPermission.READ,
        };
        const steps = [
            {
                name: "Checkout",
                uses: "actions/checkout@v2",
                with: branch ? { ref: branch } : undefined,
            },
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.createUploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: permissions,
                runsOn: runsOn !== null && runsOn !== void 0 ? runsOn : ["ubuntu-latest"],
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        var _c, _d, _e, _f, _g, _h, _j, _k;
        const secretName = (_d = (_c = this.options.workflowOptions) === null || _c === void 0 ? void 0 : _c.secret) !== null && _d !== void 0 ? _d : workflow.projenTokenSecret;
        if (((_e = this.options.workflowOptions) === null || _e === void 0 ? void 0 : _e.secret) === workflow.projenTokenSecret) {
            logging_1.warn(`No need to specify "workflowOptions.secret" when it is the same as the default workflow projen token secret ("${workflow.projenTokenSecret}").`);
        }
        const token = context(`secrets.${secretName}`);
        const runsOn = (_g = (_f = this.options.workflowOptions) === null || _f === void 0 ? void 0 : _f.runsOn) !== null && _g !== void 0 ? _g : ["ubuntu-latest"];
        const workflowName = workflow.name;
        const branchName = `github-actions/${workflowName}`;
        const prStepId = "create-pr";
        const title = `chore(deps): ${this.pullRequestTitle}`;
        const description = [
            "Upgrades project dependencies. See details in [workflow run].",
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflow.name}" workflow*`,
        ].join("\n");
        const committer = `${this.gitIdentity.name} <${this.gitIdentity.email}>`;
        const steps = [
            ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                token: `\${{ secrets.${workflow.projenTokenSecret} }}`,
                ref: upgrade.ref,
            }),
            ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
            {
                name: "Create Pull Request",
                id: prStepId,
                uses: "peter-evans/create-pull-request@v3",
                with: {
                    // the pr can modify workflow files, so we need to use the custom
                    // secret if one is configured.
                    token: token,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    title: title,
                    labels: ((_j = (_h = this.options.workflowOptions) === null || _h === void 0 ? void 0 : _h.labels) === null || _j === void 0 ? void 0 : _j.join(",")) || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: (_k = this.options.signoff) !== null && _k !== void 0 ? _k : true,
                },
            },
        ];
        return {
            job: {
                name: "Create Pull Request",
                if: `\${{ needs.${upgrade.jobId}.outputs.${PATCH_CREATED_OUTPUT} }}`,
                needs: [upgrade.jobId],
                permissions: {
                    contents: github_1.workflows.JobPermission.WRITE,
                    pullRequests: github_1.workflows.JobPermission.WRITE,
                },
                runsOn: runsOn !== null && runsOn !== void 0 ? runsOn : ["ubuntu-latest"],
                steps: steps,
            },
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.52.53" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    constructor(cron) {
        this.cron = cron;
    }
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.52.53" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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