"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogLevel = exports.Logger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const chalk = require("chalk");
const common_1 = require("./common");
const component_1 = require("./component");
const logging_1 = require("./logging");
const util_1 = require("./util");
/**
 * Project-level logging utilities.
 */
class Logger extends component_1.Component {
    constructor(project, options = {}) {
        var _b, _c;
        super(project);
        // if we are running inside a test, default to no logs
        const defaultLevel = common_1.IS_TEST_RUN ? LogLevel.OFF : LogLevel.INFO;
        this.level = (_b = options.level) !== null && _b !== void 0 ? _b : defaultLevel;
        this.usePrefix = (_c = options.usePrefix) !== null && _c !== void 0 ? _c : false;
    }
    /**
     * Log a message to stderr with a given logging level. The message will be
     * printed as long as `logger.level` is set to the message's severity or higher.
     *
     * @param level Logging verbosity
     * @param text strings or objects to print
     */
    log(level, ...text) {
        if (level === LogLevel.OFF) {
            throw new Error("Cannot log a message with level LogLevel.OFF");
        }
        let maxLevel = this.level;
        // logging level can be overridden through the --debug CLI option
        if (util_1.isTruthy(process.env.DEBUG) && maxLevel < LogLevel.DEBUG) {
            maxLevel = LogLevel.DEBUG;
        }
        if (level <= maxLevel) {
            const color = this.colorForLogLevel(level);
            const prefix = this.usePrefix ? `[${this.project.name}] ` : "";
            console.error(`${logging_1.ICON} ${prefix}${color(...text)}`);
        }
    }
    /**
     * Log a message to stderr with VERBOSE severity
     * @param text strings or objects to print
     */
    verbose(...text) {
        this.log(LogLevel.VERBOSE, ...text);
    }
    /**
     * Log a message to stderr with DEBUG severity
     * @param text strings or objects to print
     */
    debug(...text) {
        this.log(LogLevel.DEBUG, ...text);
    }
    /**
     * Log a message to stderr with INFO severity
     * @param text strings or objects to print
     */
    info(...text) {
        this.log(LogLevel.INFO, ...text);
    }
    /**
     * Log a message to stderr with WARN severity
     * @param text strings or objects to print
     */
    warn(...text) {
        this.log(LogLevel.WARN, ...text);
    }
    /**
     * Log a message to stderr with ERROR severity
     * @param text strings or objects to print
     */
    error(...text) {
        this.log(LogLevel.ERROR, ...text);
    }
    colorForLogLevel(level) {
        switch (level) {
            case LogLevel.ERROR:
                return chalk.red;
            case LogLevel.WARN:
                return chalk.yellow;
            case LogLevel.INFO:
                return chalk.cyan;
            case LogLevel.DEBUG:
                return chalk.gray;
            case LogLevel.VERBOSE:
                return chalk.white;
            default:
                return chalk.white;
        }
    }
}
exports.Logger = Logger;
_a = JSII_RTTI_SYMBOL_1;
Logger[_a] = { fqn: "projen.Logger", version: "0.52.53" };
/**
 * Logging verbosity.
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["OFF"] = "00.off";
    LogLevel["ERROR"] = "10.error";
    LogLevel["WARN"] = "20.warn";
    LogLevel["INFO"] = "30.info";
    LogLevel["DEBUG"] = "40.debug";
    LogLevel["VERBOSE"] = "50.verbose";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
//# sourceMappingURL=data:application/json;base64,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