"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const glob = require("glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
/**
 * Base project
 */
class Project {
    constructor(options) {
        var _b, _c, _d;
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.initProject = render_options_1.resolveInitProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = util_1.isTruthy(process.env.PROJEN_EJECTING);
        if (this.ejected) {
            this.projenCommand = "scripts/run-task";
        }
        else {
            this.projenCommand = (_b = options.projenCommand) !== null && _b !== void 0 ? _b : "npx projen";
        }
        this.outdir = this.determineOutdir(options.outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        (_c = this.parent) === null || _c === void 0 ? void 0 : _c._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore");
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            this.ejectTask = this.tasks.addTask("eject", {
                description: "Remove projen from the project",
                env: {
                    PROJEN_EJECTING: "true",
                },
            });
            this.ejectTask.spawn(this.defaultTask);
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = (_d = options.projenrcJson) !== null && _d !== void 0 ? _d : false;
        if (projenrcJson) {
            new projenrc_json_1.Projenrc(this, options.projenrcJsonOptions);
        }
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => f.path.replace(/\\/g, "/")),
                }),
            });
        }
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return [...this._components];
    }
    /**
     * All files in this project.
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.files.map((f) => f.path.replace(/\\/g, "/")), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                nodir: true,
                absolute: true,
            });
            for (const file of files) {
                fs_1.renameSync(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
    /**
     * Resolves the project's output directory.
     */
    determineOutdir(outdirOption) {
        if (this.parent && outdirOption && path.isAbsolute(outdirOption)) {
            throw new Error('"outdir" must be a relative path');
        }
        // if this is a subproject, it is relative to the parent
        if (this.parent) {
            if (!outdirOption) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            return path.resolve(this.parent.outdir, outdirOption);
        }
        // if this is running inside a test and outdir is not explicitly set
        // use a temp directory (unless cwd is aleady under tmp)
        if (common_1.IS_TEST_RUN && !outdirOption) {
            const realCwd = fs_1.realpathSync(process.cwd());
            const realTmp = fs_1.realpathSync(os_1.tmpdir());
            if (realCwd.startsWith(realTmp)) {
                return path.resolve(realCwd, outdirOption !== null && outdirOption !== void 0 ? outdirOption : ".");
            }
            return fs_1.mkdtempSync(path.join(os_1.tmpdir(), "projen."));
        }
        return path.resolve(outdirOption !== null && outdirOption !== void 0 ? outdirOption : ".");
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.52.53" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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