"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PoetryPyproject = exports.Poetry = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const task_runtime_1 = require("../task-runtime");
const toml_1 = require("../toml");
const util_1 = require("../util");
/**
 * Manage project dependencies, virtual environments, and packaging through the
 * poetry CLI tool.
 */
class Poetry extends component_1.Component {
    constructor(project, options) {
        var _c;
        super(project);
        this.installTask = project.addTask("install", {
            description: "Install and upgrade dependencies",
            exec: "poetry update",
        });
        this.project.tasks.addEnvironment("VIRTUAL_ENV", "$(poetry env info -p)");
        this.project.tasks.addEnvironment("PATH", "$(echo $(poetry env info -p)/bin:$PATH)");
        project.packageTask.exec("poetry build");
        this.publishTestTask = project.addTask("publish:test", {
            description: "Uploads the package against a test PyPI endpoint.",
            exec: "poetry publish -r testpypi",
        });
        this.publishTask = project.addTask("publish", {
            description: "Uploads the package to PyPI.",
            exec: "poetry publish",
        });
        new PoetryPyproject(project, {
            name: project.name,
            version: options.version,
            description: (_c = options.description) !== null && _c !== void 0 ? _c : "",
            license: options.license,
            authors: [`${options.authorName} <${options.authorEmail}>`],
            homepage: options.homepage,
            classifiers: options.classifiers,
            ...options.poetryOptions,
            dependencies: () => this.synthDependencies(),
            devDependencies: () => this.synthDevDependencies(),
        });
        new toml_1.TomlFile(project, "poetry.toml", {
            committed: false,
            obj: {
                repositories: {
                    testpypi: {
                        url: "https://test.pypi.org/legacy/",
                    },
                },
            },
        });
    }
    synthDependencies() {
        const dependencies = {};
        let pythonDefined = false;
        for (const pkg of this.project.deps.all) {
            if (pkg.name === "python") {
                pythonDefined = true;
            }
            if (pkg.type === dependencies_1.DependencyType.RUNTIME) {
                dependencies[pkg.name] = pkg.version;
            }
        }
        if (!pythonDefined) {
            // Python version must be defined for poetry projects. Default to ^3.6.
            dependencies.python = "^3.6";
        }
        return dependencies;
    }
    synthDevDependencies() {
        const dependencies = {};
        for (const pkg of this.project.deps.all) {
            if ([dependencies_1.DependencyType.DEVENV].includes(pkg.type)) {
                dependencies[pkg.name] = pkg.version;
            }
        }
        return dependencies;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.DEVENV);
    }
    /**
     * Initializes the virtual environment if it doesn't exist (called during post-synthesis).
     */
    setupEnvironment() {
        const result = util_1.execOrUndefined("which poetry", {
            cwd: this.project.outdir,
        });
        if (!result) {
            this.project.logger.info("Unable to setup an environment since poetry is not installed. Please install poetry (https://python-poetry.org/docs/) or use a different component for managing environments such as 'venv'.");
        }
        let envPath = util_1.execOrUndefined("poetry env info -p", {
            cwd: this.project.outdir,
        });
        if (!envPath) {
            this.project.logger.info("Setting up a virtual environment...");
            util_1.exec("poetry env use python", { cwd: this.project.outdir });
            envPath = util_1.execOrUndefined("poetry env info -p", {
                cwd: this.project.outdir,
            });
            this.project.logger.info(`Environment successfully created (located in ${envPath}}).`);
        }
    }
    /**
     * Installs dependencies (called during post-synthesis).
     */
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        runtime.runTask(this.installTask.name);
    }
}
exports.Poetry = Poetry;
_a = JSII_RTTI_SYMBOL_1;
Poetry[_a] = { fqn: "projen.python.Poetry", version: "0.52.53" };
/**
 * Represents configuration of a pyproject.toml file for a Poetry project.
 *
 * @see https://python-poetry.org/docs/pyproject/
 */
class PoetryPyproject extends component_1.Component {
    constructor(project, options) {
        super(project);
        const decamelisedOptions = util_1.decamelizeKeysRecursively(options, {
            separator: "-",
        });
        this.file = new toml_1.TomlFile(project, "pyproject.toml", {
            omitEmpty: false,
            obj: {
                "build-system": {
                    requires: ["poetry_core>=1.0.0"],
                    "build-backend": "poetry.core.masonry.api",
                },
                tool: {
                    poetry: {
                        ...decamelisedOptions,
                    },
                },
            },
        });
    }
}
exports.PoetryPyproject = PoetryPyproject;
_b = JSII_RTTI_SYMBOL_1;
PoetryPyproject[_b] = { fqn: "projen.python.PoetryPyproject", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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