"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * Allows writing projenrc files in python.
 *
 * This will install `projen` as a Python dependency and will add a
 * `synth` task which will run `.projenrc.py`.
 */
class Projenrc extends component_1.Component {
    constructor(project, options = {}) {
        var _b, _c, _d;
        super(project);
        const projenVersion = (_b = options.projenVersion) !== null && _b !== void 0 ? _b : common_1.PROJEN_VERSION;
        this.rcfile = (_c = options.filename) !== null && _c !== void 0 ? _c : ".projenrc.py";
        project.deps.addDependency(`projen@${projenVersion}`, dependencies_1.DependencyType.DEVENV);
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        (_d = project.defaultTask) === null || _d === void 0 ? void 0 : _d.exec("python .projenrc.py");
        // if this is a new project, generate a skeleton for projenrc.py
        this.generateProjenrc();
    }
    generateProjenrc() {
        var _b, _c, _d;
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const optionsTypeFqn = (_d = (_c = (_b = jsiiType.initializer) === null || _b === void 0 ? void 0 : _b.parameters) === null || _c === void 0 ? void 0 : _c[0].type) === null || _d === void 0 ? void 0 : _d.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const pythonFile = path_1.join(this.project.outdir, this.rcfile);
        // skip if file exists
        if (fs_extra_1.existsSync(pythonFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        // const openBlock = (line: string = '') => { emit(line + ' {'); indent++; };
        // const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = option.fqn;
            }
        }
        const { renderedOptions, imports } = renderPythonOptions(indent, optionFqns, bootstrap.args);
        emit(toPythonImport(jsiiFqn));
        for (const fqn of imports) {
            emit(toPythonImport(fqn));
        }
        emit();
        emit(`project = ${jsiiType.name}(${renderedOptions})`);
        emit();
        emit("project.synth()");
        fs_extra_1.mkdirpSync(path_1.dirname(pythonFile));
        fs_extra_1.writeFileSync(pythonFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${pythonFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.python.Projenrc", version: "0.52.53" };
function renderPythonOptions(indent, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        // no options
        return { renderedOptions: "", imports };
    }
    const lines = [""];
    for (const [name, value] of Object.entries(initOptions)) {
        const { pythonValue, importName } = toPythonValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`${toPythonProperty(name)}=${pythonValue},`);
    }
    const renderedOptions = lines
        .join(`\n${" ".repeat((indent + 1) * 4)}`)
        .concat("\n");
    return { renderedOptions, imports };
}
function toPythonProperty(prop) {
    return case_1.snake(prop);
}
function toPythonValue(value, name, optionFqns) {
    if (typeof value === "boolean") {
        return { pythonValue: value ? "True" : "False" };
    }
    else if (typeof value === "number") {
        return { pythonValue: JSON.stringify(value) };
    }
    else if (typeof value === "string") {
        if (optionFqns[name] !== undefined) {
            const parts = optionFqns[name].split(".");
            const base = parts[parts.length - 1];
            const choice = String(value).toUpperCase().replace(/-/g, "_");
            return { pythonValue: `${base}.${choice}`, importName: optionFqns[name] };
        }
        else {
            return { pythonValue: JSON.stringify(value) };
        }
    }
    else if (value === undefined || value === null) {
        return { pythonValue: "None" };
    }
    else {
        return { pythonValue: JSON.stringify(value) };
    }
}
function toPythonImport(fqn) {
    const parts = fqn.split(".");
    if (parts.length === 1) {
        return `import ${parts[0]}`;
    }
    else {
        return `from ${parts.slice(0, -1).join(".")} import ${parts[parts.length - 1]}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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