"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    var _a;
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const prefix = (_a = options.tagPrefix) !== null && _a !== void 0 ? _a : "";
    const bumpFile = path_1.join(cwd, options.bumpFile);
    const changelogFile = path_1.join(cwd, options.changelog);
    const releaseTagFile = path_1.join(cwd, options.releaseTagFile);
    await fs_extra_1.mkdirp(path_1.dirname(bumpFile));
    await fs_extra_1.mkdirp(path_1.dirname(changelogFile));
    await fs_extra_1.mkdirp(path_1.dirname(releaseTagFile));
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({
        cwd,
        major,
        prerelease,
        prefix,
    });
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // check if the latest commit already has a version tag
    const currentTags = util_1.execCapture("git tag --points-at HEAD", { cwd })
        .toString("utf8")
        .split("\n");
    logging.info(`Tags listed on current commit: ${currentTags}`);
    let skipBump = false;
    if (currentTags.includes(latestTag)) {
        logging.info("Skipping bump...");
        skipBump = true;
        // delete the existing tag (locally)
        // if we don't do this, standard-version generates an empty changelog
        util_1.exec(`git tag --delete ${latestTag}`, { cwd });
    }
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, ".versionrc.json");
    await generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, options.versionrcOptions);
    const cmd = ["npx", "standard-version@^9"];
    if (isFirstRelease) {
        cmd.push("--first-release");
    }
    util_1.exec(cmd.join(" "), { cwd });
    // add the tag back if it was previously removed
    if (currentTags.includes(latestTag)) {
        util_1.exec(`git tag ${latestTag}`, { cwd });
    }
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_extra_1.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_extra_1.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, "utf8"));
}
function generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, configOptions) {
    return fs_extra_1.writeFile(rcfile, JSON.stringify({
        ...{
            packageFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            bumpFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            commitAll: false,
            infile: changelogFile,
            prerelease: prerelease,
            header: "",
            skip: {
                commit: true,
                tag: true,
                bump: skipBump,
            },
            ...configOptions,
        },
    }, undefined, 2));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    const prefixFilter = major !== undefined ? `${prefix}v${major}.*` : `${prefix}v*`;
    const listGitTags = [
        "git",
        '-c "versionsort.suffix=-"',
        "tag",
        '--sort="-version:refname"',
        "--list",
        `"${prefixFilter}"`,
    ].join(" ");
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString("utf8");
    let tags = stdout === null || stdout === void 0 ? void 0 : stdout.split("\n");
    // if "pre" is set, filter versions that end with "-PRE.ddd".
    if (prerelease) {
        tags = tags.filter((x) => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    }
    tags = tags.filter((x) => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major !== null && major !== void 0 ? major : 0}.0.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith("v")) {
        latestVersion = latestVersion.substr(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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