"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * Environment variables:
 *
 * - OUTFILE: (required) the name of the JSON output file (the "version" field
 *   will be updated with the latest version)
 * - PRERELEASE: (optional) a prerelease tag to use (e.g. "beta")
 * - MAJOR: major version number NN to filter (tags are filtered by "vNN."
 *   prefix). if not specified, the last major version is selected
 * - CHANGELOG: name of changelog file to create
 * - RELEASE_TAG_PREFIX: (optional) a prefix to apply to the release tag
 *
 */
const logging = require("../logging");
const bump_version_1 = require("./bump-version");
const versionFile = process.env.OUTFILE;
const prerelease = process.env.PRERELEASE;
const major = process.env.MAJOR;
const changelog = process.env.CHANGELOG;
const bumpFile = process.env.BUMPFILE;
const releaseTagFile = process.env.RELEASETAG;
const prefix = process.env.RELEASE_TAG_PREFIX;
const versionrcOptions = process.env.VERSIONRCOPTIONS;
if (!versionFile) {
    throw new Error("OUTFILE is required");
}
if (!changelog) {
    throw new Error("CHANGELOG is required");
}
if (!bumpFile) {
    throw new Error("BUMPFILE is required");
}
if (!releaseTagFile) {
    throw new Error("RELEASETAG is required");
}
const majorVersion = major == null || major === "" ? undefined : parseInt(major);
if (Number.isNaN(majorVersion)) {
    throw new Error(`MAJOR must be a number: ${majorVersion}`);
}
const opts = {
    versionFile: versionFile,
    changelog: changelog,
    majorVersion: majorVersion,
    prerelease: prerelease,
    bumpFile: bumpFile,
    releaseTagFile: releaseTagFile,
    tagPrefix: prefix,
    // doesn't work with long customization
    versionrcOptions: JSON.parse(versionrcOptions !== null && versionrcOptions !== void 0 ? versionrcOptions : "{}"),
};
logging.debug(opts);
bump_version_1.bump(process.cwd(), opts).catch((e) => {
    console.log(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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