"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isAwsCodeArtifactRegistry = exports.Publisher = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11.x" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.16.0" } },
    dotnet: { dotnet: { version: "3.x" } },
};
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion = (_c = (_b = options.publibVersion) !== null && _b !== void 0 ? _b : options.jsiiReleaseVersion) !== null && _c !== void 0 ? _c : PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = (_d = options.dryRun) !== null && _d !== void 0 ? _d : false;
        this.failureIssue = (_e = options.failureIssue) !== null && _e !== void 0 ? _e : false;
        this.failureIssueLabel = (_f = options.failureIssueLabel) !== null && _f !== void 0 ? _f : "failed-release";
        this.runsOn = (_g = options.workflowRunsOn) !== null && _g !== void 0 ? _g : ["ubuntu-latest"];
        this.publishTasks = (_h = options.publishTasks) !== null && _h !== void 0 ? _h : false;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        var _b;
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = (_b = options.gitBranch) !== null && _b !== void 0 ? _b : "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? "publish:git"
            : `publish:git:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile !== null && projectChangelogFile !== void 0 ? projectChangelogFile : "",
                VERSION_FILE: versionFile,
            },
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghRelease = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_REF",
        ].join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        this.addPublishJob(() => {
            var _b;
            return {
                name: "github",
                registryName: "GitHub Releases",
                prePublishSteps: (_b = options.prePublishSteps) !== null && _b !== void 0 ? _b : this._gitHubPrePublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                    GITHUB_REPOSITORY: "${{ github.repository }}",
                    GITHUB_REF: "${{ github.ref }}",
                },
                run: idempotentRelease,
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        var _b;
        const isGitHubPackages = (_b = options.registry) === null || _b === void 0 ? void 0 : _b.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const npmToken = node_package_1.defaultNpmToken(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        this.addPublishJob((_branch, branchOptions) => {
            var _b, _c, _d, _e, _f, _g, _h, _j;
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            return {
                name: "npm",
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps: (_b = options.prePublishSteps) !== null && _b !== void 0 ? _b : [],
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: (_d = (_c = branchOptions.npmDistTag) !== null && _c !== void 0 ? _c : options.distTag) !== null && _d !== void 0 ? _d : "latest",
                    NPM_REGISTRY: options.registry,
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact
                        ? secret((_f = (_e = options.codeArtifactOptions) === null || _e === void 0 ? void 0 : _e.accessKeyIdSecret) !== null && _f !== void 0 ? _f : "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact
                        ? secret((_h = (_g = options.codeArtifactOptions) === null || _g === void 0 ? void 0 : _g.secretAccessKeySecret) !== null && _h !== void 0 ? _h : "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact
                        ? (_j = options.codeArtifactOptions) === null || _j === void 0 ? void 0 : _j.roleToAssume : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        this.addPublishJob((_branch, _branchOptions) => {
            var _b, _c;
            return ({
                name: "nuget",
                publishTools: PUBLIB_TOOLCHAIN.dotnet,
                prePublishSteps: (_b = options.prePublishSteps) !== null && _b !== void 0 ? _b : [],
                run: this.publibCommand("publib-nuget"),
                registryName: "NuGet Gallery",
                workflowEnv: {
                    NUGET_API_KEY: secret((_c = options.nugetApiKeySecret) !== null && _c !== void 0 ? _c : "NUGET_API_KEY"),
                },
            });
        });
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        var _b, _c;
        const isGitHubPackages = (_b = options.mavenRepositoryUrl) === null || _b === void 0 ? void 0 : _b.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = (_c = options.mavenServerId) !== null && _c !== void 0 ? _c : (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        this.addPublishJob((_branch, _branchOptions) => {
            var _b, _c, _d, _e, _f, _g;
            return ({
                name: "maven",
                registryName: "Maven Central",
                publishTools: PUBLIB_TOOLCHAIN.java,
                prePublishSteps: (_b = options.prePublishSteps) !== null && _b !== void 0 ? _b : [],
                run: this.publibCommand("publib-maven"),
                env: {
                    MAVEN_ENDPOINT: options.mavenEndpoint,
                    MAVEN_SERVER_ID: mavenServerId,
                    MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
                },
                workflowEnv: {
                    MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                        ? undefined
                        : secret((_c = options.mavenGpgPrivateKeySecret) !== null && _c !== void 0 ? _c : "MAVEN_GPG_PRIVATE_KEY"),
                    MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                        ? undefined
                        : secret((_d = options.mavenGpgPrivateKeyPassphrase) !== null && _d !== void 0 ? _d : "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                    MAVEN_PASSWORD: secret((_e = options.mavenPassword) !== null && _e !== void 0 ? _e : (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                    MAVEN_USERNAME: isGitHubActor
                        ? "${{ github.actor }}"
                        : secret((_f = options.mavenUsername) !== null && _f !== void 0 ? _f : "MAVEN_USERNAME"),
                    MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                        ? undefined
                        : secret((_g = options.mavenStagingProfileId) !== null && _g !== void 0 ? _g : "MAVEN_STAGING_PROFILE_ID"),
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
            });
        });
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        this.addPublishJob((_branch, _branchOptions) => {
            var _b, _c, _d;
            return ({
                name: "pypi",
                registryName: "PyPI",
                publishTools: PUBLIB_TOOLCHAIN.python,
                prePublishSteps: (_b = options.prePublishSteps) !== null && _b !== void 0 ? _b : [],
                run: this.publibCommand("publib-pypi"),
                env: {
                    TWINE_REPOSITORY_URL: options.twineRegistryUrl,
                },
                workflowEnv: {
                    TWINE_USERNAME: secret((_c = options.twineUsernameSecret) !== null && _c !== void 0 ? _c : "TWINE_USERNAME"),
                    TWINE_PASSWORD: secret((_d = options.twinePasswordSecret) !== null && _d !== void 0 ? _d : "TWINE_PASSWORD"),
                },
            });
        });
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        this.addPublishJob((_branch, _branchOptions) => {
            var _b, _c, _d, _e;
            return ({
                name: "golang",
                publishTools: PUBLIB_TOOLCHAIN.go,
                prePublishSteps: (_b = options.prePublishSteps) !== null && _b !== void 0 ? _b : [],
                run: this.publibCommand("publib-golang"),
                registryName: "GitHub Go Module Repository",
                env: {
                    GITHUB_REPO: options.githubRepo,
                    GIT_BRANCH: options.gitBranch,
                    GIT_USER_NAME: (_c = options.gitUserName) !== null && _c !== void 0 ? _c : constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                    GIT_USER_EMAIL: (_d = options.gitUserEmail) !== null && _d !== void 0 ? _d : constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                    GIT_COMMIT_MESSAGE: options.gitCommitMessage,
                },
                workflowEnv: {
                    GITHUB_TOKEN: secret((_e = options.githubTokenSecret) !== null && _e !== void 0 ? _e : "GO_GITHUB_TOKEN"),
                },
            });
        });
    }
    addPublishJob(factory) {
        this._jobFactories.push((branch, branchOptions) => {
            var _b, _c;
            const opts = factory(branch, branchOptions);
            const jobname = `release_${opts.name}`;
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries((_b = opts.workflowEnv) !== null && _b !== void 0 ? _b : {}).filter(([_, value]) => value != undefined);
            for (const [name, expression] of workflowEnvEntries) {
                requiredEnv.push(name);
                jobEnv[name] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${opts.name.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v2",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR,
                    },
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
            ];
            const perms = (_c = opts.permissions) !== null && _c !== void 0 ? _c : { contents: workflows_model_1.JobPermission.READ };
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        run: 'echo "::set-output name=VERSION::$(cat dist/version.txt)"',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    tools: {
                        node: { version: "14.x" },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId],
                    runsOn: this.runsOn,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.52.53" };
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
//# sourceMappingURL=data:application/json;base64,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