"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const LATEST_COMMIT_OUTPUT = "latest_commit";
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(project);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = project.github;
        this.buildTask = options.task;
        this.preBuildSteps = (_b = options.releaseWorkflowSetupSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.artifactsDirectory = (_d = options.artifactsDirectory) !== null && _d !== void 0 ? _d : "dist";
        this.versionFile = options.versionFile;
        this.releaseTrigger = (_e = options.releaseTrigger) !== null && _e !== void 0 ? _e : release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!(((_f = options.releaseEveryCommit) !== null && _f !== void 0 ? _f : true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            workflowRunsOn: options.workflowRunsOn,
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
        });
        const githubRelease = (_g = options.githubRelease) !== null && _g !== void 0 ? _g : true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            workflowName: (_h = options.releaseWorkflowName) !== null && _h !== void 0 ? _h : "release",
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries((_j = options.releaseBranches) !== null && _j !== void 0 ? _j : {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        var _b;
        const workflowName = (_b = branch.workflowName) !== null && _b !== void 0 ? _b : `release-${branchName}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.RELEASE_TAG_PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec("git diff --ignore-space-at-eol --exit-code");
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            run: `echo ::set-output name=${LATEST_COMMIT_OUTPUT}::"$(git ls-remote origin -h \${{ github.ref }} | cut -f1)"`,
        });
        postBuildSteps.push({
            name: "Upload artifact",
            if: noNewCommits,
            uses: "actions/upload-artifact@v2.1.1",
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: this.artifactsDirectory,
            },
        });
        if (this.github && !this.releaseTrigger.isManual) {
            return new github_1.TaskWorkflow(this.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseTrigger.schedule
                        ? [{ cron: this.releaseTrigger.schedule }]
                        : undefined,
                    push: this.releaseTrigger.isContinuous
                        ? { branches: [branchName] }
                        : undefined,
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    "fetch-depth": 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
                runsOn: this.workflowRunsOn,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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