"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SampleDir = exports.SampleFile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs = require("fs-extra");
const glob = require("glob");
const component_1 = require("./component");
const util_1 = require("./util");
/**
 * Produces a file with the given contents but only once, if the file doesn't already exist.
 * Use this for creating example code files or other resources.
 */
class SampleFile extends component_1.Component {
    /**
     * Creates a new SampleFile object
     * @param project - the project to tie this file to.
     * @param filePath - the relative path in the project to put the file
     * @param options - the options for the file.
     */
    constructor(project, filePath, options) {
        super(project);
        if (options.contents && options.sourcePath) {
            throw new Error("Cannot specify both 'contents' and 'source' fields.");
        }
        if (!options.contents && !options.sourcePath) {
            throw new Error("Must specify at least one of 'contents' or 'source'.");
        }
        this.filePath = filePath;
        this.options = options;
    }
    synthesize() {
        let contents;
        if (this.options.contents) {
            contents = this.options.contents;
        }
        else if (this.options.sourcePath) {
            contents = fs.readFileSync(this.options.sourcePath);
        }
        this.writeOnceFileContents(this.project.outdir, this.filePath, contents !== null && contents !== void 0 ? contents : "");
    }
    /**
     * A helper function that will write the file once and return if it was written or not.
     * @param dir - the directory for the new file
     * @param filename - the filename for the new file
     * @param contents - the contents of the file to write
     * @return boolean - whether a new file was written or not.
     * @private
     */
    writeOnceFileContents(dir, filename, contents) {
        const fullFilename = path.join(dir, filename);
        if (fs.existsSync(fullFilename)) {
            return;
        }
        util_1.writeFile(fullFilename, contents, { readonly: false });
    }
}
exports.SampleFile = SampleFile;
_a = JSII_RTTI_SYMBOL_1;
SampleFile[_a] = { fqn: "projen.SampleFile", version: "0.52.53" };
/**
 * Renders the given files into the directory if the directory does not exist. Use this to create sample code files
 */
class SampleDir extends component_1.Component {
    /**
     * Create sample files in the given directory if the given directory does not exist
     * @param project Parent project to add files to.
     * @param dir directory to add files to. If directory already exists, nothing is added.
     * @param options options for which files to create.
     */
    constructor(project, dir, options) {
        super(project);
        if (!options.files && !options.sourceDir) {
            throw new Error("Must specify at least one of 'files' or 'source'.");
        }
        this.dir = dir;
        this.options = options;
    }
    synthesize() {
        const fullOutdir = path.join(this.project.outdir, this.dir);
        if (fs.pathExistsSync(fullOutdir)) {
            return;
        }
        if (this.options.sourceDir) {
            const basedir = this.options.sourceDir;
            const files = glob.sync("**", {
                cwd: basedir,
                nodir: true,
                dot: true,
            }); // returns relative file paths with POSIX separators
            for (const file of files) {
                const sourcePath = path.join(basedir, file);
                const targetPath = path.join(fullOutdir, file);
                fs.mkdirpSync(path.dirname(targetPath));
                fs.copyFileSync(sourcePath, targetPath);
                fs.chmodSync(targetPath, util_1.getFilePermissions({ readonly: false, executable: false }));
            }
        }
        for (const filename in this.options.files) {
            util_1.writeFile(path.join(fullOutdir, filename), this.options.files[filename]);
        }
    }
}
exports.SampleDir = SampleDir;
_b = JSII_RTTI_SYMBOL_1;
SampleDir[_b] = { fqn: "projen.SampleDir", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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