"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const path = require("path");
const util_1 = require("util");
const chalk = require("chalk");
const common_1 = require("./common");
const logging = require("./logging");
const ENV_TRIM_LEN = 20;
/**
 * The runtime component of the tasks engine.
 */
class TaskRuntime {
    constructor(workdir) {
        this.workdir = path_1.resolve(workdir);
        const manifestPath = path_1.join(this.workdir, TaskRuntime.MANIFEST_FILE);
        this.manifest = fs_1.existsSync(manifestPath)
            ? JSON.parse(fs_1.readFileSync(manifestPath, "utf-8"))
            : { tasks: {} };
    }
    /**
     * The tasks in this project.
     */
    get tasks() {
        var _b;
        return Object.values((_b = this.manifest.tasks) !== null && _b !== void 0 ? _b : {});
    }
    /**
     * Find a task by name, or `undefined` if not found.
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * Runs the task.
     * @param name The task name.
     */
    runTask(name, parents = []) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.TaskRuntime", version: "0.52.53" };
/**
 * The project-relative path of the tasks manifest file.
 */
TaskRuntime.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, "tasks.json");
class RunTask {
    constructor(runtime, task, parents = []) {
        var _b, _c, _d, _e;
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = (_b = task.cwd) !== null && _b !== void 0 ? _b : this.runtime.workdir;
        this.parents = parents;
        if (!task.steps || task.steps.length === 0) {
            this.logDebug(chalk.gray("No actions have been specified for this task."));
            return;
        }
        this.env = this.resolveEnvironment(parents);
        const envlogs = [];
        for (const [k, v] of Object.entries(this.env)) {
            const vv = v !== null && v !== void 0 ? v : "";
            const trimmed = vv.length > ENV_TRIM_LEN ? vv.substr(0, ENV_TRIM_LEN) + "..." : vv;
            envlogs.push(`${k}=${trimmed}`);
        }
        if (envlogs.length) {
            this.logDebug(chalk.gray(`${chalk.underline("env")}: ${envlogs.join(" ")}`));
        }
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log("condition exited with non-zero - skipping");
            return;
        }
        // verify we required environment variables are defined
        const merged = { ...process.env, ...this.env };
        const missing = new Array();
        for (const name of (_c = task.requiredEnv) !== null && _c !== void 0 ? _c : []) {
            if (!(name in merged)) {
                missing.push(name);
            }
        }
        if (missing.length > 0) {
            throw new Error(`missing required environment variables: ${missing.join(",")}`);
        }
        for (const step of task.steps) {
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name]);
            }
            const execs = step.exec ? [step.exec] : [];
            if (step.builtin) {
                execs.push(this.renderBuiltin(step.builtin));
            }
            for (const exec of execs) {
                let command = "";
                let hasError = false;
                const cmd = exec.split(" ")[0];
                if (os_1.platform() == "win32" && ["mkdir", "mv", "rm"].includes(cmd)) {
                    command = `shx ${exec}`;
                }
                else {
                    command = exec;
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if ((_e = (_d = e) === null || _d === void 0 ? void 0 : _d.message) === null || _e === void 0 ? void 0 : _e.startsWith("non-zero exit code:")) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${path_1.resolve(cwd !== null && cwd !== void 0 ? cwd : this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(task) {
        // no condition, carry on
        if (!task.condition) {
            return true;
        }
        this.log(chalk.gray(`${chalk.underline("condition")}: ${task.condition}`));
        const result = this.shell({
            command: task.condition,
            logprefix: "condition: ",
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     */
    resolveEnvironment(parents) {
        var _b, _c, _d, _e, _f, _g;
        let env = (_b = this.runtime.manifest.env) !== null && _b !== void 0 ? _b : {};
        // add env from all parent tasks one by one
        for (const parent of parents) {
            env = {
                ...env,
                ...((_d = (_c = this.runtime.tryFindTask(parent)) === null || _c === void 0 ? void 0 : _c.env) !== null && _d !== void 0 ? _d : {}),
            };
        }
        // apply the task's environment last
        env = {
            ...env,
            ...((_e = this.task.env) !== null && _e !== void 0 ? _e : {}),
        };
        const output = {};
        for (const [key, value] of Object.entries(env !== null && env !== void 0 ? env : {})) {
            if (value.startsWith("$(") && value.endsWith(")")) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : (_g = (_f = result.stderr) === null || _f === void 0 ? void 0 : _f.toString()) !== null && _g !== void 0 ? _g : "unknown error";
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString("utf-8").trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(" » ");
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    logDebug(...args) {
        logging.debug(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return util_1.format(`${chalk.underline(this.fullname)} |`, ...args);
    }
    shell(options) {
        var _b, _c;
        const quiet = (_b = options.quiet) !== null && _b !== void 0 ? _b : false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(" "));
        }
        const cwd = (_c = options.cwd) !== null && _c !== void 0 ? _c : this.workdir;
        if (!fs_1.existsSync(cwd) || !fs_1.statSync(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return child_process_1.spawnSync(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: "inherit",
            env: {
                ...process.env,
                ...this.env,
            },
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ["inherit", "pipe", "inherit"],
            },
        });
    }
    renderBuiltin(builtin) {
        const moduleRoot = path_1.dirname(require.resolve("../package.json"));
        const program = require.resolve(path_1.join(moduleRoot, "lib", `${builtin}.task.js`));
        return `${process.execPath} ${program}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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