"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Task = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A task that can be performed on the project. Modeled as a series of shell
 * commands and subtasks.
 */
class Task {
    constructor(name, props = {}) {
        var _b;
        this.name = name;
        this._description = props.description;
        this.condition = props.condition;
        this.cwd = props.cwd;
        this._locked = false;
        this._env = (_b = props.env) !== null && _b !== void 0 ? _b : {};
        this._steps = [];
        this.requiredEnv = props.requiredEnv;
        if (props.exec) {
            this.exec(props.exec);
        }
    }
    /**
     * Forbid additional changes to this task.
     */
    lock() {
        this._locked = true;
    }
    /**
     * Returns the description of this task.
     */
    get description() {
        return this._description;
    }
    /**
     * Sets the description of this task.
     */
    set description(desc) {
        this._description = desc;
    }
    /**
     * Reset the task so it no longer has any commands.
     * @param command the first command to add to the task after it was cleared.
     */
    reset(command, options = {}) {
        this.assertUnlocked();
        while (this._steps.length) {
            this._steps.shift();
        }
        if (command) {
            this.exec(command, options);
        }
    }
    /**
     * Executes a shell command
     * @param command Shell command
     * @param options Options
     */
    exec(command, options = {}) {
        this.assertUnlocked();
        this._steps.push({ exec: command, ...options });
    }
    /**
     * Execute a builtin task.
     *
     * Builtin tasks are programs bundled as part of projen itself and used as
     * helpers for various components.
     *
     * In the future we should support built-in tasks from external modules.
     *
     * @param name The name of the builtin task to execute (e.g.
     * `release/resolve-version`).
     */
    builtin(name) {
        this.assertUnlocked();
        this._steps.push({ builtin: name });
    }
    /**
     * Say something.
     * @param message Your message
     * @param options Options
     */
    say(message, options = {}) {
        this.assertUnlocked();
        this._steps.push({ say: message, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     *
     * @deprecated use `prependExec()`
     */
    prepend(shell, options = {}) {
        this.assertUnlocked();
        this.prependExec(shell, options);
    }
    /**
     * Spawns a sub-task.
     * @param subtask The subtask to execute.
     */
    spawn(subtask, options = {}) {
        this.assertUnlocked();
        this._steps.push({ spawn: subtask.name, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     */
    prependExec(shell, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            exec: shell,
            ...options,
        });
    }
    /**
     * Adds a spawn instruction at the beginning of the task.
     * @param subtask The subtask to execute.
     */
    prependSpawn(subtask, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            spawn: subtask.name,
            ...options,
        });
    }
    /**
     * Says something at the beginning of the task.
     * @param message Your message
     */
    prependSay(message, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            say: message,
            ...options,
        });
    }
    /**
     * Adds an environment variable to this task.
     * @param name The name of the variable
     * @param value The value. If the value is surrounded by `$()`, we will
     * evaluate it within a subshell and use the result as the value of the
     * environment variable.
     */
    env(name, value) {
        this.assertUnlocked();
        this._env[name] = value;
    }
    /**
     * Returns an immutable copy of all the step specifications of the task.
     */
    get steps() {
        return [...this._steps];
    }
    /**
     * Renders a task spec into the manifest.
     *
     * @internal
     */
    _renderSpec() {
        return {
            name: this.name,
            description: this.description,
            env: this._env,
            requiredEnv: this.requiredEnv,
            steps: this._steps,
            condition: this.condition,
            cwd: this.cwd,
        };
    }
    assertUnlocked() {
        if (this._locked) {
            throw new Error(`Task "${this.name}" is locked for changes`);
        }
    }
}
exports.Task = Task;
_a = JSII_RTTI_SYMBOL_1;
Task[_a] = { fqn: "projen.Task", version: "0.52.53" };
//# sourceMappingURL=data:application/json;base64,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