"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isExecutable = exports.isWritable = exports.tryReadFileSync = exports.tryReadFile = exports.kebabCaseKeys = exports.getGitVersion = exports.formatAsPythonModule = exports.sorted = exports.dedupArray = exports.deepMerge = exports.isObject = exports.isTruthy = exports.decamelizeKeysRecursively = exports.writeFile = exports.getFilePermissions = exports.execOrUndefined = exports.execCapture = exports.exec = void 0;
const child_process = require("child_process");
const path = require("path");
const Case = require("case");
const fs = require("fs-extra");
const logging = require("./logging");
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.debug(command);
    child_process.execSync(command, {
        stdio: ["inherit", process.stderr, "pipe"],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.exec = exec;
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.debug(command);
    return child_process.execSync(command, {
        stdio: ["inherit", "pipe", "pipe"],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.execCapture = execCapture;
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process
            .execSync(command, {
            stdio: ["inherit", "pipe", "pipe"],
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        })
            .toString("utf-8")
            .trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch {
        return undefined;
    }
}
exports.execOrUndefined = execOrUndefined;
function getFilePermissions(options) {
    var _a, _b;
    const readonly = (_a = options.readonly) !== null && _a !== void 0 ? _a : false;
    const executable = (_b = options.executable) !== null && _b !== void 0 ? _b : false;
    if (readonly && executable) {
        return "500";
    }
    else if (readonly) {
        return "400";
    }
    else if (executable) {
        return "755";
    }
    else {
        return "644";
    }
}
exports.getFilePermissions = getFilePermissions;
function writeFile(filePath, data, options = {}) {
    if (fs.existsSync(filePath)) {
        fs.chmodSync(filePath, "600");
    }
    fs.mkdirpSync(path.dirname(filePath));
    fs.writeFileSync(filePath, data);
    fs.chmodSync(filePath, getFilePermissions(options));
}
exports.writeFile = writeFile;
function decamelizeKeysRecursively(input, opt) {
    var _a, _b, _c, _d;
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = (_a = opt === null || opt === void 0 ? void 0 : opt.shouldDecamelize) !== null && _a !== void 0 ? _a : shouldAlwaysDecamelize;
    const separator = (_b = opt === null || opt === void 0 ? void 0 : opt.separator) !== null && _b !== void 0 ? _b : "_";
    const path_ = (_c = opt === null || opt === void 0 ? void 0 : opt.path) !== null && _c !== void 0 ? _c : [];
    const maxDepth = (_d = opt === null || opt === void 0 ? void 0 : opt.maxDepth) !== null && _d !== void 0 ? _d : 10;
    if (path_.length > maxDepth) {
        throw new Error("Decamelled too deeply - check that the input has no circular references");
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === "object" && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined ||
        ["null", "undefined", "0", "false", ""].includes(value.toLocaleLowerCase()));
}
exports.isTruthy = isTruthy;
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return (x !== null &&
        typeof x === "object" &&
        !Array.isArray(x) &&
        x.constructor.name === "Object");
}
exports.isObject = isObject;
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" &&
                    Object.keys(output).length === 0 &&
                    destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== "undefined") {
                target[key] = value;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
exports.dedupArray = dedupArray;
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === "object") {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
exports.sorted = sorted;
function formatAsPythonModule(name) {
    return name.replace(/-/g, "_").replace(/\./g, "_");
}
exports.formatAsPythonModule = formatAsPythonModule;
/**
 * Extract git version number from command line
 *
 * @param gitVersionOutput the output from `git version` CLI
 * @returns the version of git
 */
function getGitVersion(gitVersionOutput) {
    const match = gitVersionOutput.match(/\d+.\d+.\d+/);
    if (!match) {
        throw new Error("Unable to retrieve git version");
    }
    return match[0];
}
exports.getGitVersion = getGitVersion;
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/gm, "-")] = v;
    }
    return result;
}
exports.kebabCaseKeys = kebabCaseKeys;
async function tryReadFile(file) {
    if (!(await fs.pathExists(file))) {
        return "";
    }
    return fs.readFile(file, "utf8");
}
exports.tryReadFile = tryReadFile;
function tryReadFileSync(file) {
    if (!fs.pathExistsSync(file)) {
        return undefined;
    }
    return fs.readFileSync(file, "utf8");
}
exports.tryReadFileSync = tryReadFileSync;
function isWritable(file) {
    try {
        fs.accessSync(file, fs.constants.W_OK);
        return true;
    }
    catch {
        return false;
    }
}
exports.isWritable = isWritable;
function isExecutable(file) {
    try {
        fs.accessSync(file, fs.constants.X_OK);
        return true;
    }
    catch {
        return false;
    }
}
exports.isExecutable = isExecutable;
function decamelize(s, sep = "_") {
    if (Case.of(s) === "camel") {
        return Case.lower(s, sep);
    }
    else {
        return s;
    }
}
//# sourceMappingURL=data:application/json;base64,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