"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshot = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
/**
 * Creates a snapshot of the files generated by a project. Ignores any non-text
 * files so that the snapshots are human readable.
 */
function synthSnapshot(project) {
    // defensive: verify that "outdir" is actually in a temporary directory
    if (!path.resolve(project.outdir).startsWith(os.tmpdir()) &&
        !project.outdir.includes("project-temp-dir")) {
        throw new Error("Trying to capture a snapshot of a project outside of tmpdir, which implies this test might corrupt an existing project");
    }
    const synthed = Symbol.for("synthed");
    if (synthed in project) {
        throw new Error("duplicate synth()");
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = "true";
        project.synth();
        const ignoreExts = ["png", "ico"];
        return directorySnapshot(project.outdir, {
            excludeGlobs: ignoreExts.map((ext) => `**/*.${ext}`),
        });
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync("**", {
        ignore: [".git/**", ...((_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [])],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (!options.onlyFileNames) {
            if (path.extname(filePath) === ".json") {
                content = fs.readJsonSync(filePath);
            }
            else {
                content = fs.readFileSync(filePath, "utf-8");
            }
        }
        else {
            content = true;
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
//# sourceMappingURL=data:application/json;base64,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