#!/usr/bin/env python
# coding: utf-8

# Copyright (c) Saga Inc.
# Distributed under the terms of the GPL License.

"""
The main entry point for the mitosheet package, this command
line interface allows you to set some toggles in the user.json
"""
import sys
from mitosheet.hello import run_hello
from mitosheet.experiments.experiment_utils import set_experiment

from mitosheet.user.db import set_user_field
from mitosheet.user import initialize_user
from mitosheet.user.schemas import UJ_MITOSHEET_PRO, UJ_MITOSHEET_TELEMETRY, UJ_RECEIVED_CHECKLISTS, UJ_RECEIVED_TOURS, UJ_USER_EMAIL
from mitosheet.startup.startup_utils import create_startup_file, remove_startup_file
from mitosheet.user.utils import check_pro_acccess_code


def main() -> None:
    """
    Currently, the main usage of this function is:
    python -m mitosheet turnofflogging
    python -m mitosheet turnonlogging
    python -m mitosheet turnoffpro
    python -m mitosheet turnoffdataframebutton
    python -m mitosheet turnondataframebutton

    We also have 
    python -m mitosheet clearemail
    """
    # Make sure the user is initalized first, but do not identify
    # then, in case they are turning off logging
    initialize_user(call_identify=False)

    # Then, if we are being told to turn off logging, turn off logging
    if len(sys.argv) > 1:
        if sys.argv[-1] == 'hello':
            run_hello()
        if sys.argv[-1] == 'turnofflogging':
            print("Turning off all logging")
            set_user_field(UJ_MITOSHEET_TELEMETRY, False)
            print("Logging turned off!")
        if sys.argv[-1] == 'turnonlogging':
            print("Turning on all logging")
            set_user_field(UJ_MITOSHEET_TELEMETRY, True)
            print("Logging turned on!")
        if sys.argv[-1] == 'turnoffpro':
            print("Turning off Mitosheet pro")
            set_user_field(UJ_MITOSHEET_PRO, False)
            print("Mitosheet Pro is false!")
        if sys.argv[-2] == 'turnonpro':
            print("Turning on Mitosheet pro", sys.argv[-1])
            if check_pro_acccess_code(sys.argv[-1].strip()):
                set_user_field(UJ_MITOSHEET_PRO, True)
                set_user_field(UJ_MITOSHEET_TELEMETRY, False)
                print("Mitosheet Pro is now active!")
            else:
                print("Invalid access code...")
        if sys.argv[-1] == 'clearemail':
            print("Clearing email")
            set_user_field(UJ_USER_EMAIL, '')
            print("Email cleared")
        if sys.argv[-1] == 'cleartours':
            print("Clearing tours")
            set_user_field(UJ_RECEIVED_TOURS, [])
        if sys.argv[-1] == 'clearchecklists':
            print("Clearing checklists")
            set_user_field(UJ_RECEIVED_CHECKLISTS, {})
            print("Checklists cleared")
        if sys.argv[-1] == 'turnoffdataframebutton':
            print("Turning off the 'View in Mito' dataframe button")
            remove_startup_file()
            print("Turned off the 'View in Mito' dataframe button\nThe next time you launch Jupyter Lab, the button will no longer be visible")
        if sys.argv[-1] == 'turnondataframebutton':
            print("Turning on the 'View in Mito' dataframe button")
            create_startup_file()
            print("Turned on the 'View in Mito' dataframe button\nThe next time you launch Jupyter Lab, the button will be visible")
        if len(sys.argv) > 2:
            if sys.argv[-3] == 'setexperiment':
                print("Setting experiment", sys.argv[-2], "to variant", sys.argv[-1])
                set_experiment(sys.argv[-2], sys.argv[-1])
                print("Experiment set")
        
if __name__ == '__main__':
    main()