# Quick Start Guide

Get started with the PredictionData Python client in 5 minutes.

## Installation

```bash
pip install predictiondata
```

## Your First Query

```python
import asyncio
from predictiondata import PredictionDataClient, Channel


async def main():
    # Initialize client with your API key
    client = PredictionDataClient(api_key="<YOUR_API_KEY>")
    
    # Stream historical trades
    messages = client.replay(
        exchange="polymarket",
        from_date="2024-11-01",
        to_date="2024-11-02",
        filters=[Channel(name="trades", symbols=["will-trump-win-2024/YES"])]
    )
    
    # Process messages
    async for local_timestamp, message in messages:
        print(f"Trade at {local_timestamp}ms: {message['side']} {message['size']} @ ${message['price']}")
    
    await client.close()


if __name__ == "__main__":
    asyncio.run(main())
```

## Data Types

### 1. Trades

Real-time executed trades from the order book:

```python
Channel(name="trades", symbols=["market-slug/YES"])
```

Each message contains:
- `exchange_timestamp`: When the trade occurred
- `local_timestamp`: When it was captured
- `side`: "BUY" or "SELL"
- `size`: Number of contracts
- `price`: Trade price

### 2. Order Books

Incremental order book snapshots:

```python
Channel(name="books", symbols=["market-slug/YES"])
```

Each message contains:
- `exchange_timestamp`: Book update time
- `local_timestamp`: Capture time
- `ask_prices`: Comma-separated ask prices
- `ask_sizes`: Comma-separated ask sizes
- `bid_prices`: Comma-separated bid prices
- `bid_sizes`: Comma-separated bid sizes

### 3. On-chain Fills

Settlement data from Polygon blockchain:

```python
Channel(name="onchain_fills", symbols=["market-slug/YES"])
```

Each message contains:
- `block_number`: Blockchain block number
- `block_timestamp`: Block time
- `side`: "BUY" or "SELL"
- `size`: Contracts filled
- `price`: Fill price
- `maker`: Maker address
- `taker`: Taker address

## Common Patterns

### Multiple Markets

```python
filters=[
    Channel(name="trades", symbols=[
        "will-trump-win-2024/YES",
        "will-trump-win-2024/NO",
        "btc-above-100k/YES"
    ])
]
```

### Using Token IDs

```python
Channel(name="onchain_fills", token_ids=["0x1234..."])
```

### Fetch Single Day (Non-streaming)

```python
data = await client.fetch_day(
    exchange="polymarket",
    data_type="trades",
    identifier="market-slug/YES",
    date="2024-11-15"
)
```

### Context Manager

```python
async with PredictionDataClient(api_key="key") as client:
    # Client automatically closes when done
    messages = client.replay(...)
    async for ts, msg in messages:
        process(msg)
```

## Market Identifiers

Markets use format: `{event-slug}/{outcome}`

Examples:
- `will-trump-win-2024/YES`
- `will-trump-win-2024/NO`
- `btc-above-100k-by-2025/YES`

You can also use token IDs (contract addresses) directly.

## Date Formats

All dates use `YYYY-MM-DD` format:
- `from_date="2024-11-01"`
- `to_date="2024-11-30"`

The date range is inclusive on both ends.

## Timestamps

All timestamps are in **milliseconds** since Unix epoch:
- `local_timestamp`: When data was captured by PredictionData servers
- `exchange_timestamp`: When event occurred on exchange (for books/trades)
- `block_timestamp`: Blockchain timestamp (for on-chain fills)

## Processing Messages

Each message is a dictionary with:
- Original CSV fields (varies by data type)
- Metadata fields (prefixed with `_`):
  - `_symbol`: Market identifier
  - `_data_type`: "trades", "books", or "onchain/fills"
  - `_date`: Date string

Example:
```python
async for local_timestamp, message in messages:
    print(f"Market: {message['_symbol']}")
    print(f"Type: {message['_data_type']}")
    print(f"Date: {message['_date']}")
    
    if message['_data_type'] == 'trades':
        print(f"Trade: {message['side']} {message['size']} @ ${message['price']}")
```

## Error Handling

The client handles common errors gracefully:
- Missing data (404) is skipped automatically
- Network errors are logged but don't stop iteration
- Invalid API keys raise authentication errors

```python
try:
    messages = client.replay(...)
    async for ts, msg in messages:
        process(msg)
except Exception as e:
    print(f"Error: {e}")
finally:
    await client.close()
```

## Next Steps

- Check out the [examples/](examples/) directory for more use cases
- Read the full [README.md](README.md) for API reference
- See [HOW_TO_PUBLISH.md](HOW_TO_PUBLISH.md) if you want to contribute

## Getting Help

- Documentation: https://predictiondata.dev/docs
- Issues: https://github.com/predictiondata/predictiondata_client/issues
- Email: support@predictiondata.dev

Happy coding! 🚀

